"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findDependencyDirectory = findDependencyDirectory;
exports.findPackageJsonUp = findPackageJsonUp;
exports.findUp = findUp;
exports.isBuiltinModule = isBuiltinModule;
const node_fs_1 = require("node:fs");
const path = require("node:path");
const util_1 = require("./util");
/**
 * Find the directory that contains a given dependency, identified by its 'package.json', from a starting search directory
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
async function findDependencyDirectory(dependencyName, searchStart) {
    // Explicitly do not use 'require("dep/package.json")' because that will fail if the
    // package does not export that particular file.
    const entryPoint = require.resolve(dependencyName, {
        paths: [searchStart],
    });
    // Search up from the given directory, looking for a package.json that matches
    // the dependency name (so we don't accidentally find stray 'package.jsons').
    const depPkgJsonPath = await findPackageJsonUp(dependencyName, path.dirname(entryPoint));
    if (!depPkgJsonPath) {
        throw new Error(`Could not find dependency '${dependencyName}' from '${searchStart}'`);
    }
    return depPkgJsonPath;
}
/**
 * Find the package.json for a given package upwards from the given directory
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
async function findPackageJsonUp(packageName, directory) {
    return findUp(directory, async (dir) => {
        const pjFile = path.join(dir, 'package.json');
        return (await (0, util_1.pathExists)(pjFile)) && JSON.parse(await node_fs_1.promises.readFile(pjFile, 'utf-8')).name === packageName;
    });
}
// eslint-disable-next-line @typescript-eslint/promise-function-async
function findUp(directory, pred) {
    const result = pred(directory);
    if (result instanceof Promise) {
        return result.then((thisDirectory) => (thisDirectory ? directory : recurse()));
    }
    return result ? directory : recurse();
    function recurse() {
        const parent = path.dirname(directory);
        if (parent === directory) {
            return undefined;
        }
        return findUp(parent, pred);
    }
}
/**
 * Whether the given dependency is a built-in
 *
 * Some dependencies that occur in `package.json` are also built-ins in modern Node
 * versions (most egregious example: 'punycode'). Detect those and filter them out.
 */
function isBuiltinModule(depName) {
    // eslint-disable-next-line @typescript-eslint/no-require-imports,@typescript-eslint/no-var-requires
    const { builtinModules } = require('node:module');
    return (builtinModules ?? []).includes(depName);
}
//# sourceMappingURL=find-utils.js.map