"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findUp = exports.findPackageJsonUp = exports.isBuiltinModule = exports.findDependencyDirectory = void 0;
const fs = require("fs-extra");
const path = require("path");
/**
 * Find the directory that contains a given dependency, identified by its 'package.json', from a starting search directory
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
async function findDependencyDirectory(dependencyName, searchStart) {
    // Explicitly do not use 'require("dep/package.json")' because that will fail if the
    // package does not export that particular file.
    const entryPoint = require.resolve(dependencyName, {
        paths: [searchStart],
    });
    // Search up from the given directory, looking for a package.json that matches
    // the dependency name (so we don't accidentally find stray 'package.jsons').
    const depPkgJsonPath = await findPackageJsonUp(dependencyName, path.dirname(entryPoint));
    if (!depPkgJsonPath) {
        throw new Error(`Could not find dependency '${dependencyName}' from '${searchStart}'`);
    }
    return depPkgJsonPath;
}
exports.findDependencyDirectory = findDependencyDirectory;
/**
 * Whether the given dependency is a built-in
 *
 * Some dependencies that occur in `package.json` are also built-ins in modern Node
 * versions (most egregious example: 'punycode'). Detect those and filter them out.
 */
function isBuiltinModule(depName) {
    // eslint-disable-next-line @typescript-eslint/no-require-imports,@typescript-eslint/no-var-requires
    const { builtinModules } = require('module');
    return (builtinModules ?? []).includes(depName);
}
exports.isBuiltinModule = isBuiltinModule;
/**
 * Find the package.json for a given package upwards from the given directory
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
async function findPackageJsonUp(packageName, directory) {
    return findUp(directory, async (dir) => {
        const pjFile = path.join(dir, 'package.json');
        return ((await fs.pathExists(pjFile)) &&
            (await fs.readJson(pjFile)).name === packageName);
    });
}
exports.findPackageJsonUp = findPackageJsonUp;
/**
 * Find a directory up the tree from a starting directory matching a condition
 *
 * Will return `undefined` if no directory matches
 *
 * (This code is duplicated among jsii/jsii-pacmak/jsii-reflect. Changes should be done in all
 * 3 locations, and we should unify these at some point: https://github.com/aws/jsii/issues/3236)
 */
async function findUp(directory, pred) {
    // eslint-disable-next-line no-constant-condition
    while (true) {
        // eslint-disable-next-line no-await-in-loop
        if (await pred(directory)) {
            return directory;
        }
        const parent = path.dirname(directory);
        if (parent === directory) {
            return undefined;
        }
        directory = parent;
    }
}
exports.findUp = findUp;
//# sourceMappingURL=util.js.map