"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tarballName = exports.substituteReservedWords = exports.getParamDependencies = exports.getMemberDependencies = exports.flatMap = exports.goPackageNameForAssembly = exports.findTypeInTree = void 0;
/*
 * Recursively search module for type with fqn
 */
function findTypeInTree(module, fqn) {
    const result = module.types.find((t) => t.type.fqn === fqn);
    if (result) {
        return result;
    }
    return module.submodules.reduce((accum, sm) => {
        return accum ?? findTypeInTree(sm, fqn);
    }, undefined);
}
exports.findTypeInTree = findTypeInTree;
/*
 * Format NPM package names as idiomatic Go module name
 */
function goPackageNameForAssembly(assembly) {
    const config = assembly.targets?.go ?? {};
    if (config.packageName) {
        return config.packageName;
    }
    return assembly.name.replace(/[^a-z0-9.]/gi, '').toLowerCase();
}
exports.goPackageNameForAssembly = goPackageNameForAssembly;
function flatMap(collection, mapper) {
    return collection
        .map(mapper)
        .reduce((acc, elt) => acc.concat(elt), new Array());
}
exports.flatMap = flatMap;
/*
 * Return module dependencies of a class or interface members
 */
function getMemberDependencies(members) {
    return members.flatMap((member) => member.reference?.dependencies ?? []);
}
exports.getMemberDependencies = getMemberDependencies;
function getParamDependencies(methods) {
    return methods.flatMap(({ parameters }) => parameters.flatMap((param) => param.reference?.dependencies ?? []));
}
exports.getParamDependencies = getParamDependencies;
const RESERVED_WORDS = {
    break: 'break_',
    default: 'default_',
    func: 'func_',
    interface: 'interface_',
    select: 'select_',
    case: 'case_',
    defer: 'defer_',
    go: 'go_',
    map: 'map_',
    struct: 'struct_',
    chan: 'chan_',
    else: 'else_',
    goto: 'goto_',
    package: 'package_',
    switch: 'switch_',
    const: 'const_',
    fallthrough: 'fallthrough_',
    if: 'if_',
    range: 'range_',
    type: 'type_',
    continue: 'continue_',
    for: 'for_',
    import: 'import_',
    return: 'return_',
    var: 'var_',
    _: '_arg',
};
/*
 * Sanitize reserved words
 */
function substituteReservedWords(name) {
    return RESERVED_WORDS[name] || name;
}
exports.substituteReservedWords = substituteReservedWords;
/**
 * Computes a safe tarball name for the provided assembly.
 *
 * @param assm the assembly.
 *
 * @returns a tarball name.
 */
function tarballName(assm) {
    const name = assm.name.replace(/^@/, '').replace(/\//g, '-');
    return `${name}-${assm.version}.tgz`;
}
exports.tarballName = tarballName;
//# sourceMappingURL=util.js.map