"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidateProviderPresence = void 0;
const terraform_provider_1 = require("../terraform-provider");
const terraform_resource_1 = require("../terraform-resource");
const terraform_data_source_1 = require("../terraform-data-source");
const terraform_stack_1 = require("../terraform-stack");
/**
 * A validation that is added by default, ensuring that all providers
 * used are defined via Constructs.
 */
class ValidateProviderPresence {
    /**
     * Creates a ValidateProviderPresence
     */
    constructor(host) {
        this.host = host;
        /**
         * All provider names found in the construct tree.
         */
        this.providerNames = new Set();
        /**
         * All TerraformProviders found in the construct tree.
         */
        this.foundProviders = [];
    }
    /**
     * Checks the construct tree recursively
     */
    check(node) {
        if (terraform_resource_1.TerraformResource.isTerraformResource(node) ||
            terraform_data_source_1.TerraformDataSource.isTerraformDataSource(node)) {
            if (node.terraformGeneratorMetadata &&
                node.terraformGeneratorMetadata.providerName !== "terraform") {
                this.providerNames.add(node.terraformGeneratorMetadata.providerName);
            }
        }
        if (terraform_provider_1.TerraformProvider.isTerraformProvider(node)) {
            this.foundProviders.push(node);
        }
        for (const child of node.node.children) {
            this.check(child);
        }
    }
    /**
     * Run the validation
     */
    validate() {
        this.check(this.host);
        const missingProviders = Array.from(this.providerNames).filter((name) => {
            return !this.foundProviders.some((p) => p.terraformGeneratorMetadata?.providerName === name);
        });
        if (missingProviders.length === 0) {
            return [];
        }
        else {
            const stack = terraform_stack_1.TerraformStack.of(this.host);
            return [
                `Found resources without a matching provider construct. Please make sure to add provider constructs [e.g. new RandomProvider(...)] to your stack '${stack.node.id}' for the following providers: ${missingProviders.join(", ")}`,
            ];
        }
    }
}
exports.ValidateProviderPresence = ValidateProviderPresence;
//# sourceMappingURL=data:application/json;base64,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