"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidateBinaryVersion = void 0;
const child_process_1 = require("child_process");
const semver = require("semver");
/**
 * A validation that can be applied to a construct that will error if the
 * construct is being used in an environment with a version of a binary lower than the specified version.
 */
class ValidateBinaryVersion {
    constructor(binary, versionConstraint, versionCommand, hint) {
        this.binary = binary;
        this.versionConstraint = versionConstraint;
        this.versionCommand = versionCommand;
        this.hint = hint;
    }
    validate() {
        try {
            const versionOfCommand = (0, child_process_1.execSync)(this.versionCommand, {
                stdio: "pipe",
            }).toString();
            const semverRegex = /\d+\.\d+\.\d+/;
            const version = versionOfCommand.match(semverRegex);
            if (!version) {
                return [
                    `Could not determine version of ${this.binary} (running ${this.versionCommand})`,
                ];
            }
            if (!semver.satisfies(version[0], this.versionConstraint)) {
                return [
                    `${this.binary} version ${version[0]} is lower than the required version ${this.versionConstraint} for this construct. ${this.hint || ""}`,
                ];
            }
            return [];
        }
        catch (e) {
            return [
                `Could not determine version of ${this.binary}, ${this.versionCommand} failed: ${e}`,
            ];
        }
    }
}
exports.ValidateBinaryVersion = ValidateBinaryVersion;
//# sourceMappingURL=data:application/json;base64,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