"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.terraformBinaryName = void 0;
exports.deepMerge = deepMerge;
exports.snakeCase = snakeCase;
exports.keysToSnakeCase = keysToSnakeCase;
exports.processDynamicAttributesForHcl = processDynamicAttributesForHcl;
exports.processDynamicAttributes = processDynamicAttributes;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const errors_1 = require("./errors");
const terraform_dynamic_block_1 = require("./terraform-dynamic-block");
const token_1 = require("./tokens/token");
exports.terraformBinaryName = process.env.TERRAFORM_BINARY_NAME || "terraform";
/**
 * Merges `source` into `target`, overriding any existing values.
 * `undefined` will cause a value to be deleted.
 */
function deepMerge(target, ...sources) {
    if (token_1.Tokenization.isResolvable(target) && sources.length > 0) {
        throw (0, errors_1.targetNotResolvableWithOverrides)(target.toString());
    }
    for (const source of sources) {
        if (typeof source !== "object" || typeof target !== "object") {
            throw (0, errors_1.sourceOrTargetNotAnObject)(JSON.stringify(source), typeof source, JSON.stringify(target), typeof target);
        }
        for (const key of Object.keys(source)) {
            const value = source[key];
            if (typeof value === "object" && value != null && !Array.isArray(value)) {
                // if the value at the target is not an object, override it with an
                // object so we can continue the recursion
                if (typeof target[key] !== "object") {
                    target[key] = {};
                }
                // if the value is a resolvable we don't want to recurse into it
                if (token_1.Tokenization.isResolvable(value)) {
                    target[key] = value;
                    continue;
                }
                deepMerge(target[key], value);
            }
            else if (typeof value === "object" &&
                value != null &&
                Array.isArray(value)) {
                if (Array.isArray(target[key])) {
                    target[key] = [...target[key], ...value];
                }
                else {
                    target[key] = value;
                }
            }
            else if (value === undefined) {
                delete target[key];
            }
            else {
                target[key] = value;
            }
        }
    }
    return target;
}
/**
 * Transforms a string to snake case
 */
function snakeCase(str) {
    if (!str)
        return "";
    return String(str)
        .replace(/^[^A-Za-z0-9]*|[^A-Za-z0-9]*$/g, "")
        .replace(/([a-z0-9])([A-Z])/g, (_m, a, b) => a + "_" + b.toLowerCase())
        .replace(/[^A-Za-z0-9]+|_+/g, "_")
        .toLowerCase();
}
/**
 * Transforms all keys in a object to snake case
 */
function keysToSnakeCase(object) {
    if (token_1.Tokenization.isResolvable(object)) {
        return object;
    }
    if (Array.isArray(object)) {
        return object.map((e) => {
            return typeof e === "object" ? keysToSnakeCase(e) : e;
        });
    }
    const keys = Object.keys(object);
    return keys.reduce((newObject, key) => {
        if (key === "tags" || key === "environment") {
            newObject[key] = object[key];
            return newObject;
        }
        let value = object[key];
        if (typeof value === "object") {
            value = keysToSnakeCase(value);
        }
        if (Array.isArray(value)) {
            value = value.map((e) => {
                return typeof e === "object" ? keysToSnakeCase(e) : e;
            });
        }
        newObject[snakeCase(key)] = value;
        return newObject;
    }, {});
}
/**
 * dynamic attributes are located at a different position than normal block attributes
 * This method detects them and moves them from .attributeName to .dynamic.attributeName
 * It also invokes the .toTerraform() method on the dynamic attribute to get the correct
 * Terraform representation
 */
function processDynamicAttributesForHcl(attributes) {
    const result = {};
    Object.entries(attributes).forEach(([attributeName, value]) => {
        if (terraform_dynamic_block_1.TerraformDynamicBlock.isTerraformDynamicBlock(value)) {
            if (!result.dynamic) {
                result.dynamic = {};
            }
            result.dynamic[attributeName] = value.toTerraformDynamicBlockJson();
        }
        else {
            const recurse = value !== null &&
                typeof value === "object" &&
                value.constructor === Object; // only descend into plain objects
            result[attributeName] = recurse
                ? processDynamicAttributesForHcl(value)
                : value;
        }
    });
    return result;
}
/**
 * dynamic attributes are located at a different position than normal block attributes
 * This method detects them and moves them from .attributeName to .dynamic.attributeName
 * It also invokes the .toTerraform() method on the dynamic attribute to get the correct
 * Terraform representation
 */
function processDynamicAttributes(attributes) {
    const result = {};
    Object.entries(attributes).forEach(([attributeName, value]) => {
        if (terraform_dynamic_block_1.TerraformDynamicBlock.isTerraformDynamicBlock(value)) {
            if (!result.dynamic) {
                result.dynamic = {};
            }
            result.dynamic[attributeName] = value.toTerraformDynamicBlockJson();
        }
        else {
            const recurse = value !== null &&
                typeof value === "object" &&
                value.constructor === Object; // only descend into plain objects
            result[attributeName] = recurse ? processDynamicAttributes(value) : value;
        }
    });
    return result;
}
//# sourceMappingURL=data:application/json;base64,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