"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MigrateIds = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const _1 = require(".");
const crypto = require("crypto");
const errors_1 = require("./errors");
const path = require("path");
/**
 * We have to copy this from the old CDKTF version for now, so that we can
 * calculate the old logical IDs for now and for later for the migration.
 */
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const UNIQUE_SEP = "_";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueIdOldVersion(components, allowSepChars) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw (0, errors_1.cannotCalcIdForEmptySetOfComponents)();
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeDisallowedCharacters(components[0], allowSepChars);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map((s) => removeDisallowedCharacters(s, allowSepChars))
        .join(UNIQUE_SEP)
        .slice(0, MAX_HUMAN_LEN);
    return human + UNIQUE_SEP + hash;
}
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 *
 * @param s
 * @param allowSepChars
 * @returns
 */
function removeDisallowedCharacters(s, allowSepChars) {
    if (allowSepChars) {
        return removeNonAlphanumericSep(s);
    }
    else {
        return removeNonAlphanumeric(s);
    }
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, "");
}
/**
 *
 * @param s
 * @returns
 */
function removeNonAlphanumericSep(s) {
    return s.replace(/[^A-Za-z0-9_-]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
/**
 *
 * @param tfElement
 * @returns
 */
function allocateLogicalIdOldVersion(tfElement) {
    const node = _1.TerraformElement.isTerraformElement(tfElement)
        ? tfElement.node
        : tfElement;
    // This is the previous behavior, which we want for now.
    const stackIndex = 0;
    const components = node.scopes.slice(stackIndex + 1).map((c) => c.node.id);
    return components.length > 0 ? makeUniqueIdOldVersion(components, false) : "";
}
/**
 * For migrating past 0.17 where the feature flag for the old id generation logic was removed after being deprecated since 0.15
 */
class MigrateIds {
    visit(node) {
        // eslint-disable-next-line no-instanceof/no-instanceof
        if (node instanceof _1.TerraformResource) {
            const oldId = allocateLogicalIdOldVersion(node);
            node.moveFromId(`${node.terraformResourceType}.${oldId}`);
        }
        // eslint-disable-next-line no-instanceof/no-instanceof
        if (node instanceof _1.TerraformModule) {
            const oldId = allocateLogicalIdOldVersion(node);
            const appManifest = _1.App.of(node).manifest;
            const stackManifest = appManifest.forStack(_1.TerraformStack.of(node));
            const stackOutdir = path.join(appManifest.outdir, stackManifest.workingDirectory);
            _1.Annotations.of(node)
                .addWarning(`Found module with new id ${node.friendlyUniqueId}. Moving this module requires a manual state migration.
If this module has not been moved yet, run "terraform state mv module.${oldId} module.${node.friendlyUniqueId}" in the output directory "${stackOutdir}" to migrate the existing state to its new id.
Refer to the following page for more information: https://developer.hashicorp.com/terraform/cdktf/examples-and-guides/refactoring#moving-or-renaming-modules`);
        }
    }
}
exports.MigrateIds = MigrateIds;
_a = JSII_RTTI_SYMBOL_1;
MigrateIds[_a] = { fqn: "cdktf.MigrateIds", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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