"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Tokenization = exports.Token = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const lazy_1 = require("./lazy");
const encoding_1 = require("./private/encoding");
const intrinsic_1 = require("./private/intrinsic");
const resolve_1 = require("./private/resolve");
const token_map_1 = require("./private/token-map");
/**
 * Represents a special or lazily-evaluated value.
 *
 * Can be used to delay evaluation of a certain value in case, for example,
 * that it requires some context or late-bound data. Can also be used to
 * mark values that need special processing at document rendering time.
 *
 * Tokens can be embedded into strings while retaining their original
 * semantics.
 */
class Token {
    /**
     * Returns true if obj represents an unresolved value
     *
     * One of these must be true:
     *
     * - `obj` is an IResolvable
     * - `obj` is a string containing at least one encoded `IResolvable`
     * - `obj` is either an encoded number or list
     *
     * This does NOT recurse into lists or objects to see if they
     * containing resolvables.
     *
     * @param obj The object to test.
     */
    static isUnresolved(obj) {
        return (0, encoding_1.unresolved)(obj);
    }
    /**
     * Return a reversible string representation of this token
     *
     * If the Token is initialized with a literal, the stringified value of the
     * literal is returned. Otherwise, a special quoted string representation
     * of the Token is returned that can be embedded into other strings.
     *
     * Strings with quoted Tokens in them can be restored back into
     * complex values with the Tokens restored by calling `resolve()`
     * on the string.
     */
    static asString(value, options = {}) {
        if (typeof value === "string") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerString(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible number representation of this token
     */
    static asNumber(value) {
        if (typeof value === "number") {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumber(Token.asAny(value));
    }
    /**
     * Return a reversible list representation of this token
     */
    static asList(value, options = {}) {
        if (Array.isArray(value) && value.every((x) => typeof x === "string")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerList(Token.asAny(value), options.displayHint);
    }
    /**
     * Return a reversible list representation of this token
     */
    static asNumberList(value) {
        if (Array.isArray(value) && value.every((x) => typeof x === "number")) {
            return value;
        }
        return token_map_1.TokenMap.instance().registerNumberList(Token.asAny(value));
    }
    /**
     * Return a reversible map representation of this token
     */
    static asMap(value, mapValue, options = {}) {
        // since the return value is basically an object, just encode always
        return token_map_1.TokenMap.instance().registerMap(Token.asAny(value), mapValue, options.displayHint);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asStringMap(value, options = {}) {
        return this.asMap(value, Token.STRING_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asNumberMap(value, options = {}) {
        return this.asMap(value, Token.NUMBER_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asBooleanMap(value, options = {}) {
        return this.asMap(value, true, options);
    }
    /**
     * Return a reversible map representation of this token
     */
    static asAnyMap(value, options = {}) {
        return this.asMap(value, Token.ANY_MAP_TOKEN_VALUE, options);
    }
    /**
     * Return a resolvable representation of the given value
     */
    static asAny(value) {
        return Tokenization.isResolvable(value) ? value : new intrinsic_1.Intrinsic(value);
    }
    /**
     * Return a Token containing a `null` value
     *
     * Note: This is different than `undefined`, `nil`, `None` or similar
     * as it will end up in the Terraform config and can be used to explicitly
     * not set an attribute (which is sometimes required by Terraform providers)
     *
     * @returns a Token resolving to `null` as understood by Terraform
     */
    static nullValue() {
        // passing null works in Typescript, this is why we can do this here
        // but users wouldn't be able to call e.g. Token.as_any(None) in Python
        return Token.asAny(null);
    }
}
exports.Token = Token;
_a = JSII_RTTI_SYMBOL_1;
Token[_a] = { fqn: "cdktf.Token", version: "0.21.0" };
/**
 * String Map token value representation
 */
Token.STRING_MAP_TOKEN_VALUE = "String Map Token Value";
/**
 * Number Map token value representation
 */
Token.NUMBER_MAP_TOKEN_VALUE = -123456789;
/**
 * Any map token representation
 */
Token.ANY_MAP_TOKEN_VALUE = "Any Map Token Value";
/**
 * Less oft-needed functions to manipulate Tokens
 */
class Tokenization {
    /**
     * Reverse any value into Resolvables, if possible
     */
    static reverse(x) {
        if (Tokenization.isResolvable(x)) {
            return [x];
        }
        if (typeof x === "string") {
            const reversedString = Tokenization.reverseString(x);
            return [...reversedString.tokens, ...reversedString.intrinsic];
        }
        if (Array.isArray(x)) {
            const reversedList = Tokenization.reverseList(x);
            if (reversedList) {
                return [reversedList];
            }
            const reversedNumberList = Tokenization.reverseNumberList(x);
            return reversedNumberList ? [reversedNumberList] : [];
        }
        if (typeof x === "number") {
            const reversedNumber = Tokenization.reverseNumber(x);
            return reversedNumber ? [reversedNumber] : [];
        }
        if (typeof x === "object" && x !== null) {
            const reversedMap = Tokenization.reverseMap(x);
            return reversedMap ? [reversedMap] : [];
        }
        return []; // null or undefined cannot be reversed
    }
    /**
     * Un-encode a string potentially containing encoded tokens
     */
    static reverseString(s) {
        return token_map_1.TokenMap.instance().splitString(s);
    }
    /**
     * Un-encode a Tokenized value from a number
     */
    static reverseNumber(n) {
        return token_map_1.TokenMap.instance().lookupNumberToken(n);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseList(l) {
        return token_map_1.TokenMap.instance().lookupList(l);
    }
    /**
     * Un-encode a Tokenized value from a list
     */
    static reverseNumberList(l) {
        return token_map_1.TokenMap.instance().lookupNumberList(l);
    }
    /**
     * Un-encode a Tokenized value from a map
     */
    static reverseMap(m) {
        return token_map_1.TokenMap.instance().lookupMap(m);
    }
    /**
     * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
     * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
     *
     * @param obj The object to resolve.
     * @param options Prefix key path components for diagnostics.
     */
    static resolve(obj, options) {
        return (0, resolve_1.resolve)(obj, {
            scope: options.scope,
            resolver: options.resolver,
            preparing: options.preparing !== undefined ? options.preparing : false,
        });
    }
    /**
     * Return whether the given object is an IResolvable object
     *
     * This is different from Token.isUnresolved() which will also check for
     * encoded Tokens, whereas this method will only do a type check on the given
     * object.
     */
    static isResolvable(obj) {
        return (typeof obj === "object" &&
            obj !== null &&
            typeof obj.resolve === "function");
    }
    /**
     * Stringify a number directly or lazily if it's a Token. If it is an object (i.e., { Ref: 'SomeLogicalId' }), return it as-is.
     */
    static stringifyNumber(x) {
        // only convert numbers to strings so that Refs, conditions, and other things don't end up synthesizing as [object object]
        if (Token.isUnresolved(x)) {
            return lazy_1.Lazy.stringValue({
                produce: (context) => {
                    const resolved = context.resolve(x);
                    return typeof resolved !== "number" ? resolved : `${resolved}`;
                },
            });
        }
        else {
            return typeof x !== "number" ? x : `${x}`;
        }
    }
}
exports.Tokenization = Tokenization;
_b = JSII_RTTI_SYMBOL_1;
Tokenization[_b] = { fqn: "cdktf.Tokenization", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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