"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenizedStringFragments = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const token_1 = require("./token");
/**
 * Fragments of a concatenated string containing stringified Tokens
 * @experimental
 */
class TokenizedStringFragments {
    constructor() {
        /**
         * Fragments in the tokenized string
         */
        this.fragments = new Array();
    }
    /**
     * Returns the first token.
     */
    get firstToken() {
        const first = this.fragments[0];
        if (first.type === "token") {
            return first.token;
        }
        return undefined;
    }
    /**
     * Returns the first value.
     */
    get firstValue() {
        return fragmentValue(this.fragments[0]);
    }
    /**
     * Returns the number of fragments.
     */
    get length() {
        return this.fragments.length;
    }
    /**
     * Adds a literal fragment
     * @param lit the literal to add
     */
    addLiteral(lit) {
        this.fragments.push({ type: "literal", lit });
    }
    /**
     * Adds a token fragment
     * @param token the token to add
     */
    addToken(token) {
        this.fragments.push({ type: "token", token });
    }
    /**
     * Adds an intrinsic fragment
     * @param value the intrinsic value to add
     */
    addIntrinsic(value) {
        this.fragments.push({ type: "intrinsic", value });
    }
    addEscape(kind) {
        this.fragments.push({ type: "escape", kind });
    }
    concat(other) {
        this.fragments.push(...other.fragments);
    }
    /**
     * Return all Tokens from this string
     */
    get tokens() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "token") {
                ret.push(f.token);
            }
        }
        return ret;
    }
    /**
     * Return all literals from this string
     */
    get literals() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "literal") {
                ret.push(f.lit);
            }
        }
        return ret;
    }
    /**
     * Return all intrinsic fragments from this string
     */
    get intrinsic() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "intrinsic") {
                ret.push(f.value);
            }
        }
        return ret;
    }
    /**
     * Return all escape fragments from this string
     */
    get escapes() {
        const ret = new Array();
        for (const f of this.fragments) {
            if (f.type === "escape") {
                if (f.kind === "open")
                    ret.push("${");
                else
                    ret.push("}");
            }
        }
        return ret;
    }
    /**
     * Apply a transformation function to all tokens in the string
     */
    mapTokens(context) {
        const ret = new TokenizedStringFragments();
        const originalSupressBraces = context.suppressBraces;
        for (const f of this.fragments) {
            switch (f.type) {
                case "literal":
                    ret.addLiteral(f.lit);
                    break;
                case "escape":
                    if (context.ignoreEscapes) {
                        ret.addLiteral(f.kind === "open" ? "${" : "}");
                        break;
                    }
                    ret.addEscape(f.kind);
                    if (f.kind === "open") {
                        context.suppressBraces = true;
                    }
                    else {
                        context.suppressBraces = originalSupressBraces;
                    }
                    break;
                case "token":
                    // eslint-disable-next-line no-case-declarations
                    const mapped = context.resolve(f.token);
                    if (token_1.Tokenization.isResolvable(mapped)) {
                        ret.addToken(mapped);
                    }
                    else {
                        ret.addIntrinsic(mapped);
                    }
                    break;
                case "intrinsic":
                    ret.addIntrinsic(f.value);
                    break;
            }
        }
        return ret;
    }
    /**
     * Combine the string fragments using the given joiner.
     *
     * If there are any
     */
    join(concat) {
        if (this.fragments.length === 0) {
            return concat.join(undefined, undefined);
        }
        if (this.fragments.length === 1) {
            return this.firstValue;
        }
        const values = this.fragments.map(fragmentValue);
        while (values.length > 1) {
            const prefix = values.splice(0, 2);
            values.splice(0, 0, concat.join(prefix[0], prefix[1]));
        }
        return values[0];
    }
}
exports.TokenizedStringFragments = TokenizedStringFragments;
_a = JSII_RTTI_SYMBOL_1;
TokenizedStringFragments[_a] = { fqn: "cdktf.TokenizedStringFragments", version: "0.21.0" };
/**
 * Resolve the value from a single fragment
 *
 * If the fragment is a Token, return the string encoding of the Token.
 */
function fragmentValue(fragment) {
    switch (fragment.type) {
        case "literal":
            return fragment.lit;
        case "escape":
            return fragment.kind === "open" ? "${" : "}";
        case "token":
            return fragment.token.toString();
        case "intrinsic":
            return fragment.value;
    }
}
//# sourceMappingURL=data:application/json;base64,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