"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TokenMap = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// copied from https://github.com/aws/constructs/blob/e01e47f78ef1e9b600efcd23ff7705aa8d384017/lib/private/token-map.ts
const errors_1 = require("../../errors");
const encoding_1 = require("./encoding");
const glob = global;
const STRING_SYMBOL = Symbol.for("@cdktf/core.TokenMap.STRING");
const LIST_SYMBOL = Symbol.for("@cdktf/core.TokenMap.LIST");
const NUMBER_SYMBOL = Symbol.for("@cdktf/core.TokenMap.NUMBER");
const NUMBER_LIST_SYMBOL = Symbol.for("@cdktf/core.TokenMap.NUMBER_LIST");
const MAP_SYMBOL = Symbol.for("@cdktf/core.TokenMap.MAP");
/**
 * Central place where we keep a mapping from Tokens to their String representation
 *
 * The string representation is used to embed token into strings,
 * and stored to be able to reverse that mapping.
 *
 * All instances of TokenStringMap share the same storage, so that this process
 * works even when different copies of the library are loaded.
 */
class TokenMap {
    constructor() {
        this.stringTokenMap = new Map();
        this.numberTokenMap = new Map();
        this.tokenCounter = 0;
    }
    /**
     * Singleton instance of the token string map
     */
    static instance() {
        if (!glob.__cdktfTokenMap) {
            glob.__cdktfTokenMap = new TokenMap();
        }
        return glob.__cdktfTokenMap;
    }
    /**
     * Generate a unique string for this Token, returning a key
     *
     * Every call for the same Token will produce a new unique string, no
     * attempt is made to deduplicate. Token objects should cache the
     * value themselves, if required.
     *
     * The token can choose (part of) its own representation string with a
     * hint. This may be used to produce aesthetically pleasing and
     * recognizable token representations for humans.
     */
    registerString(token, displayHint) {
        return cachedValue(token, STRING_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return `${encoding_1.BEGIN_STRING_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`;
        });
    }
    /**
     * Generate a unique string for this Token, returning a key
     */
    registerList(token, displayHint) {
        return cachedValue(token, LIST_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return [`${encoding_1.BEGIN_LIST_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`];
        });
    }
    /**
     * Create a unique number representation for this Token and return it
     */
    registerNumberList(token) {
        return cachedValue(token, NUMBER_LIST_SYMBOL, () => {
            const key = this.registerNumberKey(token, true);
            return [key];
        });
    }
    /**
     * Create a unique number representation for this Token and return it
     */
    registerNumber(token) {
        return cachedValue(token, NUMBER_SYMBOL, () => {
            return this.registerNumberKey(token, false);
        });
    }
    /**
     * Generate a unique string for this Token, returning a key
     */
    registerMap(token, mapValue, displayHint) {
        return cachedValue(token, MAP_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return {
                [`${encoding_1.BEGIN_MAP_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`]: mapValue,
            };
        });
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupString(s) {
        const fragments = this.splitString(s);
        if (fragments.tokens.length > 0 && fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupList(xs) {
        if (xs.length !== 1) {
            return undefined;
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const fragments = str.split(this.lookupToken.bind(this));
        if (fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupMap(xs) {
        const keys = Object.keys(xs);
        if (keys.length !== 1) {
            return undefined;
        }
        const str = encoding_1.TokenString.forMapToken(keys[0]);
        const fragments = str.split(this.lookupToken.bind(this));
        if (fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Split a string into literals and Tokens
     */
    splitString(s) {
        const str = encoding_1.TokenString.forString(s);
        return str.split(this.lookupToken.bind(this));
    }
    /**
     * Reverse a number encoding into a Token, or undefined if the number wasn't a Token
     */
    lookupNumberToken(x) {
        const tokenIndex = (0, encoding_1.extractTokenDouble)(x, false);
        if (tokenIndex === undefined) {
            return undefined;
        }
        const t = this.numberTokenMap.get(tokenIndex);
        if (t === undefined) {
            throw (0, errors_1.unknownNumberTokenFound)();
        }
        return t;
    }
    /**
     * Reverse a number encoding into a Token, or undefined if the number wasn't a Token
     */
    lookupNumberList(xs) {
        if (xs.length !== 1) {
            return undefined;
        }
        const tokenIndex = (0, encoding_1.extractTokenDouble)(xs[0], true);
        if (tokenIndex === undefined) {
            return undefined;
        }
        const t = this.numberTokenMap.get(tokenIndex);
        if (t === undefined) {
            throw (0, errors_1.unknownNumberTokenFound)();
        }
        return t;
    }
    /**
     * Find a Token by key.
     *
     * This excludes the token markers.
     */
    lookupToken(key) {
        const token = this.stringTokenMap.get(key);
        if (!token) {
            throw (0, errors_1.unrecognizedTokenKey)(key);
        }
        return token;
    }
    registerStringKey(token, displayHint) {
        const counter = this.tokenCounter++;
        const representation = (displayHint || `TOKEN`).replace(new RegExp(`[^${encoding_1.VALID_KEY_CHARS}]`, "g"), ".");
        const key = `${representation}.${counter}`;
        this.stringTokenMap.set(key, token);
        return key;
    }
    registerNumberKey(token, list) {
        const counter = this.tokenCounter++;
        this.numberTokenMap.set(counter, token);
        return (0, encoding_1.createTokenDouble)(counter, list);
    }
}
exports.TokenMap = TokenMap;
/**
 * Get a cached value for an object, storing it on the object in a symbol
 */
function cachedValue(x, sym, prod) {
    let cached = x[sym];
    if (cached === undefined) {
        cached = prod();
        Object.defineProperty(x, sym, { value: cached });
    }
    return cached;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9rZW4tbWFwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidG9rZW4tbWFwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtCQUErQjtBQUMvQixtQ0FBbUM7QUFDbkMsdUhBQXVIO0FBQ3ZILHlDQUE2RTtBQUc3RSx5Q0FTb0I7QUFFcEIsTUFBTSxJQUFJLEdBQUcsTUFBYSxDQUFDO0FBRTNCLE1BQU0sYUFBYSxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztBQUNoRSxNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLDJCQUEyQixDQUFDLENBQUM7QUFDNUQsTUFBTSxhQUFhLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO0FBQ2hFLE1BQU0sa0JBQWtCLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO0FBQzFFLE1BQU0sVUFBVSxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsQ0FBQztBQUUxRDs7Ozs7Ozs7R0FRRztBQUNILE1BQWEsUUFBUTtJQUFyQjtRQVdtQixtQkFBYyxHQUFHLElBQUksR0FBRyxFQUF1QixDQUFDO1FBQ2hELG1CQUFjLEdBQUcsSUFBSSxHQUFHLEVBQXVCLENBQUM7UUFDekQsaUJBQVksR0FBRyxDQUFDLENBQUM7SUFpTDNCLENBQUM7SUE3TEM7O09BRUc7SUFDSSxNQUFNLENBQUMsUUFBUTtRQUNwQixJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO1lBQzFCLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxRQUFRLEVBQUUsQ0FBQztRQUN4QyxDQUFDO1FBQ0QsT0FBTyxJQUFJLENBQUMsZUFBZSxDQUFDO0lBQzlCLENBQUM7SUFNRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksY0FBYyxDQUFDLEtBQWtCLEVBQUUsV0FBb0I7UUFDNUQsT0FBTyxXQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7WUFDNUMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztZQUN2RCxPQUFPLEdBQUcsb0NBQXlCLEdBQUcsR0FBRyxHQUFHLDJCQUFnQixFQUFFLENBQUM7UUFDakUsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxZQUFZLENBQUMsS0FBa0IsRUFBRSxXQUFvQjtRQUMxRCxPQUFPLFdBQVcsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRTtZQUMxQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBQ3ZELE9BQU8sQ0FBQyxHQUFHLGtDQUF1QixHQUFHLEdBQUcsR0FBRywyQkFBZ0IsRUFBRSxDQUFDLENBQUM7UUFDakUsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxrQkFBa0IsQ0FBQyxLQUFrQjtRQUMxQyxPQUFPLFdBQVcsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUUsR0FBRyxFQUFFO1lBQ2pELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7WUFDaEQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2YsQ0FBQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxjQUFjLENBQUMsS0FBa0I7UUFDdEMsT0FBTyxXQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7WUFDNUMsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQzlDLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVyxDQUNoQixLQUFrQixFQUNsQixRQUFXLEVBQ1gsV0FBb0I7UUFFcEIsT0FBTyxXQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxHQUFHLEVBQUU7WUFDekMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztZQUN2RCxPQUFPO2dCQUNMLENBQUMsR0FBRyxpQ0FBc0IsR0FBRyxHQUFHLEdBQUcsMkJBQWdCLEVBQUUsQ0FBQyxFQUFFLFFBQVE7YUFDakUsQ0FBQztRQUNKLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksWUFBWSxDQUFDLENBQVM7UUFDM0IsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN0QyxJQUFJLFNBQVMsQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQzFELE9BQU8sU0FBUyxDQUFDLFVBQVUsQ0FBQztRQUM5QixDQUFDO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksVUFBVSxDQUFDLEVBQVk7UUFDNUIsSUFBSSxFQUFFLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQ3BCLE9BQU8sU0FBUyxDQUFDO1FBQ25CLENBQUM7UUFDRCxNQUFNLEdBQUcsR0FBRyxzQkFBVyxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1QyxNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDekQsSUFBSSxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQzNCLE9BQU8sU0FBUyxDQUFDLFVBQVUsQ0FBQztRQUM5QixDQUFDO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksU0FBUyxDQUFDLEVBQTBCO1FBQ3pDLE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDN0IsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQ3RCLE9BQU8sU0FBUyxDQUFDO1FBQ25CLENBQUM7UUFDRCxNQUFNLEdBQUcsR0FBRyxzQkFBVyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3QyxNQUFNLFNBQVMsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDekQsSUFBSSxTQUFTLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1lBQzNCLE9BQU8sU0FBUyxDQUFDLFVBQVUsQ0FBQztRQUM5QixDQUFDO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVyxDQUFDLENBQVM7UUFDMUIsTUFBTSxHQUFHLEdBQUcsc0JBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckMsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVEOztPQUVHO0lBQ0ksaUJBQWlCLENBQUMsQ0FBUztRQUNoQyxNQUFNLFVBQVUsR0FBRyxJQUFBLDZCQUFrQixFQUFDLENBQUMsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUNoRCxJQUFJLFVBQVUsS0FBSyxTQUFTLEVBQUUsQ0FBQztZQUM3QixPQUFPLFNBQVMsQ0FBQztRQUNuQixDQUFDO1FBQ0QsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFLENBQUM7WUFDcEIsTUFBTSxJQUFBLGdDQUF1QixHQUFFLENBQUM7UUFDbEMsQ0FBQztRQUNELE9BQU8sQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUVEOztPQUVHO0lBQ0ksZ0JBQWdCLENBQUMsRUFBWTtRQUNsQyxJQUFJLEVBQUUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDcEIsT0FBTyxTQUFTLENBQUM7UUFDbkIsQ0FBQztRQUNELE1BQU0sVUFBVSxHQUFHLElBQUEsNkJBQWtCLEVBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ25ELElBQUksVUFBVSxLQUFLLFNBQVMsRUFBRSxDQUFDO1lBQzdCLE9BQU8sU0FBUyxDQUFDO1FBQ25CLENBQUM7UUFDRCxNQUFNLENBQUMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsS0FBSyxTQUFTLEVBQUUsQ0FBQztZQUNwQixNQUFNLElBQUEsZ0NBQXVCLEdBQUUsQ0FBQztRQUNsQyxDQUFDO1FBQ0QsT0FBTyxDQUFDLENBQUM7SUFDWCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFdBQVcsQ0FBQyxHQUFXO1FBQzVCLE1BQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNYLE1BQU0sSUFBQSw2QkFBb0IsRUFBQyxHQUFHLENBQUMsQ0FBQztRQUNsQyxDQUFDO1FBQ0QsT0FBTyxLQUFLLENBQUM7SUFDZixDQUFDO0lBRU8saUJBQWlCLENBQUMsS0FBa0IsRUFBRSxXQUFvQjtRQUNoRSxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsWUFBWSxFQUFFLENBQUM7UUFDcEMsTUFBTSxjQUFjLEdBQUcsQ0FBQyxXQUFXLElBQUksT0FBTyxDQUFDLENBQUMsT0FBTyxDQUNyRCxJQUFJLE1BQU0sQ0FBQyxLQUFLLDBCQUFlLEdBQUcsRUFBRSxHQUFHLENBQUMsRUFDeEMsR0FBRyxDQUNKLENBQUM7UUFDRixNQUFNLEdBQUcsR0FBRyxHQUFHLGNBQWMsSUFBSSxPQUFPLEVBQUUsQ0FBQztRQUMzQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDcEMsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0lBRU8saUJBQWlCLENBQUMsS0FBa0IsRUFBRSxJQUFhO1FBQ3pELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFBLDRCQUFpQixFQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQztJQUMxQyxDQUFDO0NBQ0Y7QUE5TEQsNEJBOExDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLFdBQVcsQ0FBc0IsQ0FBSSxFQUFFLEdBQVcsRUFBRSxJQUFhO0lBQ3hFLElBQUksTUFBTSxHQUFJLENBQVMsQ0FBQyxHQUFVLENBQUMsQ0FBQztJQUNwQyxJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUUsQ0FBQztRQUN6QixNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUM7UUFDaEIsTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFDLEVBQUUsR0FBRyxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUNELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgKGMpIEhhc2hpQ29ycCwgSW5jXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTVBMLTIuMFxuLy8gY29waWVkIGZyb20gaHR0cHM6Ly9naXRodWIuY29tL2F3cy9jb25zdHJ1Y3RzL2Jsb2IvZTAxZTQ3Zjc4ZWYxZTliNjAwZWZjZDIzZmY3NzA1YWE4ZDM4NDAxNy9saWIvcHJpdmF0ZS90b2tlbi1tYXAudHNcbmltcG9ydCB7IHVua25vd25OdW1iZXJUb2tlbkZvdW5kLCB1bnJlY29nbml6ZWRUb2tlbktleSB9IGZyb20gXCIuLi8uLi9lcnJvcnNcIjtcbmltcG9ydCB7IElSZXNvbHZhYmxlIH0gZnJvbSBcIi4uL3Jlc29sdmFibGVcIjtcbmltcG9ydCB7IFRva2VuaXplZFN0cmluZ0ZyYWdtZW50cyB9IGZyb20gXCIuLi9zdHJpbmctZnJhZ21lbnRzXCI7XG5pbXBvcnQge1xuICBCRUdJTl9MSVNUX1RPS0VOX01BUktFUixcbiAgQkVHSU5fU1RSSU5HX1RPS0VOX01BUktFUixcbiAgY3JlYXRlVG9rZW5Eb3VibGUsXG4gIEVORF9UT0tFTl9NQVJLRVIsXG4gIGV4dHJhY3RUb2tlbkRvdWJsZSxcbiAgVG9rZW5TdHJpbmcsXG4gIFZBTElEX0tFWV9DSEFSUyxcbiAgQkVHSU5fTUFQX1RPS0VOX01BUktFUixcbn0gZnJvbSBcIi4vZW5jb2RpbmdcIjtcblxuY29uc3QgZ2xvYiA9IGdsb2JhbCBhcyBhbnk7XG5cbmNvbnN0IFNUUklOR19TWU1CT0wgPSBTeW1ib2wuZm9yKFwiQGNka3RmL2NvcmUuVG9rZW5NYXAuU1RSSU5HXCIpO1xuY29uc3QgTElTVF9TWU1CT0wgPSBTeW1ib2wuZm9yKFwiQGNka3RmL2NvcmUuVG9rZW5NYXAuTElTVFwiKTtcbmNvbnN0IE5VTUJFUl9TWU1CT0wgPSBTeW1ib2wuZm9yKFwiQGNka3RmL2NvcmUuVG9rZW5NYXAuTlVNQkVSXCIpO1xuY29uc3QgTlVNQkVSX0xJU1RfU1lNQk9MID0gU3ltYm9sLmZvcihcIkBjZGt0Zi9jb3JlLlRva2VuTWFwLk5VTUJFUl9MSVNUXCIpO1xuY29uc3QgTUFQX1NZTUJPTCA9IFN5bWJvbC5mb3IoXCJAY2RrdGYvY29yZS5Ub2tlbk1hcC5NQVBcIik7XG5cbi8qKlxuICogQ2VudHJhbCBwbGFjZSB3aGVyZSB3ZSBrZWVwIGEgbWFwcGluZyBmcm9tIFRva2VucyB0byB0aGVpciBTdHJpbmcgcmVwcmVzZW50YXRpb25cbiAqXG4gKiBUaGUgc3RyaW5nIHJlcHJlc2VudGF0aW9uIGlzIHVzZWQgdG8gZW1iZWQgdG9rZW4gaW50byBzdHJpbmdzLFxuICogYW5kIHN0b3JlZCB0byBiZSBhYmxlIHRvIHJldmVyc2UgdGhhdCBtYXBwaW5nLlxuICpcbiAqIEFsbCBpbnN0YW5jZXMgb2YgVG9rZW5TdHJpbmdNYXAgc2hhcmUgdGhlIHNhbWUgc3RvcmFnZSwgc28gdGhhdCB0aGlzIHByb2Nlc3NcbiAqIHdvcmtzIGV2ZW4gd2hlbiBkaWZmZXJlbnQgY29waWVzIG9mIHRoZSBsaWJyYXJ5IGFyZSBsb2FkZWQuXG4gKi9cbmV4cG9ydCBjbGFzcyBUb2tlbk1hcCB7XG4gIC8qKlxuICAgKiBTaW5nbGV0b24gaW5zdGFuY2Ugb2YgdGhlIHRva2VuIHN0cmluZyBtYXBcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgaW5zdGFuY2UoKTogVG9rZW5NYXAge1xuICAgIGlmICghZ2xvYi5fX2Nka3RmVG9rZW5NYXApIHtcbiAgICAgIGdsb2IuX19jZGt0ZlRva2VuTWFwID0gbmV3IFRva2VuTWFwKCk7XG4gICAgfVxuICAgIHJldHVybiBnbG9iLl9fY2RrdGZUb2tlbk1hcDtcbiAgfVxuXG4gIHByaXZhdGUgcmVhZG9ubHkgc3RyaW5nVG9rZW5NYXAgPSBuZXcgTWFwPHN0cmluZywgSVJlc29sdmFibGU+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgbnVtYmVyVG9rZW5NYXAgPSBuZXcgTWFwPG51bWJlciwgSVJlc29sdmFibGU+KCk7XG4gIHByaXZhdGUgdG9rZW5Db3VudGVyID0gMDtcblxuICAvKipcbiAgICogR2VuZXJhdGUgYSB1bmlxdWUgc3RyaW5nIGZvciB0aGlzIFRva2VuLCByZXR1cm5pbmcgYSBrZXlcbiAgICpcbiAgICogRXZlcnkgY2FsbCBmb3IgdGhlIHNhbWUgVG9rZW4gd2lsbCBwcm9kdWNlIGEgbmV3IHVuaXF1ZSBzdHJpbmcsIG5vXG4gICAqIGF0dGVtcHQgaXMgbWFkZSB0byBkZWR1cGxpY2F0ZS4gVG9rZW4gb2JqZWN0cyBzaG91bGQgY2FjaGUgdGhlXG4gICAqIHZhbHVlIHRoZW1zZWx2ZXMsIGlmIHJlcXVpcmVkLlxuICAgKlxuICAgKiBUaGUgdG9rZW4gY2FuIGNob29zZSAocGFydCBvZikgaXRzIG93biByZXByZXNlbnRhdGlvbiBzdHJpbmcgd2l0aCBhXG4gICAqIGhpbnQuIFRoaXMgbWF5IGJlIHVzZWQgdG8gcHJvZHVjZSBhZXN0aGV0aWNhbGx5IHBsZWFzaW5nIGFuZFxuICAgKiByZWNvZ25pemFibGUgdG9rZW4gcmVwcmVzZW50YXRpb25zIGZvciBodW1hbnMuXG4gICAqL1xuICBwdWJsaWMgcmVnaXN0ZXJTdHJpbmcodG9rZW46IElSZXNvbHZhYmxlLCBkaXNwbGF5SGludD86IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGNhY2hlZFZhbHVlKHRva2VuLCBTVFJJTkdfU1lNQk9MLCAoKSA9PiB7XG4gICAgICBjb25zdCBrZXkgPSB0aGlzLnJlZ2lzdGVyU3RyaW5nS2V5KHRva2VuLCBkaXNwbGF5SGludCk7XG4gICAgICByZXR1cm4gYCR7QkVHSU5fU1RSSU5HX1RPS0VOX01BUktFUn0ke2tleX0ke0VORF9UT0tFTl9NQVJLRVJ9YDtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBHZW5lcmF0ZSBhIHVuaXF1ZSBzdHJpbmcgZm9yIHRoaXMgVG9rZW4sIHJldHVybmluZyBhIGtleVxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyTGlzdCh0b2tlbjogSVJlc29sdmFibGUsIGRpc3BsYXlIaW50Pzogc3RyaW5nKTogc3RyaW5nW10ge1xuICAgIHJldHVybiBjYWNoZWRWYWx1ZSh0b2tlbiwgTElTVF9TWU1CT0wsICgpID0+IHtcbiAgICAgIGNvbnN0IGtleSA9IHRoaXMucmVnaXN0ZXJTdHJpbmdLZXkodG9rZW4sIGRpc3BsYXlIaW50KTtcbiAgICAgIHJldHVybiBbYCR7QkVHSU5fTElTVF9UT0tFTl9NQVJLRVJ9JHtrZXl9JHtFTkRfVE9LRU5fTUFSS0VSfWBdO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHVuaXF1ZSBudW1iZXIgcmVwcmVzZW50YXRpb24gZm9yIHRoaXMgVG9rZW4gYW5kIHJldHVybiBpdFxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyTnVtYmVyTGlzdCh0b2tlbjogSVJlc29sdmFibGUpOiBudW1iZXJbXSB7XG4gICAgcmV0dXJuIGNhY2hlZFZhbHVlKHRva2VuLCBOVU1CRVJfTElTVF9TWU1CT0wsICgpID0+IHtcbiAgICAgIGNvbnN0IGtleSA9IHRoaXMucmVnaXN0ZXJOdW1iZXJLZXkodG9rZW4sIHRydWUpO1xuICAgICAgcmV0dXJuIFtrZXldO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHVuaXF1ZSBudW1iZXIgcmVwcmVzZW50YXRpb24gZm9yIHRoaXMgVG9rZW4gYW5kIHJldHVybiBpdFxuICAgKi9cbiAgcHVibGljIHJlZ2lzdGVyTnVtYmVyKHRva2VuOiBJUmVzb2x2YWJsZSk6IG51bWJlciB7XG4gICAgcmV0dXJuIGNhY2hlZFZhbHVlKHRva2VuLCBOVU1CRVJfU1lNQk9MLCAoKSA9PiB7XG4gICAgICByZXR1cm4gdGhpcy5yZWdpc3Rlck51bWJlcktleSh0b2tlbiwgZmFsc2UpO1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdlbmVyYXRlIGEgdW5pcXVlIHN0cmluZyBmb3IgdGhpcyBUb2tlbiwgcmV0dXJuaW5nIGEga2V5XG4gICAqL1xuICBwdWJsaWMgcmVnaXN0ZXJNYXA8Vj4oXG4gICAgdG9rZW46IElSZXNvbHZhYmxlLFxuICAgIG1hcFZhbHVlOiBWLFxuICAgIGRpc3BsYXlIaW50Pzogc3RyaW5nLFxuICApOiB7IFtrZXk6IHN0cmluZ106IFYgfSB7XG4gICAgcmV0dXJuIGNhY2hlZFZhbHVlKHRva2VuLCBNQVBfU1lNQk9MLCAoKSA9PiB7XG4gICAgICBjb25zdCBrZXkgPSB0aGlzLnJlZ2lzdGVyU3RyaW5nS2V5KHRva2VuLCBkaXNwbGF5SGludCk7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBbYCR7QkVHSU5fTUFQX1RPS0VOX01BUktFUn0ke2tleX0ke0VORF9UT0tFTl9NQVJLRVJ9YF06IG1hcFZhbHVlLFxuICAgICAgfTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXZlcnNlIGEgc3RyaW5nIHJlcHJlc2VudGF0aW9uIGludG8gYSBUb2tlbiBvYmplY3RcbiAgICovXG4gIHB1YmxpYyBsb29rdXBTdHJpbmcoczogc3RyaW5nKTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgIGNvbnN0IGZyYWdtZW50cyA9IHRoaXMuc3BsaXRTdHJpbmcocyk7XG4gICAgaWYgKGZyYWdtZW50cy50b2tlbnMubGVuZ3RoID4gMCAmJiBmcmFnbWVudHMubGVuZ3RoID09PSAxKSB7XG4gICAgICByZXR1cm4gZnJhZ21lbnRzLmZpcnN0VG9rZW47XG4gICAgfVxuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICAvKipcbiAgICogUmV2ZXJzZSBhIHN0cmluZyByZXByZXNlbnRhdGlvbiBpbnRvIGEgVG9rZW4gb2JqZWN0XG4gICAqL1xuICBwdWJsaWMgbG9va3VwTGlzdCh4czogc3RyaW5nW10pOiBJUmVzb2x2YWJsZSB8IHVuZGVmaW5lZCB7XG4gICAgaWYgKHhzLmxlbmd0aCAhPT0gMSkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgY29uc3Qgc3RyID0gVG9rZW5TdHJpbmcuZm9yTGlzdFRva2VuKHhzWzBdKTtcbiAgICBjb25zdCBmcmFnbWVudHMgPSBzdHIuc3BsaXQodGhpcy5sb29rdXBUb2tlbi5iaW5kKHRoaXMpKTtcbiAgICBpZiAoZnJhZ21lbnRzLmxlbmd0aCA9PT0gMSkge1xuICAgICAgcmV0dXJuIGZyYWdtZW50cy5maXJzdFRva2VuO1xuICAgIH1cbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldmVyc2UgYSBzdHJpbmcgcmVwcmVzZW50YXRpb24gaW50byBhIFRva2VuIG9iamVjdFxuICAgKi9cbiAgcHVibGljIGxvb2t1cE1hcCh4czogeyBba2V5OiBzdHJpbmddOiBhbnkgfSk6IElSZXNvbHZhYmxlIHwgdW5kZWZpbmVkIHtcbiAgICBjb25zdCBrZXlzID0gT2JqZWN0LmtleXMoeHMpO1xuICAgIGlmIChrZXlzLmxlbmd0aCAhPT0gMSkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgY29uc3Qgc3RyID0gVG9rZW5TdHJpbmcuZm9yTWFwVG9rZW4oa2V5c1swXSk7XG4gICAgY29uc3QgZnJhZ21lbnRzID0gc3RyLnNwbGl0KHRoaXMubG9va3VwVG9rZW4uYmluZCh0aGlzKSk7XG4gICAgaWYgKGZyYWdtZW50cy5sZW5ndGggPT09IDEpIHtcbiAgICAgIHJldHVybiBmcmFnbWVudHMuZmlyc3RUb2tlbjtcbiAgICB9XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBTcGxpdCBhIHN0cmluZyBpbnRvIGxpdGVyYWxzIGFuZCBUb2tlbnNcbiAgICovXG4gIHB1YmxpYyBzcGxpdFN0cmluZyhzOiBzdHJpbmcpOiBUb2tlbml6ZWRTdHJpbmdGcmFnbWVudHMge1xuICAgIGNvbnN0IHN0ciA9IFRva2VuU3RyaW5nLmZvclN0cmluZyhzKTtcbiAgICByZXR1cm4gc3RyLnNwbGl0KHRoaXMubG9va3VwVG9rZW4uYmluZCh0aGlzKSk7XG4gIH1cblxuICAvKipcbiAgICogUmV2ZXJzZSBhIG51bWJlciBlbmNvZGluZyBpbnRvIGEgVG9rZW4sIG9yIHVuZGVmaW5lZCBpZiB0aGUgbnVtYmVyIHdhc24ndCBhIFRva2VuXG4gICAqL1xuICBwdWJsaWMgbG9va3VwTnVtYmVyVG9rZW4oeDogbnVtYmVyKTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgIGNvbnN0IHRva2VuSW5kZXggPSBleHRyYWN0VG9rZW5Eb3VibGUoeCwgZmFsc2UpO1xuICAgIGlmICh0b2tlbkluZGV4ID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIGNvbnN0IHQgPSB0aGlzLm51bWJlclRva2VuTWFwLmdldCh0b2tlbkluZGV4KTtcbiAgICBpZiAodCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyB1bmtub3duTnVtYmVyVG9rZW5Gb3VuZCgpO1xuICAgIH1cbiAgICByZXR1cm4gdDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXZlcnNlIGEgbnVtYmVyIGVuY29kaW5nIGludG8gYSBUb2tlbiwgb3IgdW5kZWZpbmVkIGlmIHRoZSBudW1iZXIgd2Fzbid0IGEgVG9rZW5cbiAgICovXG4gIHB1YmxpYyBsb29rdXBOdW1iZXJMaXN0KHhzOiBudW1iZXJbXSk6IElSZXNvbHZhYmxlIHwgdW5kZWZpbmVkIHtcbiAgICBpZiAoeHMubGVuZ3RoICE9PSAxKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBjb25zdCB0b2tlbkluZGV4ID0gZXh0cmFjdFRva2VuRG91YmxlKHhzWzBdLCB0cnVlKTtcbiAgICBpZiAodG9rZW5JbmRleCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICBjb25zdCB0ID0gdGhpcy5udW1iZXJUb2tlbk1hcC5nZXQodG9rZW5JbmRleCk7XG4gICAgaWYgKHQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgdW5rbm93bk51bWJlclRva2VuRm91bmQoKTtcbiAgICB9XG4gICAgcmV0dXJuIHQ7XG4gIH1cblxuICAvKipcbiAgICogRmluZCBhIFRva2VuIGJ5IGtleS5cbiAgICpcbiAgICogVGhpcyBleGNsdWRlcyB0aGUgdG9rZW4gbWFya2Vycy5cbiAgICovXG4gIHB1YmxpYyBsb29rdXBUb2tlbihrZXk6IHN0cmluZyk6IElSZXNvbHZhYmxlIHtcbiAgICBjb25zdCB0b2tlbiA9IHRoaXMuc3RyaW5nVG9rZW5NYXAuZ2V0KGtleSk7XG4gICAgaWYgKCF0b2tlbikge1xuICAgICAgdGhyb3cgdW5yZWNvZ25pemVkVG9rZW5LZXkoa2V5KTtcbiAgICB9XG4gICAgcmV0dXJuIHRva2VuO1xuICB9XG5cbiAgcHJpdmF0ZSByZWdpc3RlclN0cmluZ0tleSh0b2tlbjogSVJlc29sdmFibGUsIGRpc3BsYXlIaW50Pzogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBjb25zdCBjb3VudGVyID0gdGhpcy50b2tlbkNvdW50ZXIrKztcbiAgICBjb25zdCByZXByZXNlbnRhdGlvbiA9IChkaXNwbGF5SGludCB8fCBgVE9LRU5gKS5yZXBsYWNlKFxuICAgICAgbmV3IFJlZ0V4cChgW14ke1ZBTElEX0tFWV9DSEFSU31dYCwgXCJnXCIpLFxuICAgICAgXCIuXCIsXG4gICAgKTtcbiAgICBjb25zdCBrZXkgPSBgJHtyZXByZXNlbnRhdGlvbn0uJHtjb3VudGVyfWA7XG4gICAgdGhpcy5zdHJpbmdUb2tlbk1hcC5zZXQoa2V5LCB0b2tlbik7XG4gICAgcmV0dXJuIGtleTtcbiAgfVxuXG4gIHByaXZhdGUgcmVnaXN0ZXJOdW1iZXJLZXkodG9rZW46IElSZXNvbHZhYmxlLCBsaXN0OiBib29sZWFuKTogbnVtYmVyIHtcbiAgICBjb25zdCBjb3VudGVyID0gdGhpcy50b2tlbkNvdW50ZXIrKztcbiAgICB0aGlzLm51bWJlclRva2VuTWFwLnNldChjb3VudGVyLCB0b2tlbik7XG4gICAgcmV0dXJuIGNyZWF0ZVRva2VuRG91YmxlKGNvdW50ZXIsIGxpc3QpO1xuICB9XG59XG5cbi8qKlxuICogR2V0IGEgY2FjaGVkIHZhbHVlIGZvciBhbiBvYmplY3QsIHN0b3JpbmcgaXQgb24gdGhlIG9iamVjdCBpbiBhIHN5bWJvbFxuICovXG5mdW5jdGlvbiBjYWNoZWRWYWx1ZTxBIGV4dGVuZHMgb2JqZWN0LCBCPih4OiBBLCBzeW06IHN5bWJvbCwgcHJvZDogKCkgPT4gQikge1xuICBsZXQgY2FjaGVkID0gKHggYXMgYW55KVtzeW0gYXMgYW55XTtcbiAgaWYgKGNhY2hlZCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgY2FjaGVkID0gcHJvZCgpO1xuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eSh4LCBzeW0sIHsgdmFsdWU6IGNhY2hlZCB9KTtcbiAgfVxuICByZXR1cm4gY2FjaGVkO1xufVxuIl19