"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RememberingTokenResolver = void 0;
exports.resolve = resolve;
exports.findTokens = findTokens;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
const token_1 = require("../token");
const errors_1 = require("../../errors");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = "/" + prefix.join("/");
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            suppressBraces: options.previousContext?.suppressBraces,
            ignoreEscapes: options.previousContext?.ignoreEscapes,
            warnEscapes: options.previousContext?.warnEscapes,
            iteratorContext: options.previousContext?.iteratorContext,
            registerPostProcessor(pp) {
                postProcessor = pp;
            },
            resolve(x) {
                return resolve(x, {
                    ...options,
                    prefix: newPrefix,
                    previousContext: context,
                });
            },
        };
        return [
            context,
            {
                postProcess(x) {
                    return postProcessor ? postProcessor.postProcess(x, context) : x;
                },
            },
        ];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw (0, errors_1.unableToResolveCircularReference)(pathName);
    }
    //
    // undefined
    //
    if (typeof obj === "undefined") {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof obj === "function") {
        throw (0, errors_1.cannotResolveFunction)(pathName, obj);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof obj === "string") {
        // If this is a "list element" Token, it should never occur by itself in string context
        if (encoding_1.TokenString.forListToken(obj).test()) {
            throw (0, errors_1.encodedListTokenInScalarStringContext)();
        }
        if (obj === token_1.Token.STRING_MAP_TOKEN_VALUE ||
            obj === token_1.Token.ANY_MAP_TOKEN_VALUE) {
            throw (0, errors_1.encodedMapTokenInScalarStringContext)();
        }
        let str = obj;
        const context = makeContext()[0];
        const tokenStr = encoding_1.TokenString.forString(str, !context.ignoreEscapes, context.warnEscapes);
        if (tokenStr.test()) {
            const fragments = tokenStr.split(tokenMap.lookupToken.bind(tokenMap));
            str = options.resolver.resolveString(fragments, makeContext()[0]);
        }
        // replace concatenated token numbers
        const tokenNumberStr = encoding_1.TokenString.forNumbers(str);
        if (tokenNumberStr.test()) {
            const fragments = tokenNumberStr.split((id) => {
                return token_map_1.TokenMap.instance().lookupNumberToken(parseFloat(id));
            });
            const context = makeContext()[0];
            str = fragments.mapTokens(context).join(new resolvable_1.StringConcat());
        }
        return str;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof obj === "number") {
        if (obj === token_1.Token.NUMBER_MAP_TOKEN_VALUE) {
            throw (0, errors_1.encodedMapTokenInScalarNumberContext)();
        }
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    // eslint-disable-next-line no-instanceof/no-instanceof
    if (typeof obj !== "object" || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if ((0, encoding_1.containsStringListTokenElement)(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        if ((0, encoding_1.containsNumberListTokenElement)(obj)) {
            return options.resolver.resolveNumberList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter((x) => typeof x !== "undefined");
        return arr;
    }
    // check for tokenized map
    if ((0, encoding_1.containsMapToken)(obj)) {
        return options.resolver.resolveMap(obj, makeContext()[0]);
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if ((0, encoding_1.unresolved)(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw (0, errors_1.constructsCannotBeResolved)(pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof resolvedKey !== "string") {
            throw (0, errors_1.mapKeyMustResolveToString)(pathName, key, resolvedKey);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof value === "undefined") {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
/**
 * Resolves a number token
 */
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb2x2ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc29sdmUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBMkRBLDBCQXFNQztBQUtELGdDQU1DO0FBdlFELDhDQU91QjtBQUV2Qix5Q0FNb0I7QUFDcEIsMkNBQXVDO0FBQ3ZDLG9DQUFpQztBQUNqQyx5Q0FRc0I7QUFFdEIscUdBQXFHO0FBRXJHLE1BQU0sUUFBUSxHQUFHLG9CQUFRLENBQUMsUUFBUSxFQUFFLENBQUM7QUFrQnJDOzs7Ozs7R0FNRztBQUNILFNBQWdCLE9BQU8sQ0FBQyxHQUFRLEVBQUUsT0FBd0I7SUFDeEQsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUM7SUFDcEMsTUFBTSxRQUFRLEdBQUcsR0FBRyxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFeEM7O09BRUc7SUFDSCxTQUFTLFdBQVcsQ0FBQyxVQUFtQjtRQUN0QyxNQUFNLFNBQVMsR0FDYixVQUFVLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUUxRSxJQUFJLGFBQXlDLENBQUM7UUFFOUMsTUFBTSxPQUFPLEdBQW9CO1lBQy9CLFNBQVMsRUFBRSxPQUFPLENBQUMsU0FBUztZQUM1QixLQUFLLEVBQUUsT0FBTyxDQUFDLEtBQUs7WUFDcEIsY0FBYyxFQUFFLE9BQU8sQ0FBQyxlQUFlLEVBQUUsY0FBYztZQUN2RCxhQUFhLEVBQUUsT0FBTyxDQUFDLGVBQWUsRUFBRSxhQUFhO1lBQ3JELFdBQVcsRUFBRSxPQUFPLENBQUMsZUFBZSxFQUFFLFdBQVc7WUFDakQsZUFBZSxFQUFFLE9BQU8sQ0FBQyxlQUFlLEVBQUUsZUFBZTtZQUN6RCxxQkFBcUIsQ0FBQyxFQUFFO2dCQUN0QixhQUFhLEdBQUcsRUFBRSxDQUFDO1lBQ3JCLENBQUM7WUFDRCxPQUFPLENBQUMsQ0FBTTtnQkFDWixPQUFPLE9BQU8sQ0FBQyxDQUFDLEVBQUU7b0JBQ2hCLEdBQUcsT0FBTztvQkFDVixNQUFNLEVBQUUsU0FBUztvQkFDakIsZUFBZSxFQUFFLE9BQU87aUJBQ3pCLENBQUMsQ0FBQztZQUNMLENBQUM7U0FDRixDQUFDO1FBRUYsT0FBTztZQUNMLE9BQU87WUFDUDtnQkFDRSxXQUFXLENBQUMsQ0FBQztvQkFDWCxPQUFPLGFBQWEsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDbkUsQ0FBQzthQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRCx1REFBdUQ7SUFDdkQsSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLEdBQUcsRUFBRSxDQUFDO1FBQ3hCLE1BQU0sSUFBQSx5Q0FBZ0MsRUFBQyxRQUFRLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRUQsRUFBRTtJQUNGLFlBQVk7SUFDWixFQUFFO0lBRUYsSUFBSSxPQUFPLEdBQUcsS0FBSyxXQUFXLEVBQUUsQ0FBQztRQUMvQixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQsRUFBRTtJQUNGLE9BQU87SUFDUCxFQUFFO0lBRUYsSUFBSSxHQUFHLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDakIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsRUFBRTtJQUNGLHdEQUF3RDtJQUN4RCxFQUFFO0lBRUYsSUFBSSxPQUFPLEdBQUcsS0FBSyxVQUFVLEVBQUUsQ0FBQztRQUM5QixNQUFNLElBQUEsOEJBQXFCLEVBQUMsUUFBUSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRCxFQUFFO0lBQ0Ysc0RBQXNEO0lBQ3RELEVBQUU7SUFDRixJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsRUFBRSxDQUFDO1FBQzVCLHVGQUF1RjtRQUN2RixJQUFJLHNCQUFXLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUM7WUFDekMsTUFBTSxJQUFBLDhDQUFxQyxHQUFFLENBQUM7UUFDaEQsQ0FBQztRQUVELElBQ0UsR0FBRyxLQUFLLGFBQUssQ0FBQyxzQkFBc0I7WUFDcEMsR0FBRyxLQUFLLGFBQUssQ0FBQyxtQkFBbUIsRUFDakMsQ0FBQztZQUNELE1BQU0sSUFBQSw2Q0FBb0MsR0FBRSxDQUFDO1FBQy9DLENBQUM7UUFFRCxJQUFJLEdBQUcsR0FBVyxHQUFHLENBQUM7UUFFdEIsTUFBTSxPQUFPLEdBQUcsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDakMsTUFBTSxRQUFRLEdBQUcsc0JBQVcsQ0FBQyxTQUFTLENBQ3BDLEdBQUcsRUFDSCxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQ3RCLE9BQU8sQ0FBQyxXQUFXLENBQ3BCLENBQUM7UUFDRixJQUFJLFFBQVEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDO1lBQ3BCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztZQUN0RSxHQUFHLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEUsQ0FBQztRQUVELHFDQUFxQztRQUNyQyxNQUFNLGNBQWMsR0FBRyxzQkFBVyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNuRCxJQUFJLGNBQWMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDO1lBQzFCLE1BQU0sU0FBUyxHQUFHLGNBQWMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRTtnQkFDNUMsT0FBTyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQy9ELENBQUMsQ0FBQyxDQUFDO1lBRUgsTUFBTSxPQUFPLEdBQUcsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDakMsR0FBRyxHQUFHLFNBQVMsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUkseUJBQVksRUFBRSxDQUFDLENBQUM7UUFDOUQsQ0FBQztRQUVELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztJQUVELEVBQUU7SUFDRiwrQ0FBK0M7SUFDL0MsRUFBRTtJQUNGLElBQUksT0FBTyxHQUFHLEtBQUssUUFBUSxFQUFFLENBQUM7UUFDNUIsSUFBSSxHQUFHLEtBQUssYUFBSyxDQUFDLHNCQUFzQixFQUFFLENBQUM7WUFDekMsTUFBTSxJQUFBLDZDQUFvQyxHQUFFLENBQUM7UUFDL0MsQ0FBQztRQUVELE9BQU8sa0JBQWtCLENBQUMsR0FBRyxFQUFFLFdBQVcsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVELEVBQUU7SUFDRixxQkFBcUI7SUFDckIsRUFBRTtJQUVGLHVEQUF1RDtJQUN2RCxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxHQUFHLFlBQVksSUFBSSxFQUFFLENBQUM7UUFDbkQsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0lBRUQsRUFBRTtJQUNGLHdFQUF3RTtJQUN4RSxFQUFFO0lBRUYsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7UUFDdkIsSUFBSSxJQUFBLHlDQUE4QixFQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDeEMsT0FBTyxPQUFPLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM3RCxDQUFDO1FBRUQsSUFBSSxJQUFBLHlDQUE4QixFQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDeEMsT0FBTyxPQUFPLENBQUMsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEdBQUcsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25FLENBQUM7UUFFRCxNQUFNLEdBQUcsR0FBRyxHQUFHO2FBQ1osR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDaEQsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxPQUFPLENBQUMsS0FBSyxXQUFXLENBQUMsQ0FBQztRQUUzQyxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFRCwwQkFBMEI7SUFDMUIsSUFBSSxJQUFBLDJCQUFnQixFQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7UUFDMUIsT0FBTyxPQUFPLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxHQUFHLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBRUQsRUFBRTtJQUNGLGdFQUFnRTtJQUNoRSxFQUFFO0lBRUYsSUFBSSxJQUFBLHFCQUFVLEVBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztRQUNwQixNQUFNLENBQUMsT0FBTyxFQUFFLGFBQWEsQ0FBQyxHQUFHLFdBQVcsRUFBRSxDQUFDO1FBQy9DLE9BQU8sT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxhQUFhLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQsRUFBRTtJQUNGLG9DQUFvQztJQUNwQyxFQUFFO0lBRUYsNEVBQTRFO0lBQzVFLDhFQUE4RTtJQUM5RSxxQ0FBcUM7SUFDckMsSUFBSSxXQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztRQUNyQixNQUFNLElBQUEsbUNBQTBCLEVBQUMsUUFBUSxDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVELE1BQU0sTUFBTSxHQUFRLEVBQUUsQ0FBQztJQUN2QixLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztRQUNuQyxNQUFNLFdBQVcsR0FBRyxXQUFXLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDbEQsSUFBSSxPQUFPLFdBQVcsS0FBSyxRQUFRLEVBQUUsQ0FBQztZQUNwQyxNQUFNLElBQUEsa0NBQXlCLEVBQUMsUUFBUSxFQUFFLEdBQUcsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM5RCxDQUFDO1FBRUQsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUVwRCxpQkFBaUI7UUFDakIsSUFBSSxPQUFPLEtBQUssS0FBSyxXQUFXLEVBQUUsQ0FBQztZQUNqQyxTQUFTO1FBQ1gsQ0FBQztRQUVELE1BQU0sQ0FBQyxXQUFXLENBQUMsR0FBRyxLQUFLLENBQUM7SUFDOUIsQ0FBQztJQUVELE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLFVBQVUsQ0FBQyxLQUFpQixFQUFFLEVBQWE7SUFDekQsTUFBTSxRQUFRLEdBQUcsSUFBSSx3QkFBd0IsQ0FBQyxJQUFJLHlCQUFZLEVBQUUsQ0FBQyxDQUFDO0lBRWxFLE9BQU8sQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztJQUVoRSxPQUFPLFFBQVEsQ0FBQyxNQUFNLENBQUM7QUFDekIsQ0FBQztBQUVEOztHQUVHO0FBQ0gsTUFBYSx3QkFBeUIsU0FBUSxpQ0FBb0I7SUFBbEU7O1FBQ21CLGVBQVUsR0FBRyxJQUFJLEdBQUcsRUFBZSxDQUFDO0lBbUJ2RCxDQUFDO0lBakJRLFlBQVksQ0FDakIsQ0FBYyxFQUNkLE9BQXdCLEVBQ3hCLGFBQTZCO1FBRTdCLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3ZCLE9BQU8sS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLEVBQUUsT0FBTyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0lBQ3ZELENBQUM7SUFFTSxhQUFhLENBQUMsQ0FBMkIsRUFBRSxPQUF3QjtRQUN4RSxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUM1QyxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFRCxJQUFXLE1BQU07UUFDZixPQUFPLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7Q0FDRjtBQXBCRCw0REFvQkM7QUFFRDs7Ozs7R0FLRztBQUNILFNBQVMsV0FBVyxDQUFDLENBQU07SUFDekIsT0FBTyxDQUFDLENBQUMsU0FBUyxLQUFLLFNBQVMsSUFBSSxDQUFDLENBQUMsU0FBUyxLQUFLLFNBQVMsQ0FBQztBQUNoRSxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFTLGtCQUFrQixDQUFDLENBQVMsRUFBRSxPQUF3QjtJQUM3RCxNQUFNLEtBQUssR0FBRyxvQkFBUSxDQUFDLFFBQVEsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRSxDQUFDO1FBQ3hCLE9BQU8sQ0FBQyxDQUFDO0lBQ1gsQ0FBQztJQUNELE9BQU8sT0FBTyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztBQUNoQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IChjKSBIYXNoaUNvcnAsIEluY1xuLy8gU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IE1QTC0yLjBcbi8vIGNvcGllZCBmcm9tIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvY29uc3RydWN0cy9ibG9iL2UwMWU0N2Y3OGVmMWU5YjYwMGVmY2QyM2ZmNzcwNWFhOGQzODQwMTcvbGliL3ByaXZhdGUvcmVzb2x2ZS50c1xuaW1wb3J0IHsgSUNvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQge1xuICBEZWZhdWx0VG9rZW5SZXNvbHZlcixcbiAgSVBvc3RQcm9jZXNzb3IsXG4gIElSZXNvbHZhYmxlLFxuICBJUmVzb2x2ZUNvbnRleHQsXG4gIElUb2tlblJlc29sdmVyLFxuICBTdHJpbmdDb25jYXQsXG59IGZyb20gXCIuLi9yZXNvbHZhYmxlXCI7XG5pbXBvcnQgeyBUb2tlbml6ZWRTdHJpbmdGcmFnbWVudHMgfSBmcm9tIFwiLi4vc3RyaW5nLWZyYWdtZW50c1wiO1xuaW1wb3J0IHtcbiAgY29udGFpbnNTdHJpbmdMaXN0VG9rZW5FbGVtZW50LFxuICBUb2tlblN0cmluZyxcbiAgdW5yZXNvbHZlZCxcbiAgY29udGFpbnNOdW1iZXJMaXN0VG9rZW5FbGVtZW50LFxuICBjb250YWluc01hcFRva2VuLFxufSBmcm9tIFwiLi9lbmNvZGluZ1wiO1xuaW1wb3J0IHsgVG9rZW5NYXAgfSBmcm9tIFwiLi90b2tlbi1tYXBcIjtcbmltcG9ydCB7IFRva2VuIH0gZnJvbSBcIi4uL3Rva2VuXCI7XG5pbXBvcnQge1xuICBjYW5ub3RSZXNvbHZlRnVuY3Rpb24sXG4gIGNvbnN0cnVjdHNDYW5ub3RCZVJlc29sdmVkLFxuICBlbmNvZGVkTGlzdFRva2VuSW5TY2FsYXJTdHJpbmdDb250ZXh0LFxuICBlbmNvZGVkTWFwVG9rZW5JblNjYWxhck51bWJlckNvbnRleHQsXG4gIGVuY29kZWRNYXBUb2tlbkluU2NhbGFyU3RyaW5nQ29udGV4dCxcbiAgbWFwS2V5TXVzdFJlc29sdmVUb1N0cmluZyxcbiAgdW5hYmxlVG9SZXNvbHZlQ2lyY3VsYXJSZWZlcmVuY2UsXG59IGZyb20gXCIuLi8uLi9lcnJvcnNcIjtcblxuLy8gVGhpcyBmaWxlIHNob3VsZCBub3QgYmUgZXhwb3J0ZWQgdG8gY29uc3VtZXJzLCByZXNvbHZpbmcgc2hvdWxkIGhhcHBlbiB0aHJvdWdoIENvbnN0cnVjdC5yZXNvbHZlKClcblxuY29uc3QgdG9rZW5NYXAgPSBUb2tlbk1hcC5pbnN0YW5jZSgpO1xuXG4vKipcbiAqIE9wdGlvbnMgdG8gdGhlIHJlc29sdmUoKSBvcGVyYXRpb25cbiAqXG4gKiBOT1QgdGhlIHNhbWUgYXMgdGhlIFJlc29sdmVDb250ZXh0OyBSZXNvbHZlQ29udGV4dCBpcyBleHBvc2VkIHRvIFRva2VuXG4gKiBpbXBsZW1lbnRvcnMgYW5kIHJlc29sdXRpb24gaG9va3MsIHdoZXJlYXMgdGhpcyBzdHJ1Y3QgaXMganVzdCB0byBidW5kbGVcbiAqIGEgbnVtYmVyIG9mIHRoaW5ncyB0aGF0IHdvdWxkIG90aGVyd2lzZSBiZSBhcmd1bWVudHMgdG8gcmVzb2x2ZSgpIGluIGFcbiAqIHJlYWRhYmxlIHdheS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUmVzb2x2ZU9wdGlvbnMge1xuICBzY29wZTogSUNvbnN0cnVjdDtcbiAgcHJlcGFyaW5nOiBib29sZWFuO1xuICByZXNvbHZlcjogSVRva2VuUmVzb2x2ZXI7XG4gIHByZWZpeD86IHN0cmluZ1tdO1xuICBwcmV2aW91c0NvbnRleHQ/OiBJUmVzb2x2ZUNvbnRleHQ7XG59XG5cbi8qKlxuICogUmVzb2x2ZXMgYW4gb2JqZWN0IGJ5IGV2YWx1YXRpbmcgYWxsIHRva2VucyBhbmQgcmVtb3ZpbmcgYW55IHVuZGVmaW5lZCBvciBlbXB0eSBvYmplY3RzIG9yIGFycmF5cy5cbiAqIFZhbHVlcyBjYW4gb25seSBiZSBwcmltaXRpdmVzLCBhcnJheXMgb3IgdG9rZW5zLiBPdGhlciBvYmplY3RzIChpLmUuIHdpdGggbWV0aG9kcykgd2lsbCBiZSByZWplY3RlZC5cbiAqXG4gKiBAcGFyYW0gb2JqIFRoZSBvYmplY3QgdG8gcmVzb2x2ZS5cbiAqIEBwYXJhbSBwcmVmaXggUHJlZml4IGtleSBwYXRoIGNvbXBvbmVudHMgZm9yIGRpYWdub3N0aWNzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVzb2x2ZShvYmo6IGFueSwgb3B0aW9uczogSVJlc29sdmVPcHRpb25zKTogYW55IHtcbiAgY29uc3QgcHJlZml4ID0gb3B0aW9ucy5wcmVmaXggfHwgW107XG4gIGNvbnN0IHBhdGhOYW1lID0gXCIvXCIgKyBwcmVmaXguam9pbihcIi9cIik7XG5cbiAgLyoqXG4gICAqIE1ha2UgYSBuZXcgcmVzb2x1dGlvbiBjb250ZXh0XG4gICAqL1xuICBmdW5jdGlvbiBtYWtlQ29udGV4dChhcHBlbmRQYXRoPzogc3RyaW5nKTogW0lSZXNvbHZlQ29udGV4dCwgSVBvc3RQcm9jZXNzb3JdIHtcbiAgICBjb25zdCBuZXdQcmVmaXggPVxuICAgICAgYXBwZW5kUGF0aCAhPT0gdW5kZWZpbmVkID8gcHJlZml4LmNvbmNhdChbYXBwZW5kUGF0aF0pIDogb3B0aW9ucy5wcmVmaXg7XG5cbiAgICBsZXQgcG9zdFByb2Nlc3NvcjogSVBvc3RQcm9jZXNzb3IgfCB1bmRlZmluZWQ7XG5cbiAgICBjb25zdCBjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQgPSB7XG4gICAgICBwcmVwYXJpbmc6IG9wdGlvbnMucHJlcGFyaW5nLFxuICAgICAgc2NvcGU6IG9wdGlvbnMuc2NvcGUsXG4gICAgICBzdXBwcmVzc0JyYWNlczogb3B0aW9ucy5wcmV2aW91c0NvbnRleHQ/LnN1cHByZXNzQnJhY2VzLFxuICAgICAgaWdub3JlRXNjYXBlczogb3B0aW9ucy5wcmV2aW91c0NvbnRleHQ/Lmlnbm9yZUVzY2FwZXMsXG4gICAgICB3YXJuRXNjYXBlczogb3B0aW9ucy5wcmV2aW91c0NvbnRleHQ/Lndhcm5Fc2NhcGVzLFxuICAgICAgaXRlcmF0b3JDb250ZXh0OiBvcHRpb25zLnByZXZpb3VzQ29udGV4dD8uaXRlcmF0b3JDb250ZXh0LFxuICAgICAgcmVnaXN0ZXJQb3N0UHJvY2Vzc29yKHBwKSB7XG4gICAgICAgIHBvc3RQcm9jZXNzb3IgPSBwcDtcbiAgICAgIH0sXG4gICAgICByZXNvbHZlKHg6IGFueSkge1xuICAgICAgICByZXR1cm4gcmVzb2x2ZSh4LCB7XG4gICAgICAgICAgLi4ub3B0aW9ucyxcbiAgICAgICAgICBwcmVmaXg6IG5ld1ByZWZpeCxcbiAgICAgICAgICBwcmV2aW91c0NvbnRleHQ6IGNvbnRleHQsXG4gICAgICAgIH0pO1xuICAgICAgfSxcbiAgICB9O1xuXG4gICAgcmV0dXJuIFtcbiAgICAgIGNvbnRleHQsXG4gICAgICB7XG4gICAgICAgIHBvc3RQcm9jZXNzKHgpIHtcbiAgICAgICAgICByZXR1cm4gcG9zdFByb2Nlc3NvciA/IHBvc3RQcm9jZXNzb3IucG9zdFByb2Nlc3MoeCwgY29udGV4dCkgOiB4O1xuICAgICAgICB9LFxuICAgICAgfSxcbiAgICBdO1xuICB9XG5cbiAgLy8gcHJvdGVjdCBhZ2FpbnN0IGN5Y2xpYyByZWZlcmVuY2VzIGJ5IGxpbWl0aW5nIGRlcHRoLlxuICBpZiAocHJlZml4Lmxlbmd0aCA+IDIwMCkge1xuICAgIHRocm93IHVuYWJsZVRvUmVzb2x2ZUNpcmN1bGFyUmVmZXJlbmNlKHBhdGhOYW1lKTtcbiAgfVxuXG4gIC8vXG4gIC8vIHVuZGVmaW5lZFxuICAvL1xuXG4gIGlmICh0eXBlb2Ygb2JqID09PSBcInVuZGVmaW5lZFwiKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8vXG4gIC8vIG51bGxcbiAgLy9cblxuICBpZiAob2JqID09PSBudWxsKSB7XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cblxuICAvL1xuICAvLyBmdW5jdGlvbnMgLSBub3Qgc3VwcG9ydGVkIChvbmx5IHRva2VucyBhcmUgc3VwcG9ydGVkKVxuICAvL1xuXG4gIGlmICh0eXBlb2Ygb2JqID09PSBcImZ1bmN0aW9uXCIpIHtcbiAgICB0aHJvdyBjYW5ub3RSZXNvbHZlRnVuY3Rpb24ocGF0aE5hbWUsIG9iaik7XG4gIH1cblxuICAvL1xuICAvLyBzdHJpbmcgLSBwb3RlbnRpYWxseSByZXBsYWNlIGFsbCBzdHJpbmdpZmllZCBUb2tlbnNcbiAgLy9cbiAgaWYgKHR5cGVvZiBvYmogPT09IFwic3RyaW5nXCIpIHtcbiAgICAvLyBJZiB0aGlzIGlzIGEgXCJsaXN0IGVsZW1lbnRcIiBUb2tlbiwgaXQgc2hvdWxkIG5ldmVyIG9jY3VyIGJ5IGl0c2VsZiBpbiBzdHJpbmcgY29udGV4dFxuICAgIGlmIChUb2tlblN0cmluZy5mb3JMaXN0VG9rZW4ob2JqKS50ZXN0KCkpIHtcbiAgICAgIHRocm93IGVuY29kZWRMaXN0VG9rZW5JblNjYWxhclN0cmluZ0NvbnRleHQoKTtcbiAgICB9XG5cbiAgICBpZiAoXG4gICAgICBvYmogPT09IFRva2VuLlNUUklOR19NQVBfVE9LRU5fVkFMVUUgfHxcbiAgICAgIG9iaiA9PT0gVG9rZW4uQU5ZX01BUF9UT0tFTl9WQUxVRVxuICAgICkge1xuICAgICAgdGhyb3cgZW5jb2RlZE1hcFRva2VuSW5TY2FsYXJTdHJpbmdDb250ZXh0KCk7XG4gICAgfVxuXG4gICAgbGV0IHN0cjogc3RyaW5nID0gb2JqO1xuXG4gICAgY29uc3QgY29udGV4dCA9IG1ha2VDb250ZXh0KClbMF07XG4gICAgY29uc3QgdG9rZW5TdHIgPSBUb2tlblN0cmluZy5mb3JTdHJpbmcoXG4gICAgICBzdHIsXG4gICAgICAhY29udGV4dC5pZ25vcmVFc2NhcGVzLFxuICAgICAgY29udGV4dC53YXJuRXNjYXBlcyxcbiAgICApO1xuICAgIGlmICh0b2tlblN0ci50ZXN0KCkpIHtcbiAgICAgIGNvbnN0IGZyYWdtZW50cyA9IHRva2VuU3RyLnNwbGl0KHRva2VuTWFwLmxvb2t1cFRva2VuLmJpbmQodG9rZW5NYXApKTtcbiAgICAgIHN0ciA9IG9wdGlvbnMucmVzb2x2ZXIucmVzb2x2ZVN0cmluZyhmcmFnbWVudHMsIG1ha2VDb250ZXh0KClbMF0pO1xuICAgIH1cblxuICAgIC8vIHJlcGxhY2UgY29uY2F0ZW5hdGVkIHRva2VuIG51bWJlcnNcbiAgICBjb25zdCB0b2tlbk51bWJlclN0ciA9IFRva2VuU3RyaW5nLmZvck51bWJlcnMoc3RyKTtcbiAgICBpZiAodG9rZW5OdW1iZXJTdHIudGVzdCgpKSB7XG4gICAgICBjb25zdCBmcmFnbWVudHMgPSB0b2tlbk51bWJlclN0ci5zcGxpdCgoaWQpID0+IHtcbiAgICAgICAgcmV0dXJuIFRva2VuTWFwLmluc3RhbmNlKCkubG9va3VwTnVtYmVyVG9rZW4ocGFyc2VGbG9hdChpZCkpO1xuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IGNvbnRleHQgPSBtYWtlQ29udGV4dCgpWzBdO1xuICAgICAgc3RyID0gZnJhZ21lbnRzLm1hcFRva2Vucyhjb250ZXh0KS5qb2luKG5ldyBTdHJpbmdDb25jYXQoKSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHN0cjtcbiAgfVxuXG4gIC8vXG4gIC8vIG51bWJlciAtIHBvdGVudGlhbGx5IGRlY29kZSBUb2tlbml6ZWQgbnVtYmVyXG4gIC8vXG4gIGlmICh0eXBlb2Ygb2JqID09PSBcIm51bWJlclwiKSB7XG4gICAgaWYgKG9iaiA9PT0gVG9rZW4uTlVNQkVSX01BUF9UT0tFTl9WQUxVRSkge1xuICAgICAgdGhyb3cgZW5jb2RlZE1hcFRva2VuSW5TY2FsYXJOdW1iZXJDb250ZXh0KCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHJlc29sdmVOdW1iZXJUb2tlbihvYmosIG1ha2VDb250ZXh0KClbMF0pO1xuICB9XG5cbiAgLy9cbiAgLy8gcHJpbWl0aXZlcyAtIGFzLWlzXG4gIC8vXG5cbiAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWluc3RhbmNlb2Yvbm8taW5zdGFuY2VvZlxuICBpZiAodHlwZW9mIG9iaiAhPT0gXCJvYmplY3RcIiB8fCBvYmogaW5zdGFuY2VvZiBEYXRlKSB7XG4gICAgcmV0dXJuIG9iajtcbiAgfVxuXG4gIC8vXG4gIC8vIGFycmF5cyAtIHJlc29sdmUgYWxsIHZhbHVlcywgcmVtb3ZlIHVuZGVmaW5lZCBhbmQgcmVtb3ZlIGVtcHR5IGFycmF5c1xuICAvL1xuXG4gIGlmIChBcnJheS5pc0FycmF5KG9iaikpIHtcbiAgICBpZiAoY29udGFpbnNTdHJpbmdMaXN0VG9rZW5FbGVtZW50KG9iaikpIHtcbiAgICAgIHJldHVybiBvcHRpb25zLnJlc29sdmVyLnJlc29sdmVMaXN0KG9iaiwgbWFrZUNvbnRleHQoKVswXSk7XG4gICAgfVxuXG4gICAgaWYgKGNvbnRhaW5zTnVtYmVyTGlzdFRva2VuRWxlbWVudChvYmopKSB7XG4gICAgICByZXR1cm4gb3B0aW9ucy5yZXNvbHZlci5yZXNvbHZlTnVtYmVyTGlzdChvYmosIG1ha2VDb250ZXh0KClbMF0pO1xuICAgIH1cblxuICAgIGNvbnN0IGFyciA9IG9ialxuICAgICAgLm1hcCgoeCwgaSkgPT4gbWFrZUNvbnRleHQoYCR7aX1gKVswXS5yZXNvbHZlKHgpKVxuICAgICAgLmZpbHRlcigoeCkgPT4gdHlwZW9mIHggIT09IFwidW5kZWZpbmVkXCIpO1xuXG4gICAgcmV0dXJuIGFycjtcbiAgfVxuXG4gIC8vIGNoZWNrIGZvciB0b2tlbml6ZWQgbWFwXG4gIGlmIChjb250YWluc01hcFRva2VuKG9iaikpIHtcbiAgICByZXR1cm4gb3B0aW9ucy5yZXNvbHZlci5yZXNvbHZlTWFwKG9iaiwgbWFrZUNvbnRleHQoKVswXSk7XG4gIH1cblxuICAvL1xuICAvLyB0b2tlbnMgLSBpbnZva2UgJ3Jlc29sdmUnIGFuZCBjb250aW51ZSB0byByZXNvbHZlIHJlY3Vyc2l2ZWx5XG4gIC8vXG5cbiAgaWYgKHVucmVzb2x2ZWQob2JqKSkge1xuICAgIGNvbnN0IFtjb250ZXh0LCBwb3N0UHJvY2Vzc29yXSA9IG1ha2VDb250ZXh0KCk7XG4gICAgcmV0dXJuIG9wdGlvbnMucmVzb2x2ZXIucmVzb2x2ZVRva2VuKG9iaiwgY29udGV4dCwgcG9zdFByb2Nlc3Nvcik7XG4gIH1cblxuICAvL1xuICAvLyBvYmplY3RzIC0gZGVlcC1yZXNvbHZlIGFsbCB2YWx1ZXNcbiAgLy9cblxuICAvLyBNdXN0IG5vdCBiZSBhIENvbnN0cnVjdCBhdCB0aGlzIHBvaW50LCBvdGhlcndpc2UgeW91IHByb2JhYmx5IG1hZGUgYSB0eXBvXG4gIC8vIG1pc3Rha2Ugc29tZXdoZXJlIGFuZCByZXNvbHZlIHdpbGwgZ2V0IGludG8gYW4gaW5maW5pdGUgbG9vcCByZWN1cnNpbmcgaW50b1xuICAvLyBjaGlsZC5wYXJlbnQgPC0tLT4gcGFyZW50LmNoaWxkcmVuXG4gIGlmIChpc0NvbnN0cnVjdChvYmopKSB7XG4gICAgdGhyb3cgY29uc3RydWN0c0Nhbm5vdEJlUmVzb2x2ZWQocGF0aE5hbWUpO1xuICB9XG5cbiAgY29uc3QgcmVzdWx0OiBhbnkgPSB7fTtcbiAgZm9yIChjb25zdCBrZXkgb2YgT2JqZWN0LmtleXMob2JqKSkge1xuICAgIGNvbnN0IHJlc29sdmVkS2V5ID0gbWFrZUNvbnRleHQoKVswXS5yZXNvbHZlKGtleSk7XG4gICAgaWYgKHR5cGVvZiByZXNvbHZlZEtleSAhPT0gXCJzdHJpbmdcIikge1xuICAgICAgdGhyb3cgbWFwS2V5TXVzdFJlc29sdmVUb1N0cmluZyhwYXRoTmFtZSwga2V5LCByZXNvbHZlZEtleSk7XG4gICAgfVxuXG4gICAgY29uc3QgdmFsdWUgPSBtYWtlQ29udGV4dChrZXkpWzBdLnJlc29sdmUob2JqW2tleV0pO1xuXG4gICAgLy8gc2tpcCB1bmRlZmluZWRcbiAgICBpZiAodHlwZW9mIHZhbHVlID09PSBcInVuZGVmaW5lZFwiKSB7XG4gICAgICBjb250aW51ZTtcbiAgICB9XG5cbiAgICByZXN1bHRbcmVzb2x2ZWRLZXldID0gdmFsdWU7XG4gIH1cblxuICByZXR1cm4gcmVzdWx0O1xufVxuXG4vKipcbiAqIEZpbmQgYWxsIFRva2VucyB0aGF0IGFyZSB1c2VkIGluIHRoZSBnaXZlbiBzdHJ1Y3R1cmVcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGZpbmRUb2tlbnMoc2NvcGU6IElDb25zdHJ1Y3QsIGZuOiAoKSA9PiBhbnkpOiBJUmVzb2x2YWJsZVtdIHtcbiAgY29uc3QgcmVzb2x2ZXIgPSBuZXcgUmVtZW1iZXJpbmdUb2tlblJlc29sdmVyKG5ldyBTdHJpbmdDb25jYXQoKSk7XG5cbiAgcmVzb2x2ZShmbigpLCB7IHNjb3BlLCBwcmVmaXg6IFtdLCByZXNvbHZlciwgcHJlcGFyaW5nOiB0cnVlIH0pO1xuXG4gIHJldHVybiByZXNvbHZlci50b2tlbnM7XG59XG5cbi8qKlxuICogUmVtZW1iZXIgYWxsIFRva2VucyBlbmNvdW50ZXJlZCB3aGlsZSByZXNvbHZpbmdcbiAqL1xuZXhwb3J0IGNsYXNzIFJlbWVtYmVyaW5nVG9rZW5SZXNvbHZlciBleHRlbmRzIERlZmF1bHRUb2tlblJlc29sdmVyIHtcbiAgcHJpdmF0ZSByZWFkb25seSB0b2tlbnNTZWVuID0gbmV3IFNldDxJUmVzb2x2YWJsZT4oKTtcblxuICBwdWJsaWMgcmVzb2x2ZVRva2VuKFxuICAgIHQ6IElSZXNvbHZhYmxlLFxuICAgIGNvbnRleHQ6IElSZXNvbHZlQ29udGV4dCxcbiAgICBwb3N0UHJvY2Vzc29yOiBJUG9zdFByb2Nlc3NvcixcbiAgKSB7XG4gICAgdGhpcy50b2tlbnNTZWVuLmFkZCh0KTtcbiAgICByZXR1cm4gc3VwZXIucmVzb2x2ZVRva2VuKHQsIGNvbnRleHQsIHBvc3RQcm9jZXNzb3IpO1xuICB9XG5cbiAgcHVibGljIHJlc29sdmVTdHJpbmcoczogVG9rZW5pemVkU3RyaW5nRnJhZ21lbnRzLCBjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpIHtcbiAgICBjb25zdCByZXQgPSBzdXBlci5yZXNvbHZlU3RyaW5nKHMsIGNvbnRleHQpO1xuICAgIHJldHVybiByZXQ7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHRva2VucygpOiBJUmVzb2x2YWJsZVtdIHtcbiAgICByZXR1cm4gQXJyYXkuZnJvbSh0aGlzLnRva2Vuc1NlZW4pO1xuICB9XG59XG5cbi8qKlxuICogRGV0ZXJtaW5lIHdoZXRoZXIgYW4gb2JqZWN0IGlzIGEgQ29uc3RydWN0XG4gKlxuICogTm90IGluICdjb25zdHJ1Y3QudHMnIGJlY2F1c2UgdGhhdCB3b3VsZCBsZWFkIHRvIGEgZGVwZW5kZW5jeSBjeWNsZSB2aWEgJ3VuaXF1ZWlkLnRzJyxcbiAqIGFuZCB0aGlzIGlzIGEgYmVzdC1lZmZvcnQgcHJvdGVjdGlvbiBhZ2FpbnN0IGEgY29tbW9uIHByb2dyYW1taW5nIG1pc3Rha2UgYW55d2F5LlxuICovXG5mdW5jdGlvbiBpc0NvbnN0cnVjdCh4OiBhbnkpOiBib29sZWFuIHtcbiAgcmV0dXJuIHguX2NoaWxkcmVuICE9PSB1bmRlZmluZWQgJiYgeC5fbWV0YWRhdGEgIT09IHVuZGVmaW5lZDtcbn1cblxuLyoqXG4gKiBSZXNvbHZlcyBhIG51bWJlciB0b2tlblxuICovXG5mdW5jdGlvbiByZXNvbHZlTnVtYmVyVG9rZW4oeDogbnVtYmVyLCBjb250ZXh0OiBJUmVzb2x2ZUNvbnRleHQpOiBhbnkge1xuICBjb25zdCB0b2tlbiA9IFRva2VuTWFwLmluc3RhbmNlKCkubG9va3VwTnVtYmVyVG9rZW4oeCk7XG4gIGlmICh0b2tlbiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgcmV0dXJuIHg7XG4gIH1cbiAgcmV0dXJuIGNvbnRleHQucmVzb2x2ZSh0b2tlbik7XG59XG4iXX0=