"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NullConcat = exports.TokenString = exports.VALID_KEY_CHARS = exports.END_TOKEN_MARKER = exports.BEGIN_MAP_TOKEN_MARKER = exports.BEGIN_LIST_TOKEN_MARKER = exports.BEGIN_STRING_TOKEN_MARKER = void 0;
exports.regexQuote = regexQuote;
exports.containsStringListTokenElement = containsStringListTokenElement;
exports.containsNumberListTokenElement = containsNumberListTokenElement;
exports.containsMapToken = containsMapToken;
exports.isComplexElement = isComplexElement;
exports.containsComplexElement = containsComplexElement;
exports.unresolved = unresolved;
exports.createTokenDouble = createTokenDouble;
exports.extractTokenDouble = extractTokenDouble;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// copied from https://github.com/aws/constructs/blob/e01e47f78ef1e9b600efcd23ff7705aa8d384017/lib/private/encoding.ts
const errors_1 = require("../../errors");
const string_fragments_1 = require("../string-fragments");
const token_1 = require("../token");
// Details for encoding and decoding Tokens into native types; should not be exported
exports.BEGIN_STRING_TOKEN_MARKER = "${TfToken[";
exports.BEGIN_LIST_TOKEN_MARKER = "#{TfToken[";
exports.BEGIN_MAP_TOKEN_MARKER = "&{TfToken[";
exports.END_TOKEN_MARKER = "]}";
exports.VALID_KEY_CHARS = "a-zA-Z0-9:._-";
const QUOTED_BEGIN_STRING_TOKEN_MARKER = regexQuote(exports.BEGIN_STRING_TOKEN_MARKER);
const QUOTED_BEGIN_LIST_TOKEN_MARKER = regexQuote(exports.BEGIN_LIST_TOKEN_MARKER);
const QUOTED_BEGIN_MAP_TOKEN_MARKER = regexQuote(exports.BEGIN_MAP_TOKEN_MARKER);
const QUOTED_END_TOKEN_MARKER = regexQuote(exports.END_TOKEN_MARKER);
const STRING_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_STRING_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const LIST_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_LIST_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const MAP_TOKEN_REGEX = new RegExp(`${QUOTED_BEGIN_MAP_TOKEN_MARKER}([${exports.VALID_KEY_CHARS}]+)${QUOTED_END_TOKEN_MARKER}`, "g");
const NUMBER_TOKEN_REGEX = new RegExp("[-|\\d|.\\+]+(e[-|\\d|.|e|E|\\+]+)", "g");
const ESCAPE_TOKEN_BEGIN_REGEX = /\$\{(?!TfToken\[)/g;
const ESCAPE_TOKEN_END_REGEX = /\}/g;
/**
 * A string with markers in it that can be resolved to external values
 */
class TokenString {
    /**
     * Returns a `TokenString` for this string.
     *
     * @param s The string to tokenize
     * @param includeEscapeSequences Whether to include escape sequences in the tokenization
     * @param warnIfEscapeSequences Whether to warn if escape sequences are found
     */
    static forString(s, includeEscapeSequences = false, warnIfEscapeSequences = false) {
        return new TokenString(s, STRING_TOKEN_REGEX, 1, includeEscapeSequences, warnIfEscapeSequences);
    }
    /**
     * Returns a `TokenString` for this string (must be the first string element of the list)
     */
    static forListToken(s) {
        return new TokenString(s, LIST_TOKEN_REGEX);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded numbers
     */
    static forNumbers(s) {
        return new TokenString(s, NUMBER_TOKEN_REGEX, 0);
    }
    /**
     * Returns a `TokenString` for this string that handles encoded maps
     */
    static forMapToken(s) {
        return new TokenString(s, MAP_TOKEN_REGEX);
    }
    constructor(str, re, regexMatchIndex = 1, includeEscapeSequences = false, warnIfEscapeSequences = false) {
        this.str = str;
        this.re = re;
        this.regexMatchIndex = regexMatchIndex;
        this.includeEscapeSequences = includeEscapeSequences;
        this.warnIfEscapeSequences = warnIfEscapeSequences;
    }
    testForEscapeTokens(startIndex, maxIndex) {
        ESCAPE_TOKEN_BEGIN_REGEX.lastIndex = startIndex;
        let startMatch = ESCAPE_TOKEN_BEGIN_REGEX.exec(this.str);
        if (startMatch && startMatch.index >= maxIndex) {
            startMatch = null;
        }
        return !!startMatch;
    }
    nextEscapeToken(fragments, startIndex, escapeDepth, maxIndex) {
        ESCAPE_TOKEN_BEGIN_REGEX.lastIndex = startIndex;
        ESCAPE_TOKEN_END_REGEX.lastIndex = startIndex;
        let startMatch = ESCAPE_TOKEN_BEGIN_REGEX.exec(this.str);
        let endMatch = ESCAPE_TOKEN_END_REGEX.exec(this.str);
        if (startMatch && startMatch.index >= maxIndex) {
            startMatch = null;
        }
        if (endMatch && endMatch.index >= maxIndex) {
            endMatch = null;
        }
        if (!startMatch && !endMatch) {
            return {
                index: -1,
                escapeDepth: escapeDepth,
            };
        }
        if (startMatch && endMatch) {
            if (startMatch.index > startIndex && startMatch.index > startIndex) {
                const lede = this.str.substring(startIndex, Math.min(startMatch.index, endMatch.index));
                fragments.addLiteral(lede);
            }
            if (startMatch.index < endMatch.index) {
                fragments.addEscape("open");
                return {
                    index: ESCAPE_TOKEN_BEGIN_REGEX.lastIndex,
                    escapeDepth: escapeDepth + 1,
                };
            }
            fragments.addEscape("close");
            return {
                index: ESCAPE_TOKEN_END_REGEX.lastIndex,
                escapeDepth: escapeDepth - 1,
            };
        }
        if (startMatch) {
            if (startMatch.index > startIndex) {
                const lede = this.str.substring(startIndex, startMatch.index);
                fragments.addLiteral(lede);
            }
            fragments.addEscape("open");
            return {
                index: ESCAPE_TOKEN_BEGIN_REGEX.lastIndex,
                escapeDepth: escapeDepth + 1,
            };
        }
        if (endMatch) {
            if (endMatch.index > startIndex) {
                const lede = this.str.substring(startIndex, endMatch.index);
                fragments.addLiteral(lede);
            }
            fragments.addEscape("close");
            return {
                index: ESCAPE_TOKEN_END_REGEX.lastIndex,
                escapeDepth: escapeDepth - 1,
            };
        }
        return {
            index: -1,
            escapeDepth: escapeDepth,
        };
    }
    tokenizeNext(lookup, fragments, startIndex, escapeDepth) {
        this.re.lastIndex = startIndex;
        if (startIndex >= this.str.length) {
            return {
                index: -1,
                escapeDepth,
            };
        }
        const match = this.re.exec(this.str);
        if (!match) {
            if (this.includeEscapeSequences) {
                const next = this.nextEscapeToken(fragments, startIndex, escapeDepth, this.str.length);
                if (next.index === -1) {
                    fragments.addLiteral(this.str.substring(startIndex));
                    return {
                        index: -1,
                        escapeDepth,
                    };
                }
                else {
                    return next;
                }
            }
            fragments.addLiteral(this.str.substring(startIndex));
            return {
                index: -1,
                escapeDepth: escapeDepth,
            };
        }
        if (match.index > startIndex) {
            if (this.includeEscapeSequences) {
                const next = this.nextEscapeToken(fragments, startIndex, escapeDepth, match.index);
                if (next.index === -1) {
                    fragments.addLiteral(this.str.substring(startIndex, match.index));
                    return {
                        index: match.index,
                        escapeDepth: escapeDepth,
                    };
                }
                else {
                    return next;
                }
            }
            const lede = this.str.substring(startIndex, match.index);
            fragments.addLiteral(lede);
            return {
                index: match.index,
                escapeDepth,
            };
        }
        const token = lookup(match[this.regexMatchIndex]);
        if (token) {
            fragments.addToken(token);
        }
        else {
            fragments.addLiteral(this.str.substring(match.index, this.re.lastIndex));
        }
        const nextIndex = this.re.lastIndex;
        return {
            index: nextIndex,
            escapeDepth,
        };
    }
    /**
     * Split string on markers, substituting markers with Tokens
     */
    split(lookup) {
        const ret = new string_fragments_1.TokenizedStringFragments();
        let index = 0;
        let escapeDepth = 0;
        if (this.warnIfEscapeSequences &&
            this.testForEscapeTokens(0, this.str.length)) {
            // Only print warning and act as if escape sequences are ignored
            console.warn("You're using escape sequences (${...}) with CDKTF Built-in functions. This is not supported yet, and the output may be incorrect.");
            console.warn(this.str);
        }
        while (index >= 0) {
            const iter = this.tokenizeNext(lookup, ret, index, escapeDepth);
            index = iter.index;
            escapeDepth = iter.escapeDepth;
        }
        return ret;
    }
    /**
     * Indicates if this string includes tokens.
     */
    test() {
        this.re.lastIndex = 0; // Reset
        return this.re.test(this.str);
    }
}
exports.TokenString = TokenString;
/**
 * Quote a string for use in a regex
 */
function regexQuote(s) {
    return s.replace(/[.?*+^$[\]\\(){}|-]/g, "\\$&");
}
/**
 * Concatenator that disregards the input
 *
 * Can be used when traversing the tokens is important, but the
 * result isn't.
 */
class NullConcat {
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    join(_left, _right) {
        return undefined;
    }
}
exports.NullConcat = NullConcat;
/**
 * Returns true if argument contains a string list token
 */
function containsStringListTokenElement(xs) {
    return xs.some((x) => typeof x === "string" && TokenString.forListToken(x).test());
}
/**
 * Returns true if argument contains a number list token
 */
function containsNumberListTokenElement(xs) {
    return xs.some((x) => typeof x === "number" && extractTokenDouble(x, true) !== undefined);
}
/**
 * Returns true if argument contains a string map token
 */
function containsMapToken(xs) {
    return Object.keys(xs).some((x) => typeof x === "string" && TokenString.forMapToken(x).test());
}
/**
 * Returns true if argument is a complex element
 */
function isComplexElement(xs) {
    return (typeof xs === "object" &&
        xs !== null &&
        "withinArray" in xs &&
        "interpolationAsList" in xs);
}
/**
 * Returns true if list contains a complex element
 */
function containsComplexElement(xs) {
    return xs.length > 0 && isComplexElement(xs[0]);
}
/**
 * Returns true if obj is a token (i.e. has the resolve() method or is a string
 * that includes token markers), or it's a listifictaion of a Token string.
 *
 * @param obj The object to test.
 */
function unresolved(obj) {
    if (typeof obj === "string") {
        return TokenString.forString(obj).test();
    }
    else if (typeof obj === "number") {
        return extractTokenDouble(obj, false) !== undefined;
    }
    else if (Array.isArray(obj) && obj.length === 1) {
        return ((typeof obj[0] === "string" && TokenString.forListToken(obj[0]).test()) ||
            (typeof obj[0] === "number" &&
                extractTokenDouble(obj[0], true) !== undefined));
    }
    else if (obj != null && containsMapToken(obj)) {
        return true;
    }
    else {
        return token_1.Tokenization.isResolvable(obj);
    }
}
/**
 * Bit pattern in the top 16 bits of a double to indicate a Token
 *
 * An IEEE double in LE memory order looks like this (grouped
 * into octets, then grouped into 32-bit words):
 *
 * mmmmmmmm.mmmmmmmm.mmmmmmmm.mmmmmmmm | mmmmmmmm.mmmmmmmm.EEEEmmmm.sEEEEEEE
 *
 * - m: mantissa (52 bits)
 * - E: exponent (11 bits)
 * - s: sign (1 bit)
 *
 * We put the following marker into the top 16 bits (exponent and sign), and
 * use the mantissa part to encode the token index. To save some bit twiddling
 * we use all top 16 bits for the tag. That loses us 4 mantissa bits to store
 * information in but we still have 48, which is going to be plenty for any
 * number of tokens to be created during the lifetime of any CDK application.
 *
 * Can't have all bits set because that makes a NaN, so unset the least
 * significant exponent bit.
 *
 * Currently not supporting BE architectures.
 */
// tslint:disable-next-line:no-bitwise
const DOUBLE_TOKEN_MARKER_BITS = 0xfdff << 16;
// tslint:disable-next-line:no-bitwise
const DOUBLE_LIST_TOKEN_MARKER_BITS = 0xfbff << 16;
/**
 * Highest encodable number
 */
const MAX_ENCODABLE_INTEGER = Math.pow(2, 48) - 1;
/**
 * Get 2^32 as a number, so we can do multiplication and div instead of bit shifting
 *
 * Necessary because in JavaScript, bit operations implicitly convert
 * to int32 and we need them to work on "int64"s.
 *
 * So instead of x >> 32, we do Math.floor(x / 2^32), and vice versa.
 */
const BITS32 = Math.pow(2, 32);
/**
 * Return a special Double value that encodes the given nonnegative integer
 *
 * We use this to encode Token ordinals.
 */
function createTokenDouble(x, list) {
    if (Math.floor(x) !== x || x < 0) {
        throw (0, errors_1.canOnlyEncodePositiveIntegers)();
    }
    if (x > MAX_ENCODABLE_INTEGER) {
        throw (0, errors_1.indexTooLargeToEncode)(x);
    }
    const buf = new ArrayBuffer(8);
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    ints[0] = x & 0x0000ffffffff; // Bottom 32 bits of number
    // This needs an "x >> 32" but that will make it a 32-bit number instead
    // of a 64-bit number.
    ints[1] =
        (shr32(x) & 0xffff) |
            (list ? DOUBLE_LIST_TOKEN_MARKER_BITS : DOUBLE_TOKEN_MARKER_BITS); // Top 16 bits of number and the mask
    // tslint:enable:no-bitwise
    return new Float64Array(buf)[0];
}
/**
 * Shift a 64-bit int right 32 bits
 */
function shr32(x) {
    return Math.floor(x / BITS32);
}
/**
 * Shift a 64-bit left 32 bits
 */
function shl32(x) {
    return x * BITS32;
}
/**
 * Extract the encoded integer out of the special Double value
 *
 * Returns undefined if the float is a not an encoded token.
 */
function extractTokenDouble(encoded, list) {
    const buf = new ArrayBuffer(8);
    new Float64Array(buf)[0] = encoded;
    const ints = new Uint32Array(buf);
    // tslint:disable:no-bitwise
    if ((ints[1] & 0xffff0000) !==
        (list ? DOUBLE_LIST_TOKEN_MARKER_BITS : DOUBLE_TOKEN_MARKER_BITS)) {
        return undefined;
    }
    // Must use + instead of | here (bitwise operations
    // will force 32-bits integer arithmetic, + will not).
    return ints[0] + shl32(ints[1] & 0xffff);
    // tslint:enable:no-bitwise
}
//# sourceMappingURL=data:application/json;base64,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