"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FOR_EXPRESSION_VALUE = exports.FOR_EXPRESSION_KEY = exports.OperatorExpression = void 0;
exports.rawString = rawString;
exports.ref = ref;
exports.insideTfExpression = insideTfExpression;
exports.propertyAccess = propertyAccess;
exports.conditional = conditional;
exports.call = call;
exports.forExpression = forExpression;
exports.dependable = dependable;
const intrinsic_1 = require("./tokens/private/intrinsic");
const token_1 = require("./tokens/token");
const app_1 = require("./app");
const terraform_stack_1 = require("./terraform-stack");
const constructs_1 = require("constructs");
const TERRAFORM_IDENTIFIER_REGEX = /^[_a-zA-Z][_a-zA-Z0-9]*$/;
// eslint-disable-next-line jsdoc/require-jsdoc
class TFExpression extends intrinsic_1.Intrinsic {
    resolveExpressionPart(context, arg) {
        const resolvedArg = context.resolve(arg);
        if (token_1.Tokenization.isResolvable(arg)) {
            return resolvedArg;
        }
        if (Array.isArray(resolvedArg)) {
            return `[${resolvedArg
                .map((_, index) => this.resolveArg(context, arg[index]))
                .join(", ")}]`;
        }
        if (typeof resolvedArg === "object" && resolvedArg !== null) {
            return `{${Object.keys(resolvedArg)
                .map((key) => `"${key}" = ${this.resolveArg(context, arg[key])}`)
                .join(", ")}}`;
        }
        return resolvedArg;
    }
    resolveArg(context, arg) {
        const resolvedArg = context.resolve(arg);
        if (typeof arg === "string") {
            const str = this.resolveString(arg, resolvedArg);
            // When Token.asString() is used on an object, str will be the object and needs to be resolved differently
            // This happens for example in MapTerraformIterator#_getForEachExpression and can cause issues like #3540
            if (typeof str !== "string") {
                return this.resolveExpressionPart(context, str);
            }
            return str;
        }
        return this.resolveExpressionPart(context, arg);
    }
    /**
     * Escape string removes characters from the string that are not allowed in Terraform or JSON
     * It must only be used on non-token values
     */
    escapeString(str) {
        return str
            .replace(/\n/g, "\\n") // escape newlines
            .replace(/\${/g, "$$${"); // escape ${ to $${
    }
    resolveString(str, resolvedArg) {
        const tokenList = token_1.Tokenization.reverseString(str);
        const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
        // String literal
        if (numberOfTokens === 0) {
            return resolvedArg !== `"` &&
                resolvedArg.startsWith('"') &&
                resolvedArg.endsWith('"')
                ? this.escapeString(resolvedArg)
                : `"${this.escapeString(resolvedArg)}"`;
        }
        // Only a token reference
        if (tokenList.literals.length === 0 &&
            tokenList.escapes.length === 0 &&
            numberOfTokens === 1) {
            return resolvedArg;
        }
        // String literal + token reference combination
        const joinResult = tokenList.join({
            join: (left, right) => {
                const leftTokens = token_1.Tokenization.reverse(left);
                const leftTokenList = token_1.Tokenization.reverseString(left);
                const rightTokens = token_1.Tokenization.reverse(right);
                const leftTokenCount = leftTokenList.intrinsic.length + leftTokenList.tokens.length;
                // if left is mixed, needs to be left alone (because it's a result of a previous join iteration)
                let leftValue = left;
                // if left is a string literal, then we need to escape it
                if (leftTokenList.literals.length === 1 && leftTokenCount === 0) {
                    leftValue = this.escapeString(left);
                }
                // if left is only a token, needs to be wrapped as terraform expression
                if (leftTokenList.literals.length === 0 && leftTokenCount === 1) {
                    leftValue = `\${${leftTokens[0]}}`;
                }
                const rightValue = rightTokens.length === 0
                    ? this.escapeString(right)
                    : `\${${rightTokens[0]}}`;
                return `${leftValue}${rightValue}`;
            },
        });
        return joinResult !== `"` &&
            joinResult.startsWith('"') &&
            joinResult.endsWith('"')
            ? joinResult
            : `"${joinResult}"`;
    }
}
// A string that represents an input value to be escaped
// eslint-disable-next-line jsdoc/require-jsdoc
class RawString extends TFExpression {
    constructor(str) {
        super(str);
        this.str = str;
    }
    resolve(context) {
        const qts = context.suppressBraces ? `"` : ``;
        return `${qts}${this.escapeString(this.str).replace(/\"/g, '\\"')}${qts}`; // eslint-disable-line no-useless-escape
    }
    toString() {
        return this.str;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function rawString(str) {
    return new RawString(str);
}
// eslint-disable-next-line jsdoc/require-jsdoc
class Reference extends TFExpression {
    constructor(identifier, originStack) {
        super(identifier);
        this.identifier = identifier;
        this.originStack = originStack;
        /**
         * A single reference could be used in multiple stacks,
         * e.g. if we expose the ref directly or as token on the stack.
         * We need to store the identifier for each stack,
         * so that the resolved identifier string matches the stack it's resolved in.
         */
        this.crossStackIdentifier = {};
    }
    resolve(context) {
        // We check for cross stack references on preparation, setting a new identifier
        const resolutionStack = terraform_stack_1.TerraformStack.of(context.scope);
        const stackName = resolutionStack.toString();
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        if (context.preparing) {
            // Cross stack reference
            if (this.originStack && this.originStack !== resolutionStack) {
                const app = app_1.App.of(this.originStack);
                const csr = app.crossStackReference(this.originStack, resolutionStack, this.identifier);
                this.crossStackIdentifier[stackName] = csr;
            }
        }
        // If this is a cross stack reference we will resolve to a reference within this stack.
        if (this.crossStackIdentifier[stackName]) {
            return suppressBraces
                ? this.crossStackIdentifier[stackName]
                : `\${${this.crossStackIdentifier[stackName]}}`;
        }
        return suppressBraces ? this.identifier : `\${${this.identifier}}`;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function ref(identifier, stack) {
    return new Reference(identifier, stack);
}
/**
 * marks the argument as being used in a terraform expression
 * @deprecated
 */
function insideTfExpression(arg) {
    return arg;
}
// eslint-disable-next-line jsdoc/require-jsdoc
class PropertyAccess extends TFExpression {
    constructor(target, args) {
        super({ target, args });
        this.target = target;
        this.args = args;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const serializedArgs = this.args
            .map((arg) => {
            if (arg === `*`) {
                return "[*]";
            }
            const a = this.resolveArg(context, arg);
            const isPlainString = typeof arg === "string" && TERRAFORM_IDENTIFIER_REGEX.test(arg);
            const isPlainNumber = typeof arg === "number" && !token_1.Token.isUnresolved(arg);
            if (isPlainString || isPlainNumber) {
                return `.${arg}`;
            }
            return `[${a}]`;
        }) // property access
            .join("");
        const targetExpr = constructs_1.Construct.isConstruct(this.target) && "fqn" in this.target
            ? this.target.fqn
            : this.resolveArg(context, this.target);
        const expr = `${targetExpr}${serializedArgs}`;
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function propertyAccess(target, args) {
    return new PropertyAccess(target, args);
}
// eslint-disable-next-line jsdoc/require-jsdoc
class ConditionalExpression extends TFExpression {
    constructor(condition, trueValue, falseValue) {
        super({ condition, trueValue, falseValue });
        this.condition = condition;
        this.trueValue = trueValue;
        this.falseValue = falseValue;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const condition = this.resolveArg(context, this.condition);
        const trueValue = this.resolveArg(context, this.trueValue);
        const falseValue = this.resolveArg(context, this.falseValue);
        const expr = `${condition} ? ${trueValue} : ${falseValue}`;
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
/**
 * creates an expression like `true ? 1 : 0`
 */
function conditional(condition, trueValue, falseValue) {
    return new ConditionalExpression(condition, trueValue, falseValue);
}
/**
 * Base class for Operator Expressions
 * @internal
 */
class OperatorExpression extends TFExpression {
    constructor(operator, left, right) {
        super({ operator, leftValue: left, rightValue: right });
        this.operator = operator;
        this.left = left;
        this.right = right;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const left = this.resolveArg(context, this.left);
        const right = this.right ? this.resolveArg(context, this.right) : undefined;
        let expr = "";
        switch (this.operator) {
            case "!": {
                expr = `!${left}`; // no braces needed as ! has highest precedence
                break;
            }
            case "-": {
                if (right) {
                    // subtraction
                    expr = `(${left} - ${right})`;
                }
                else {
                    // negation
                    expr = `-${left}`; // no braces needed as - has highest precedence
                }
                break;
            }
            default: {
                expr = `(${left} ${this.operator} ${right})`;
            }
        }
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
exports.OperatorExpression = OperatorExpression;
// eslint-disable-next-line jsdoc/require-jsdoc
class FunctionCall extends TFExpression {
    constructor(name, args) {
        super({ name, args });
        this.name = name;
        this.args = args;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        const originalIgnoreEscapes = context.ignoreEscapes;
        const originalWarnEscapes = context.warnEscapes;
        context.suppressBraces = true;
        context.ignoreEscapes = true;
        context.warnEscapes = true;
        const serializedArgs = this.args
            .map((arg) => this.resolveArg(context, arg))
            .join(", ");
        const expr = `${this.name}(${serializedArgs})`;
        context.ignoreEscapes = originalIgnoreEscapes;
        context.warnEscapes = originalWarnEscapes;
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function call(name, args) {
    return new FunctionCall(name, args);
}
exports.FOR_EXPRESSION_KEY = ref("key");
exports.FOR_EXPRESSION_VALUE = ref("val");
/**
 * https://developer.hashicorp.com/terraform/language/expressions/for
 */
class ForExpression extends TFExpression {
    constructor(input, valueExpression, keyExpression) {
        super({ input, valueExpression, keyExpression });
        this.input = input;
        this.valueExpression = valueExpression;
        this.keyExpression = keyExpression;
    }
    resolve(context) {
        const suppressBraces = context.suppressBraces;
        context.suppressBraces = true;
        const key = this.resolveArg(context, exports.FOR_EXPRESSION_KEY);
        const value = this.resolveArg(context, exports.FOR_EXPRESSION_VALUE);
        const input = this.resolveArg(context, this.input);
        const valueExpr = this.resolveExpressionPart(context, this.valueExpression);
        let expr;
        if (this.keyExpression) {
            const keyExpr = this.resolveExpressionPart(context, this.keyExpression);
            expr = `{ for ${key}, ${value} in ${input}: ${keyExpr} => ${valueExpr} }`;
        }
        else {
            expr = `[ for ${key}, ${value} in ${input}: ${valueExpr}]`;
        }
        return suppressBraces ? expr : `\${${expr}}`;
    }
}
/**
 * Creates a for expression, used for advanced internal use cases (e.g. in the @cdktf/aws-adapter)
 * It is currently not possible to use this via JSII in other languages. Usually you might want to
 * use TerraformIterator instead.
 *
 * @param input the input for the expression e.g. a list or an object
 * @param valueExpression will express the value e.g. [ for i in &lt;input&gt; : &lt;valueExpression&gt; ]
 * @param keyExpression if set will return an object e.g. { for k,v in &lt;input&gt; : &lt;keyExpression&gt; => &lt;valueExpression&gt;}
 * @returns a for expression
 *
 * The variables for key (for maps) / index (for lists) and value that are valid in the key- and valueExpression
 * can be referenced by using these constants: FOR_EXPRESSION_KEY and FOR_EXPRESSION_VALUE.
 *
 * Currently nesting for expressions is not supported due to this simplification.
 *
 * Currently does not support filtering elements (if clause) or grouping results
 */
function forExpression(input, valueExpression, keyExpression) {
    return new ForExpression(input, valueExpression, keyExpression);
}
// eslint-disable-next-line jsdoc/require-jsdoc
class Dependable extends TFExpression {
    constructor(dependable) {
        super(dependable);
        this.dependable = dependable;
    }
    resolve(context) {
        context.suppressBraces = true;
        return this.dependable.fqn;
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
function dependable(dependable) {
    return token_1.Token.asString(new Dependable(dependable));
}
//# sourceMappingURL=data:application/json;base64,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