"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionReturn = void 0;
exports.returnMatcherToJest = returnMatcherToJest;
exports.asymetricDeepEqualIgnoringObjectCasing = asymetricDeepEqualIgnoringObjectCasing;
exports.getToHaveDataSourceWithProperties = getToHaveDataSourceWithProperties;
exports.getToHaveResourceWithProperties = getToHaveResourceWithProperties;
exports.getToHaveProviderWithProperties = getToHaveProviderWithProperties;
exports.toBeValidTerraform = toBeValidTerraform;
exports.toPlanSuccessfully = toPlanSuccessfully;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const fs = require("fs");
const path = require("path");
const child_process_1 = require("child_process");
const util_1 = require("../util");
const errors_1 = require("../errors");
/**
 * Class representing the contents of a return by an assertion
 */
class AssertionReturn {
    /**
     * Create an AssertionReturn
     * @param message - String message containing information about the result of the assertion
     * @param pass - Boolean pass denoting the success of the assertion
     */
    constructor(message, pass) {
        this.message = message;
        this.pass = pass;
    }
}
exports.AssertionReturn = AssertionReturn;
_a = JSII_RTTI_SYMBOL_1;
AssertionReturn[_a] = { fqn: "cdktf.testingMatchers.AssertionReturn", version: "0.21.0" };
/**
 * Reformats the contents of the base testing matcher return type AssertionReturn into type useable by jest
 * @param toReturn
 * @returns {MatcherReturnJest}
 */
function returnMatcherToJest(toReturn) {
    return {
        message: () => toReturn.message,
        pass: toReturn.pass,
    };
}
/**
 * Compares expected and received. All expected properties are matched and considered equal even if
 * there are more properties in the received object than in the expected object in which case it will still return true.
 * @param expected
 * @param received
 * @returns {boolean}
 */
function asymetricDeepEqualIgnoringObjectCasing(expected, received) {
    switch (typeof expected) {
        case "object":
            if (Array.isArray(expected)) {
                return (Array.isArray(received) &&
                    expected.length === received.length &&
                    expected.every((item, index) => asymetricDeepEqualIgnoringObjectCasing(item, received[index])));
            }
            if (expected === null && received === null) {
                return true;
            }
            if (expected === undefined && received === undefined) {
                return true;
            }
            if (expected === null || received === null) {
                return false;
            }
            // recursively compare objects and allow snake case as well as camel case
            return Object.keys(expected).every((key) => {
                if (received[key] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[key]);
                }
                if (received[(0, util_1.snakeCase)(key)] !== undefined) {
                    return asymetricDeepEqualIgnoringObjectCasing(expected[key], received[(0, util_1.snakeCase)(key)]);
                }
                return false;
            });
        default:
            return expected === received;
    }
}
const defaultPassEvaluation = (items, assertedProperties) => {
    return Object.values(items).some((item) => asymetricDeepEqualIgnoringObjectCasing(assertedProperties, item));
};
// eslint-disable-next-line jsdoc/require-jsdoc
function isAsymmetric(obj) {
    return !!obj && typeof obj === "object" && "asymmetricMatch" in obj;
}
// You can use expect.Anything(), expect.ObjectContaining, etc in jest, this makes it nicer to read
// when we print error messages
// eslint-disable-next-line jsdoc/require-jsdoc
function jestAsymetricMatcherStringifyReplacer(_key, value) {
    return isAsymmetric(value) ? `expect.${value.toString()}` : value;
}
// eslint-disable-next-line jsdoc/require-jsdoc
function getAssertElementWithProperties(functionName, 
// We have the evaluation function configurable so we can make use of the specific testing frameworks capabilities
// This makes the resulting tests more native to the testing framework
customPassEvaluation) {
    const passEvaluation = customPassEvaluation || defaultPassEvaluation;
    return function getAssertElementWithProperties(type, received, itemType, properties = {}) {
        let stack;
        // Rececived could either be a JSON string or a path to a file
        const stackContent = fs.existsSync(received)
            ? fs.readFileSync(received, "utf8")
            : received;
        try {
            stack = JSON.parse(stackContent);
        }
        catch (e) {
            throw (0, errors_1.invalidStack)(functionName, stackContent);
        }
        const items = Object.values(Object.entries(stack[type] || {}) // for all data/resource entries
            .find(
        // find the object with a matching name
        ([type, _values]) => type === itemType.tfResourceType)?.[1] || {}) || []; // get a list of all configs of that type
        const pass = passEvaluation(items, properties);
        if (pass) {
            return new AssertionReturn(`Expected no ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesized stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`, pass);
        }
        else {
            return new AssertionReturn(`Expected ${itemType.tfResourceType} with properties ${JSON.stringify(properties, jestAsymetricMatcherStringifyReplacer)} to be present in synthesized stack.
Found ${items.length === 0 ? "no" : items.length} ${itemType.tfResourceType} resources instead${items.length > 0 ? ":\n" + JSON.stringify(items, null, 2) : ""}`, pass);
        }
    };
}
/**
 * Returns the function toHaveDataSourceWithProperties using the evaluation properties of customPassEvaluation
 * @param customPassEvaluation
 * @returns {getToHaveDataSourceWithProperties~toHaveDataSourceWithProperties}
 */
function getToHaveDataSourceWithProperties(customPassEvaluation) {
    /**
     * Evaluates the received stack to have the data source resourceType containing specified properties
     * @param received
     * @param resourceType
     * @param properties
     * @returns {AssertionReturn}
     */
    return function toHaveDataSourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties("toHaveDataSourceWithProperties", customPassEvaluation)("data", received, resourceType, properties);
    };
}
/**
 * Returns the function toHaveResourceWithProperties using the evaluation properties of customPassEvaluation
 * @param customPassEvaluation
 * @returns
 */
function getToHaveResourceWithProperties(customPassEvaluation) {
    /**
     * Evaluates the received stack to have the resource resourceType containing specified properties
     * @param received
     * @param resourceType
     * @param properties
     * @returns {AssertionReturn}
     */
    return function toHaveResourceWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties("toHaveResourceWithProperties", customPassEvaluation)("resource", received, resourceType, properties);
    };
}
/**
 * A helper util to verify wether an Error was caused by the Nodejs `process.spawn` API.
 *
 * @param   {Error}   err The Error object to verify
 * @returns {Boolean}     A bool indicating wether the input Error is containing process.spawn output.
 */
const isExecSpawnError = (err) => "output" in err &&
    Array.isArray(err.output) &&
    err.output.some((buf) => Buffer.isBuffer(buf));
/**
 * A helper util to append `process.spawn` output to assertion messages to improve developer expirience.
 *
 * @param   {String} message The message to optionally append process output to.
 * @param   {Error}  err     The error from which the `process.spawn` output should be retreived from.
 * @returns {String}         The finalized assertion message decorated with the `process.spawn` output.
 */
const withProcessOutput = (message, err) => {
    let output = "";
    if (isExecSpawnError(err)) {
        output =
            err.output
                ?.map((buffer) => buffer?.toString("utf8"))
                .filter(Boolean)
                .join("\n") ?? "";
    }
    const appendix = output.length ? `. Output: ${output}` : "";
    return `${message}: ${err}${appendix}.`;
};
/**
 * Returns the function toHaveProviderWithProperties using the evaluation properties of customPassEvaluation
 * @param customPassEvaluation
 * @returns {getToHaveProviderWithProperties~toHaveProviderWithProperties}
 */
function getToHaveProviderWithProperties(customPassEvaluation) {
    /**
     * Evaluates the received stack to have the provider resourceType containing specified properties
     * @param received
     * @param resourceType
     * @param properties
     * @returns {AssertionReturn}
     */
    return function toHaveProviderWithProperties(received, resourceType, properties = {}) {
        return getAssertElementWithProperties("toHaveProviderWithProperties", customPassEvaluation)("provider", received, resourceType, properties);
    };
}
/**
 * Evaluates the validity of the received stack
 * @param received
 * @returns {AssertionReturn}
 */
function toBeValidTerraform(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw (0, errors_1.matchersPathIsNotDirectory)("toBeValidTerraform");
        }
    }
    catch (e) {
        return new AssertionReturn(`Expected subject to be a terraform directory: ${e}`, false);
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([name, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "pipe",
            };
            (0, child_process_1.execSync)(`${util_1.terraformBinaryName} init`, opts);
            const out = (0, child_process_1.execSync)(`${util_1.terraformBinaryName} validate -json`, opts);
            const result = JSON.parse(out.toString());
            if (!result.valid) {
                throw (0, errors_1.matchersFoundErrorsInStack)(result.error_count, name, result.diagnostics.join("\n"));
            }
        });
        return new AssertionReturn(`Expected subject not to be a valid terraform stack`, true);
    }
    catch (e) {
        return new AssertionReturn(withProcessOutput(`Expected subject to be a valid terraform stack`, e), false);
    }
}
/**
 * Evaluates the ability for the received stack to successfully plan
 * @param received
 * @returns {AssertionReturn}
 */
function toPlanSuccessfully(received) {
    try {
        if (!fs.statSync(received).isDirectory()) {
            throw (0, errors_1.matchersPathIsNotDirectory)("toPlanSuccessfully");
        }
    }
    catch (e) {
        return new AssertionReturn(`Expected subject to be a terraform directory: ${e}`, false);
    }
    try {
        const manifest = JSON.parse(fs.readFileSync(path.resolve(received, "manifest.json"), "utf8"));
        const stacks = Object.entries(manifest.stacks);
        stacks.forEach(([, stack]) => {
            const opts = {
                cwd: path.resolve(received, stack.workingDirectory),
                env: process.env,
                stdio: "ignore",
            };
            (0, child_process_1.execSync)(`${util_1.terraformBinaryName} init`, opts);
            // Throws on a non-zero exit code
            (0, child_process_1.execSync)(`${util_1.terraformBinaryName} plan -input=false -lock=false `, opts);
        });
        return new AssertionReturn(`Expected subject not to plan successfully`, true);
    }
    catch (e) {
        return new AssertionReturn(withProcessOutput(`Expected subject to plan successfully`, e), false);
    }
}
//# sourceMappingURL=data:application/json;base64,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