"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Testing = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
/* eslint-disable @typescript-eslint/no-require-imports */
const fs = require("fs");
const path = require("path");
const os = require("os");
const stringify = require("json-stable-stringify");
const lib_1 = require("../../lib");
const manifest_1 = require("../manifest");
const features_1 = require("../features");
const jest_1 = require("./adapters/jest");
const synthesizer_1 = require("../synthesize/synthesizer");
const matchers_1 = require("./matchers");
const DefaultTestingAppConfig = {
    stackTraces: false,
    stubVersion: true,
    enableFutureFlags: true,
    fakeCdktfJsonPath: false,
};
/**
 * Testing utilities for cdktf applications.
 */
class Testing {
    /**
     * Returns an app for testing with the following properties:
     * - Output directory is a temp dir.
     */
    static app(options = {}) {
        const appConfig = { ...DefaultTestingAppConfig, ...options };
        if (!appConfig.outdir) {
            appConfig.outdir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf.outdir."));
        }
        const app = new lib_1.App({
            outdir: appConfig.outdir,
            stackTraces: appConfig.stackTraces,
            context: options.context,
        });
        if (appConfig.stubVersion) {
            this.stubVersion(app);
        }
        if (appConfig.enableFutureFlags) {
            this.enableFutureFlags(app);
        }
        if (appConfig.fakeCdktfJsonPath) {
            this.fakeCdktfJsonPath(app);
        }
        return app;
    }
    static stubVersion(app) {
        app.node.setContext("cdktfVersion", "stubbed");
        app.manifest.version = "stubbed";
        return app;
    }
    static fakeCdktfJsonPath(app) {
        app.node.setContext("cdktfJsonPath", `${process.cwd()}/cdktf.json`);
        app.node.setContext("cdktfRelativeModules", ["./"]);
        return app;
    }
    static enableFutureFlags(app) {
        const node = app.node;
        Object.entries(features_1.FUTURE_FLAGS).forEach(([key, value]) => node.setContext(key, value));
        return app;
    }
    static synthScope(fn) {
        const stack = new lib_1.TerraformStack(Testing.app(), "stack");
        fn(stack);
        return Testing.synth(stack);
    }
    /**
     * Returns the Terraform synthesized JSON.
     */
    static synth(stack, runValidations = false) {
        (0, synthesizer_1.invokeAspects)(stack);
        if (runValidations) {
            stack.runAllValidations();
        }
        const tfConfig = stack.toTerraform();
        // eslint-disable-next-line jsdoc/require-jsdoc
        function removeMetadata(item) {
            if (item !== null && typeof item === "object") {
                if (Array.isArray(item)) {
                    return item.map(removeMetadata);
                }
                const cleanedItem = Object.entries(item)
                    // order alphabetically
                    .sort(([a], [b]) => a.localeCompare(b))
                    .reduce((acc, [key, value]) => ({ ...acc, [key]: removeMetadata(value) }), {});
                // Remove metadata
                delete cleanedItem["//"];
                return cleanedItem;
            }
            return item;
        }
        const cleaned = removeMetadata(tfConfig);
        return stringify(cleaned, { space: 2 });
    }
    /**
     * Returns the Terraform synthesized JSON.
     */
    static synthHcl(stack, runValidations = false, returnMetadata = false) {
        (0, synthesizer_1.invokeAspects)(stack);
        if (runValidations) {
            stack.runAllValidations();
        }
        const config = stack.toHclTerraform();
        if (returnMetadata)
            return config.metadata;
        return config.hcl;
    }
    static fullSynth(stack) {
        const outdir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf.outdir."));
        const manifest = new manifest_1.Manifest("stubbed", outdir, false);
        stack.synthesizer.synthesize({
            outdir,
            manifest,
        });
        manifest.writeToFile();
        return outdir;
    }
    static renderConstructTree(construct) {
        return render(construct, 0, false);
        // eslint-disable-next-line jsdoc/require-jsdoc
        function render(construct, level, isLast) {
            let prefix = "";
            if (level > 0) {
                const spaces = " ".repeat((level - 1) * 4);
                const symbol = isLast ? "└" : "├";
                prefix = `${spaces}${symbol}── `;
            }
            const name = lib_1.App.isApp(construct)
                ? "App"
                : `${construct.node.id} (${construct.constructor.name})`;
            return `${prefix}${name}\n${construct.node.children
                .map((child, idx, arr) => {
                const isLast = idx === arr.length - 1;
                return render(child, level + 1, isLast);
            })
                .join("")}`;
        }
    }
    static toHaveDataSourceWithProperties(received, resourceType, properties = {}) {
        return (0, matchers_1.getToHaveDataSourceWithProperties)()(received, { tfResourceType: resourceType }, properties).pass;
    }
    static toHaveDataSource(received, resourceType) {
        return (0, matchers_1.getToHaveDataSourceWithProperties)()(received, { tfResourceType: resourceType }, {}).pass;
    }
    static toHaveResourceWithProperties(received, resourceType, properties = {}) {
        return (0, matchers_1.getToHaveResourceWithProperties)()(received, { tfResourceType: resourceType }, properties).pass;
    }
    static toHaveResource(received, resourceType) {
        return (0, matchers_1.getToHaveResourceWithProperties)()(received, { tfResourceType: resourceType }, {}).pass;
    }
    static toHaveProviderWithProperties(received, resourceType, properties = {}) {
        return (0, matchers_1.getToHaveProviderWithProperties)()(received, { tfResourceType: resourceType }, properties).pass;
    }
    static toHaveProvider(received, resourceType) {
        return (0, matchers_1.getToHaveProviderWithProperties)()(received, { tfResourceType: resourceType }, {}).pass;
    }
    static toBeValidTerraform(received) {
        return (0, matchers_1.toBeValidTerraform)(received).pass;
    }
    static setupJest() {
        (0, jest_1.setupJest)();
    }
}
exports.Testing = Testing;
_a = JSII_RTTI_SYMBOL_1;
Testing[_a] = { fqn: "cdktf.Testing", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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