"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const index_1 = require("../index");
const resource_1 = require("../../../test/helper/resource");
const matchers_1 = require("../matchers");
const data_source_1 = require("../../../test/helper/data-source");
const terraform_stack_1 = require("../../terraform-stack");
const provider_1 = require("../../../test/helper/provider");
const fs = require("fs");
const path = require("path");
function corruptSynthesizedStack(stackPath) {
    const manifest = JSON.parse(fs.readFileSync(path.resolve(stackPath, "manifest.json"), "utf8"));
    const { synthesizedStackPath } = Object.values(manifest.stacks)[0];
    const absoluteStackPath = path.resolve(stackPath, synthesizedStackPath);
    fs.writeFileSync(absoluteStackPath, fs.readFileSync(absoluteStackPath, "utf8") +
        "Weird text that makes the JSON file invalid", "utf8");
}
describe("deepEqualIgnoringObjectCasing", () => {
    it("should compare simple items", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(1, 1)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)("1", "1")).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(true, true)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(false, false)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(null, null)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(undefined, undefined)).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 3])).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)(1, 2)).toBe(false);
    });
    it("should compare arrays", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 3])).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 4])).toBe(false);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)([1, 2, 3], [1, 2, 3, 4])).toBe(false);
    });
    it("should compare objects", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 1, b: 2 }, { a: 1, b: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { c: 3 }, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 3, b: 2 }, { a: { c: 3 }, b: 2 })).toBe(false);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { c: 3 }, b: 2 }, { a: 3, b: 2 })).toBe(false);
    });
    it("should ignore case when comparing object keys", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: 1, fooBar: 2 }, { a: 1, foo_bar: 2 })).toBe(true);
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { fooBar: 2 } }, { a: { foo_bar: 2 } })).toBe(true);
    });
    it("should ignore keys not present in expectation", () => {
        expect((0, matchers_1.asymetricDeepEqualIgnoringObjectCasing)({ a: { fooBar: 2 } }, { a: { foo_bar: 2 }, b: 3 })).toBe(true);
    });
});
describe("matchers", () => {
    const toHaveResourceWithProperties = (0, matchers_1.getToHaveResourceWithProperties)();
    describe("toHaveResourceWithProperties", () => {
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test", tags: { foo: "bar" } });
            });
        });
        it("should pass with no properties", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeTruthy();
            expect(res.message).toMatchInlineSnapshot(`
        "Expected no test_resource with properties {} to be present in synthesized stack.
        Found 1 test_resource resources instead:
        [
          {
            "name": "test",
            "tags": {
              "foo": "bar"
            }
          }
        ]"
      `);
        });
        it("should fail with wrong resouce", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, data_source_1.TestDataSource);
            expect(res.pass).toBeFalsy();
            expect(res.message).toMatchInlineSnapshot(`
        "Expected test_data_source with properties {} to be present in synthesized stack.
        Found no test_data_source resources instead"
      `);
        });
        it("should work on nested elements", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.TestResource, {
                tags: { foo: "bar" },
            });
            expect(res.pass).toBeTruthy();
        });
        it("should fail if no item is found as asserted", () => {
            const res = toHaveResourceWithProperties(synthesizedStack, resource_1.DockerImage);
            expect(res.pass).toBeFalsy();
        });
    });
    describe("toHaveDataSourceWithProperties", () => {
        const toHaveDataSourceWithProperties = (0, matchers_1.getToHaveDataSourceWithProperties)();
        let synthesizedStack;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new resource_1.TestResource(scope, "test", { name: "test" });
                new data_source_1.TestDataSource(scope, "test-data-source", { name: "data" });
            });
        });
        it("should not find resources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeFalsy();
        });
        it("should find data sources", () => {
            const res = toHaveDataSourceWithProperties(synthesizedStack, data_source_1.TestDataSource, { name: "data" });
            expect(res.pass).toBeTruthy();
        });
    });
    describe("toHaveProviderWithProperties", () => {
        const toHaveProviderWithProperties = (0, matchers_1.getToHaveProviderWithProperties)();
        let synthesizedStack;
        let synthesizedStackNoProvider;
        beforeEach(() => {
            synthesizedStack = index_1.Testing.synthScope((scope) => {
                new provider_1.DockerProvider(scope, "test-provider", {
                    alias: "test-alias",
                    ssh_opts: ["-o", "StrictHostKeyChecking=no"],
                });
                new resource_1.DockerImage(scope, "test-image", { name: "test" });
            });
            synthesizedStackNoProvider = index_1.Testing.synthScope((scope) => {
                new resource_1.DockerImage(scope, "test", { name: "test" });
            });
        });
        it("should find provider", () => {
            const res = toHaveProviderWithProperties(synthesizedStack, provider_1.DockerProvider, { alias: "test-alias", ssh_opts: ["-o", "StrictHostKeyChecking=no"] });
            expect(res.pass).toBeTruthy();
        });
        it("should not find provider", () => {
            const res = toHaveProviderWithProperties(synthesizedStackNoProvider, provider_1.DockerProvider, {});
            expect(res.pass).toBeFalsy();
        });
        it("should not find resources", () => {
            const res = toHaveProviderWithProperties(synthesizedStack, resource_1.TestResource);
            expect(res.pass).toBeFalsy();
        });
        it("should not find data sources", () => {
            const res = toHaveProviderWithProperties(synthesizedStack, data_source_1.TestDataSource);
            expect(res.pass).toBeFalsy();
        });
    });
    describe("toBeValidTerraform", () => {
        it("fails if anything but a path is passed", () => {
            const res = (0, matchers_1.toBeValidTerraform)("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = (0, matchers_1.toBeValidTerraform)(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject not to be a valid terraform stack"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = (0, matchers_1.toBeValidTerraform)(result);
            expect(res.pass).toBeFalsy();
            expect(res.message).toEqual(expect.stringContaining("Expected subject to be a valid terraform stack"));
            const tf1_3 = "There are some problems with the configuration, described below";
            const tf1_4 = "Terraform encountered problems during initialisation, including problems\n with the configuration, described below";
            const tf1_5 = "Extra characters appear after the JSON value";
            expect(res.message).toEqual(expect.stringMatching(new RegExp(`${tf1_3}|${tf1_4}|${tf1_5}`)));
        });
    });
    describe("toPlanSuccessfully", () => {
        it("fails if anything but a path is passed", () => {
            const res = (0, matchers_1.toPlanSuccessfully)("not a path");
            expect(res.pass).toBeFalsy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject to be a terraform directory: Error: ENOENT: no such file or directory, stat 'not a path'"`);
        });
        it("succeeds if the terraform config passed is valid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const res = (0, matchers_1.toPlanSuccessfully)(index_1.Testing.fullSynth(stack));
            expect(res.pass).toBeTruthy();
            expect(res.message).toMatchInlineSnapshot(`"Expected subject not to plan successfully"`);
        });
        it("fails if the terraform config passed is invalid", () => {
            const app = index_1.Testing.app();
            const stack = new terraform_stack_1.TerraformStack(app, "test");
            new provider_1.DockerProvider(stack, "provider", {});
            new resource_1.DockerImage(stack, "test", { name: "test" });
            const result = index_1.Testing.fullSynth(stack);
            corruptSynthesizedStack(result);
            const res = (0, matchers_1.toPlanSuccessfully)(result);
            expect(res.pass).toBeFalsy();
            expect(res.message).toEqual(expect.stringContaining("Expected subject to plan successfully"));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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