"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformStack = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const constructs_1 = require("constructs");
const _tokens_1 = require("./_tokens");
const terraform_element_1 = require("./terraform-element");
const util_1 = require("./util");
const terraform_provider_1 = require("./terraform-provider");
const local_backend_1 = require("./backends/local-backend");
const tfExpression_1 = require("./tfExpression");
const terraform_output_1 = require("./terraform-output");
const unique_1 = require("./private/unique");
const synthesizer_1 = require("./synthesize/synthesizer");
const STACK_SYMBOL = Symbol.for("cdktf/TerraformStack");
const validations_1 = require("./validations");
const app_1 = require("./app");
const terraform_backend_1 = require("./terraform-backend");
const terraform_resource_targets_1 = require("./terraform-resource-targets");
const terraform_resource_1 = require("./terraform-resource");
const render_1 = require("./hcl/render");
const errors_1 = require("./errors");
// eslint-disable-next-line jsdoc/require-jsdoc
function throwIfIdIsGlobCharacter(str) {
    ["*", "?", "[", "]", "{", "}", "!"].forEach((char) => {
        if (str.includes(char)) {
            throw (0, errors_1.stackContainsDisallowedChar)(str, char);
        }
    });
}
// eslint-disable-next-line jsdoc/require-jsdoc
function throwIfIdContainsWhitespace(str) {
    if (/\s/.test(str)) {
        throw (0, errors_1.stackIdContainsWhitespace)(str);
    }
}
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformStack extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        this.rawOverrides = {};
        this.crossStackOutputs = {};
        this.crossStackDataSources = {};
        this.dependencies = [];
        this.moveTargets = new terraform_resource_targets_1.TerraformResourceTargets();
        throwIfIdIsGlobCharacter(id);
        throwIfIdContainsWhitespace(id);
        this.cdktfVersion = this.node.tryGetContext("cdktfVersion");
        this.synthesizer = new synthesizer_1.StackSynthesizer(this, process.env.CDKTF_CONTINUE_SYNTH_ON_ERROR_ANNOTATIONS !== undefined, process.env.SYNTH_HCL_OUTPUT === "true" ||
            process.env.SYNTH_HCL_OUTPUT === "1");
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this.node.addValidation(new validations_1.ValidateProviderPresence(this));
    }
    static isStack(x) {
        return x !== null && typeof x === "object" && STACK_SYMBOL in x;
    }
    static of(construct) {
        return _lookup(construct);
        // eslint-disable-next-line jsdoc/require-jsdoc
        function _lookup(c) {
            if (TerraformStack.isStack(c)) {
                return c;
            }
            const node = c.node;
            if (!node.scope) {
                let hint = "";
                if (construct.node.scope === c && app_1.App.isApp(c)) {
                    // the scope of the originally passed construct equals the construct c
                    // which has no scope (i.e. has no parent construct) and c is an App
                    // and our construct is a Backend
                    hint = `. You seem to have passed your root App as scope to a construct. Pass a stack (inheriting from TerraformStack) as scope to your construct instead.`;
                }
                throw (0, errors_1.noStackForConstruct)(construct.node.path, hint);
            }
            return _lookup(node.scope);
        }
    }
    findAll(predicate) {
        const items = [];
        const visit = async (node) => {
            if (predicate(node)) {
                items.push(node);
            }
            for (const child of node.node.children) {
                visit(child);
            }
        };
        visit(this);
        return items;
    }
    prepareStack() {
        // Ensure we have a backend configured
        this.ensureBackendExists();
        // A preparing resolve run might add new resources to the stack, e.g. for cross stack references.
        terraformElements(this).forEach((e) => (0, _tokens_1.resolve)(this, e.toTerraform(), true));
    }
    addOverride(path, value) {
        const parts = path.split(".");
        let curr = this.rawOverrides;
        while (parts.length > 1) {
            // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
            const key = parts.shift();
            // if we can't recurse further or the previous value is not an
            // object overwrite it with an object.
            const isObject = curr[key] != null &&
                typeof curr[key] === "object" &&
                !Array.isArray(curr[key]);
            if (!isObject) {
                curr[key] = {};
            }
            curr = curr[key];
        }
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        const lastKey = parts.shift();
        curr[lastKey] = value;
    }
    getLogicalId(tfElement) {
        // wrap the allocation for future renaming support
        return this.allocateLogicalId(tfElement);
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * @param tfElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(tfElement) {
        const node = terraform_element_1.TerraformElement.isTerraformElement(tfElement)
            ? tfElement.node
            : tfElement;
        const stack = terraform_element_1.TerraformElement.isTerraformElement(tfElement)
            ? tfElement.cdktfStack
            : this;
        const stackIndex = node.scopes.indexOf(stack);
        const components = node.scopes.slice(stackIndex + 1).map((c) => c.node.id);
        return components.length > 0 ? (0, unique_1.makeUniqueId)(components) : "";
    }
    allProviders() {
        return this.findAll(terraform_provider_1.TerraformProvider.isTerraformProvider);
    }
    ensureBackendExists() {
        const backends = this.findAll(terraform_backend_1.TerraformBackend.isBackend);
        return backends[0] || new local_backend_1.LocalBackend(this, {});
    }
    toHclTerraform() {
        const metadata = {
            version: this.cdktfVersion,
            stackName: this.node.id,
            backend: "local", // overwritten by backend implementations if used
            cloud: undefined, // overwritten by cloud and remote backend implementations
            ...(Object.keys(this.rawOverrides).length > 0
                ? { overrides: { stack: Object.keys(this.rawOverrides) } }
                : {}),
        };
        const elements = terraformElements(this);
        const metadatas = elements.map((e) => (0, _tokens_1.resolve)(this, e.toMetadata()));
        for (const meta of metadatas) {
            (0, util_1.deepMerge)(metadata, meta);
        }
        const outputs = elements.reduce((carry, item) => {
            if (!terraform_output_1.TerraformOutput.isTerraformOutput(item)) {
                return carry;
            }
            (0, util_1.deepMerge)(carry, item.node.path.split("/").reduceRight((innerCarry, part) => {
                if (Object.keys(innerCarry).length === 0) {
                    return { [part]: item.friendlyUniqueId };
                }
                return { [part]: innerCarry };
            }, {}));
            return carry;
        }, {});
        const tf = {};
        const fragments = elements.map((e) => (0, _tokens_1.resolve)(this, e.toHclTerraform()));
        const locals = {};
        const tfMeta = {
            "//": {
                metadata,
                outputs,
            },
        };
        const hclFragments = fragments
            .map((frag) => {
            let res = "";
            if (frag.resource) {
                const { hcl, metadata } = (0, render_1.renderResource)(frag.resource);
                (0, util_1.deepMerge)(tfMeta, metadata);
                res = [res, hcl].join("\n");
            }
            if (frag.data) {
                const { hcl, metadata } = (0, render_1.renderDatasource)(frag.data);
                (0, util_1.deepMerge)(tfMeta, metadata);
                res = [res, hcl].join("\n");
            }
            if (frag.provider) {
                (0, util_1.deepMerge)(tf, frag);
                res = [res, (0, render_1.renderProvider)(frag.provider)].join("\n\n");
            }
            if (frag.terraform) {
                (0, util_1.deepMerge)(tf, frag);
            }
            if (frag.module) {
                const { hcl, metadata } = (0, render_1.renderModule)(frag.module);
                (0, util_1.deepMerge)(tfMeta, metadata);
                res = [res, hcl].join("\n");
            }
            if (frag.output) {
                res = [res, (0, render_1.renderOutput)(frag.output)].join("\n\n");
            }
            if (frag.moved) {
                res = [res, (0, render_1.renderMoved)(frag.moved)].join("\n\n");
            }
            if (frag.import) {
                res = [res, (0, render_1.renderImport)(frag.import)].join("\n\n");
            }
            if (frag.locals) {
                (0, util_1.deepMerge)(locals, frag);
            }
            if (frag.variable) {
                res = [res, (0, render_1.renderVariable)(frag.variable)].join("\n\n");
            }
            return res;
        })
            .filter((frag) => frag !== undefined);
        (0, util_1.deepMerge)(tf, this.rawOverrides);
        const terraformBlock = tf?.terraform;
        let terraformBlockHcl = "";
        if (terraformBlock) {
            terraformBlockHcl = (0, render_1.renderTerraform)(terraformBlock);
            (0, util_1.deepMerge)(tfMeta, { terraform: (0, render_1.cleanForMetadata)(terraformBlock) });
        }
        let localsHcl = "";
        if (locals) {
            localsHcl = (0, render_1.renderLocals)(locals.locals);
            if (localsHcl) {
                // Hacky way to add a newline between the terraform block and the locals block
                localsHcl = "\n\n" + localsHcl;
            }
        }
        return {
            hcl: (0, _tokens_1.resolve)(this, [terraformBlockHcl, localsHcl, ...hclFragments].join("")),
            metadata: (0, _tokens_1.resolve)(this, tfMeta),
        };
    }
    toTerraform() {
        const tf = {};
        const metadata = {
            version: this.cdktfVersion,
            stackName: this.node.id,
            backend: "local", // overwritten by backend implementations if used
            cloud: undefined, // overwritten by cloud and remote backend implementations
            ...(Object.keys(this.rawOverrides).length > 0
                ? { overrides: { stack: Object.keys(this.rawOverrides) } }
                : {}),
        };
        const elements = terraformElements(this);
        const metadatas = elements.map((e) => (0, _tokens_1.resolve)(this, e.toMetadata()));
        for (const meta of metadatas) {
            (0, util_1.deepMerge)(metadata, meta);
        }
        const outputs = elements.reduce((carry, item) => {
            if (!terraform_output_1.TerraformOutput.isTerraformOutput(item)) {
                return carry;
            }
            (0, util_1.deepMerge)(carry, item.node.path.split("/").reduceRight((innerCarry, part) => {
                if (Object.keys(innerCarry).length === 0) {
                    return { [part]: item.friendlyUniqueId };
                }
                return { [part]: innerCarry };
            }, {}));
            return carry;
        }, {});
        tf["//"] = { metadata, outputs };
        const fragments = elements.map((e) => (0, _tokens_1.resolve)(this, e.toTerraform()));
        for (const fragment of fragments) {
            (0, util_1.deepMerge)(tf, fragment);
        }
        (0, util_1.deepMerge)(tf, this.rawOverrides);
        return (0, _tokens_1.resolve)(this, tf);
    }
    registerOutgoingCrossStackReference(identifier) {
        if (this.crossStackOutputs[identifier]) {
            return this.crossStackOutputs[identifier];
        }
        const output = new terraform_output_1.TerraformOutput(this, `cross-stack-output-${identifier}`, {
            value: (0, tfExpression_1.ref)(identifier, this),
            sensitive: true,
        });
        this.crossStackOutputs[identifier] = output;
        return output;
    }
    registerIncomingCrossStackReference(fromStack) {
        if (this.crossStackDataSources[String(fromStack)]) {
            return this.crossStackDataSources[String(fromStack)];
        }
        const originBackend = fromStack.ensureBackendExists();
        const originPath = fromStack.node.path;
        const remoteState = originBackend.getRemoteStateDataSource(this, `cross-stack-reference-input-${originPath}`, originPath);
        this.crossStackDataSources[originPath] = remoteState;
        return remoteState;
    }
    // Check here for loops in the dependency graph
    dependsOn(stack) {
        return (this.dependencies.includes(stack) ||
            this.dependencies.some((d) => d.dependsOn(stack)));
    }
    addDependency(dependency) {
        if (dependency.dependsOn(this)) {
            throw (0, errors_1.stackHasCircularDependency)(this, dependency);
        }
        if (this.dependencies.includes(dependency)) {
            return;
        }
        this.dependencies.push(dependency);
    }
    /**
     * Run all validations on the stack.
     */
    runAllValidations() {
        const errors = this.node
            .findAll()
            .map((node) => node.node.validate().map((error) => ({ message: error, source: node })))
            .reduce((prev, curr) => [...prev, ...curr], []);
        if (errors.length > 0) {
            const errorList = errors
                .map((e) => `[${e.source.node.path}] ${e.message}`)
                .join("\n  ");
            throw (0, errors_1.stackValidationFailure)(errorList);
        }
    }
    hasResourceMove() {
        return terraformElements(this).some((e) => {
            if (terraform_resource_1.TerraformResource.isTerraformResource(e) && e.hasResourceMove()) {
                return true;
            }
            return false;
        });
    }
}
exports.TerraformStack = TerraformStack;
_a = JSII_RTTI_SYMBOL_1;
TerraformStack[_a] = { fqn: "cdktf.TerraformStack", version: "0.21.0" };
// eslint-disable-next-line jsdoc/require-jsdoc
function terraformElements(node, into = []) {
    if (terraform_element_1.TerraformElement.isTerraformElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (TerraformStack.isStack(child)) {
            continue;
        }
        terraformElements(child, into);
    }
    return into;
}
//# sourceMappingURL=data:application/json;base64,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