"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformResource = void 0;
exports.lifecycleToTerraform = lifecycleToTerraform;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const tokens_1 = require("./tokens");
const terraform_element_1 = require("./terraform-element");
const util_1 = require("./util");
const tfExpression_1 = require("./tfExpression");
const terraform_count_1 = require("./terraform-count");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const assert = require("assert");
const validate_terraform_version_1 = require("./validations/validate-terraform-version");
const terraform_stack_1 = require("./terraform-stack");
const errors_1 = require("./errors");
const TERRAFORM_RESOURCE_SYMBOL = Symbol.for("cdktf/TerraformResource");
/**
 * prepares a lifecycle object for being rendered as JSON
 * currently this function:
 *  - converts all replaceTriggeredBy items that are ITerraformDependables to strings
 */
function lifecycleToTerraform(lifecycle) {
    if (!lifecycle) {
        return undefined;
    }
    return {
        ...lifecycle,
        ...(lifecycle?.replaceTriggeredBy?.length
            ? {
                replaceTriggeredBy: lifecycle?.replaceTriggeredBy?.map((x) => {
                    if (typeof x === "string") {
                        return x;
                    }
                    else {
                        return x.fqn;
                    }
                }),
            }
            : undefined),
    };
}
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformResource extends terraform_element_1.TerraformElement {
    constructor(scope, id, config) {
        super(scope, id, config.terraformResourceType);
        this._hasMoved = false;
        Object.defineProperty(this, TERRAFORM_RESOURCE_SYMBOL, { value: true });
        this.terraformResourceType = config.terraformResourceType;
        this.terraformGeneratorMetadata = config.terraformGeneratorMetadata;
        if (Array.isArray(config.dependsOn)) {
            this.dependsOn = config.dependsOn.map((dependency) => (0, tfExpression_1.dependable)(dependency));
        }
        this.count = config.count;
        this.provider = config.provider;
        this.lifecycle = config.lifecycle;
        this.forEach = config.forEach;
        this.provisioners = config.provisioners;
        this.connection = config.connection;
    }
    static isTerraformResource(x) {
        return (x !== null && typeof x === "object" && TERRAFORM_RESOURCE_SYMBOL in x);
    }
    hasResourceMove() {
        return this._movedById || this._movedByTarget;
    }
    getStringAttribute(terraformAttribute) {
        return tokens_1.Token.asString(this.interpolationForAttribute(terraformAttribute));
    }
    getNumberAttribute(terraformAttribute) {
        return tokens_1.Token.asNumber(this.interpolationForAttribute(terraformAttribute));
    }
    getListAttribute(terraformAttribute) {
        return tokens_1.Token.asList(this.interpolationForAttribute(terraformAttribute));
    }
    getBooleanAttribute(terraformAttribute) {
        return this.interpolationForAttribute(terraformAttribute);
    }
    getNumberListAttribute(terraformAttribute) {
        return tokens_1.Token.asNumberList(this.interpolationForAttribute(terraformAttribute));
    }
    getStringMapAttribute(terraformAttribute) {
        return tokens_1.Token.asStringMap(this.interpolationForAttribute(terraformAttribute));
    }
    getNumberMapAttribute(terraformAttribute) {
        return tokens_1.Token.asNumberMap(this.interpolationForAttribute(terraformAttribute));
    }
    getBooleanMapAttribute(terraformAttribute) {
        return tokens_1.Token.asBooleanMap(this.interpolationForAttribute(terraformAttribute));
    }
    getAnyMapAttribute(terraformAttribute) {
        return tokens_1.Token.asAnyMap(this.interpolationForAttribute(terraformAttribute));
    }
    get terraformMetaArguments() {
        assert(!this.forEach || typeof this.count === "undefined", `forEach and count are both set, but they are mutually exclusive. You can only use either of them. Check the resource at path: ${this.node.path}`);
        return {
            dependsOn: this.dependsOn,
            count: terraform_count_1.TerraformCount.isTerraformCount(this.count)
                ? this.count.toTerraform()
                : this.count,
            provider: this.provider?.fqn,
            lifecycle: lifecycleToTerraform(this.lifecycle),
            forEach: this.forEach?._getForEachExpression(),
            connection: this.connection,
        };
    }
    // jsii can't handle abstract classes?
    synthesizeAttributes() {
        return {};
    }
    synthesizeHclAttributes() {
        return {};
    }
    /**
     * Adds this resource to the terraform JSON output.
     */
    toTerraform() {
        const attributes = (0, util_1.deepMerge)((0, util_1.processDynamicAttributes)(this.synthesizeAttributes()), (0, util_1.keysToSnakeCase)(this.terraformMetaArguments), {
            provisioner: this.provisioners?.map(({ type, ...props }) => ({
                [type]: (0, util_1.keysToSnakeCase)(props),
            })),
        }, this.rawOverrides);
        attributes["//"] = {
            ...(attributes["//"] ?? {}),
            ...this.constructNodeMetadata,
        };
        const movedBlock = this._buildMovedBlock();
        return {
            resource: this._hasMoved
                ? undefined
                : {
                    [this.terraformResourceType]: {
                        [this.friendlyUniqueId]: attributes,
                    },
                },
            moved: movedBlock
                ? [
                    {
                        to: movedBlock.to,
                        from: movedBlock.from,
                    },
                ]
                : undefined,
            import: this._imported
                ? [
                    {
                        provider: this._imported.provider?.fqn,
                        id: this._imported.id,
                        to: `${this.terraformResourceType}.${this.friendlyUniqueId}`,
                    },
                ]
                : undefined,
        };
    }
    toHclTerraform() {
        const attributes = (0, util_1.deepMerge)((0, util_1.processDynamicAttributesForHcl)(this.synthesizeHclAttributes()), (0, util_1.keysToSnakeCase)(this.terraformMetaArguments), {
            provisioner: this.provisioners?.map(({ type, ...props }) => ({
                [type]: {
                    isBlock: true,
                    type: "list",
                    storageClassType: "object",
                    value: (0, util_1.keysToSnakeCase)(props),
                },
            })),
        }, this.rawOverrides);
        attributes["//"] = {
            ...(attributes["//"] ?? {}),
            ...this.constructNodeMetadata,
        };
        const movedBlock = this._buildMovedBlock();
        return {
            resource: this._hasMoved
                ? undefined
                : {
                    [this.terraformResourceType]: {
                        [this.friendlyUniqueId]: attributes,
                    },
                },
            moved: movedBlock
                ? [
                    {
                        to: {
                            value: movedBlock.to,
                            isBlock: false,
                            type: "simple",
                            storageClassType: "reference",
                        },
                        from: {
                            value: movedBlock.from,
                            isBlock: false,
                            type: "simple",
                            storageClassType: "reference",
                        },
                    },
                ]
                : undefined,
            import: this._imported
                ? [
                    {
                        to: {
                            value: `${this.terraformResourceType}.${this.friendlyUniqueId}`,
                            type: "simple",
                            storageClassType: "reference",
                        },
                        id: {
                            value: this._imported.id,
                            type: "simple",
                            storageClassType: "string",
                        },
                        provider: this._imported.provider
                            ? {
                                value: this._imported.provider.fqn,
                                type: "simple",
                                storageClassType: "reference",
                            }
                            : undefined,
                    },
                ]
                : undefined,
        };
    }
    toMetadata() {
        return {
            overrides: Object.keys(this.rawOverrides).length
                ? {
                    [this.terraformResourceType]: Object.keys(this.rawOverrides),
                }
                : undefined,
            imports: this._imported
                ? {
                    [this.terraformResourceType]: [this.friendlyUniqueId],
                }
                : undefined,
            moved: this._movedByTarget || this._movedById
                ? {
                    [this.terraformResourceType]: [this.friendlyUniqueId],
                }
                : undefined,
        };
    }
    interpolationForAttribute(terraformAttribute) {
        return (0, tfExpression_1.ref)(`${this.terraformResourceType}.${this.friendlyUniqueId}${this.forEach ? ".*" : ""}.${terraformAttribute}`, this.cdktfStack);
    }
    importFrom(id, provider) {
        this._imported = { id, provider };
        this.node.addValidation(new validate_terraform_version_1.ValidateTerraformVersion(">=1.5", `Import blocks are only supported for Terraform >=1.5. Please upgrade your Terraform version.`));
    }
    _getResourceTarget(moveTarget) {
        return terraform_stack_1.TerraformStack.of(this).moveTargets.getResourceByTarget(moveTarget);
    }
    _addResourceTarget(moveTarget) {
        return terraform_stack_1.TerraformStack.of(this).moveTargets.addResourceTarget(this, moveTarget);
    }
    _buildMovedBlockByTarget(movedTarget) {
        const { moveTarget, index } = movedTarget;
        const resourceToMoveTo = this._getResourceTarget(moveTarget);
        if (this.terraformResourceType !== resourceToMoveTo.terraformResourceType) {
            throw (0, errors_1.movedToResourceOfDifferentType)(moveTarget, this.terraformResourceType, resourceToMoveTo.terraformResourceType);
        }
        const to = index
            ? typeof index === "string"
                ? `${this.terraformResourceType}.${resourceToMoveTo.friendlyUniqueId}["${index}"]`
                : `${this.terraformResourceType}.${resourceToMoveTo.friendlyUniqueId}[${index}]`
            : `${this.terraformResourceType}.${resourceToMoveTo.friendlyUniqueId}`;
        const from = `${this.terraformResourceType}.${this.friendlyUniqueId}`;
        return { to, from };
    }
    _buildMovedBlock() {
        if (this._movedByTarget && this._movedById) {
            throw (0, errors_1.resourceGivenTwoMoveOperationsByTargetAndId)(this.node.id, this._movedByTarget.moveTarget, { to: this._movedById.to, from: this._movedById.from });
        }
        else if (this._movedByTarget) {
            const movedBlockByTarget = this._buildMovedBlockByTarget(this._movedByTarget);
            return { to: movedBlockByTarget.to, from: movedBlockByTarget.from };
        }
        else if (this._movedById) {
            return { to: this._movedById.to, from: this._movedById.from };
        }
        else {
            return undefined;
        }
    }
    /**
     * Moves this resource to the target resource given by moveTarget.
     * @param moveTarget The previously set user defined string set by .addMoveTarget() corresponding to the resource to move to.
     * @param index Optional The index corresponding to the key the resource is to appear in the foreach of a resource to move to
     */
    moveTo(moveTarget, index) {
        if (this._movedByTarget) {
            throw (0, errors_1.resourceGivenTwoMoveOperationsByTarget)(this.friendlyUniqueId, this._movedByTarget.moveTarget, moveTarget);
        }
        this._movedByTarget = { moveTarget, index };
        this._hasMoved = true;
    }
    /**
     * Adds a user defined moveTarget string to this resource to be later used in .moveTo(moveTarget) to resolve the location of the move.
     * @param moveTarget The string move target that will correspond to this resource
     */
    addMoveTarget(moveTarget) {
        this._addResourceTarget(moveTarget);
    }
    /**
     * Moves this resource to the resource corresponding to "id"
     * @param id Full id of resource to move to, e.g. "aws_s3_bucket.example"
     */
    moveToId(id) {
        if (this._movedById) {
            throw (0, errors_1.resourceGivenTwoMoveOperationsById)(this.node.id, { to: this._movedById.from, from: this._movedById.to }, {
                to: id,
                from: `${this.terraformResourceType}.${this.friendlyUniqueId}`,
            });
        }
        this._movedById = {
            to: id,
            from: `${this.terraformResourceType}.${this.friendlyUniqueId}`,
        };
        this._hasMoved = true;
    }
    /**
     * Move the resource corresponding to "id" to this resource. Note that the resource being moved from must be marked as moved using it's instance function.
     * @param id Full id of resource being moved from, e.g. "aws_s3_bucket.example"
     */
    moveFromId(id) {
        if (this._movedById) {
            throw (0, errors_1.resourceGivenTwoMoveOperationsById)(this.node.id, { to: this._movedById.from, from: this._movedById.to }, {
                to: id,
                from: `${this.terraformResourceType}.${this.friendlyUniqueId}`,
            });
        }
        this._movedById = {
            to: `${this.terraformResourceType}.${this.friendlyUniqueId}`,
            from: id,
        };
    }
}
exports.TerraformResource = TerraformResource;
_a = JSII_RTTI_SYMBOL_1;
TerraformResource[_a] = { fqn: "cdktf.TerraformResource", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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