"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformModule = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const terraform_element_1 = require("./terraform-element");
const terraform_provider_1 = require("./terraform-provider");
const util_1 = require("./util");
const tokens_1 = require("./tokens");
const tfExpression_1 = require("./tfExpression");
const errors_1 = require("./errors");
const terraform_module_asset_1 = require("./terraform-module-asset");
/**
 * TerraformModule can be used to reference a local terraform module or a module from the Terraform Registry.
 * It should be used if you can not use generated bindings for the module as you would get by adding the module
 * to your cdktf.json files "terraformModules" array and running cdktf get.
 *
 * This class is not creating a Terraform module to be used outside of CDKTF.
 * If you want to bundle certain resources together like you would do with a Terraform module,
 * you should use Constructs instead, see http://cdk.tf/constructs for more details.
 */
class TerraformModule extends terraform_element_1.TerraformElement {
    constructor(scope, id, options) {
        super(scope, id, "module");
        this.source = options.source;
        if (!options.skipAssetCreationFromLocalModules) {
            if (options.source.startsWith("./") || options.source.startsWith("../")) {
                // Create an asset or reuse existing "singleton asset" for the local module for better TFC support
                this.source = terraform_module_asset_1.TerraformModuleAsset.of(scope).getAssetPathForModule(options.source);
            }
        }
        this.version = options.version;
        this._providers = options.providers;
        this.validateIfProvidersHaveUniqueKeys();
        if (Array.isArray(options.dependsOn)) {
            this.dependsOn = options.dependsOn.map((dependency) => (0, tfExpression_1.dependable)(dependency));
        }
        this.forEach = options.forEach;
    }
    // Adds synth-time validations
    onSynth() {
        // We don't allow any nested constructs within TerraformModules, it's most likely a mistake
        // where constructs should be used instead.
        if (this.node.children.length > 0) {
            throw (0, errors_1.terraformModuleHasChildren)(this.node.path);
        }
    }
    // jsii can't handle abstract classes?
    synthesizeAttributes() {
        return {};
    }
    // jsii can't handle abstract classes?
    synthesizeHclAttributes() {
        return {};
    }
    interpolationForOutput(moduleOutput) {
        return (0, tfExpression_1.ref)(`module.${this.friendlyUniqueId}${this.forEach ? ".*" : ""}.${moduleOutput}`, this.cdktfStack);
    }
    getString(output) {
        return tokens_1.Token.asString(this.interpolationForOutput(output));
    }
    get providers() {
        return this._providers;
    }
    addProvider(provider) {
        if (!this._providers) {
            this._providers = [];
        }
        this._providers.push(provider);
        this.validateIfProvidersHaveUniqueKeys();
    }
    toHclTerraform() {
        this.onSynth();
        const attributes = (0, util_1.deepMerge)({
            ...this.synthesizeHclAttributes(),
            source: {
                value: this.source,
                type: "simple",
                isBlock: false,
                storageClassType: "string",
            },
            version: this.version
                ? {
                    value: this.version,
                    type: "simple",
                    isBlock: false,
                    storageClassType: "string",
                }
                : undefined,
            providers: this._providers?.reduce((a, p) => {
                if (terraform_provider_1.TerraformProvider.isTerraformProvider(p)) {
                    return { ...a, [p.terraformResourceType]: p.fqn };
                }
                else {
                    return {
                        ...a,
                        [`${p.provider.terraformResourceType}.${p.moduleAlias}`]: p.provider.fqn,
                    };
                }
            }, {}),
            depends_on: this.dependsOn
                ? {
                    value: this.dependsOn,
                    type: "list",
                    isBlock: false,
                    storageClassType: "string",
                }
                : undefined,
            for_each: this.forEach?._getForEachExpression(),
        }, this.rawOverrides);
        attributes["//"] = this.constructNodeMetadata;
        return {
            module: {
                [this.friendlyUniqueId]: attributes,
            },
        };
    }
    toTerraform() {
        this.onSynth();
        const attributes = (0, util_1.deepMerge)({
            ...this.synthesizeAttributes(),
            source: this.source,
            version: this.version,
            providers: this._providers?.reduce((a, p) => {
                if (terraform_provider_1.TerraformProvider.isTerraformProvider(p)) {
                    return { ...a, [p.terraformResourceType]: p.fqn };
                }
                else {
                    return {
                        ...a,
                        [`${p.provider.terraformResourceType}.${p.moduleAlias}`]: p.provider.fqn,
                    };
                }
            }, {}),
            depends_on: this.dependsOn,
            for_each: this.forEach?._getForEachExpression(),
        }, this.rawOverrides);
        attributes["//"] = this.constructNodeMetadata;
        return {
            module: {
                [this.friendlyUniqueId]: attributes,
            },
        };
    }
    toMetadata() {
        if (!Object.keys(this.rawOverrides).length) {
            return {};
        }
        return {
            overrides: {
                [`module.${this.source}`]: Object.keys(this.rawOverrides),
            },
        };
    }
    validateIfProvidersHaveUniqueKeys() {
        const moduleAliases = this._providers?.map((p) => {
            if (terraform_provider_1.TerraformProvider.isTerraformProvider(p)) {
                return p.terraformResourceType;
            }
            else {
                return `${p.provider.terraformResourceType}.${p.moduleAlias}`;
            }
        });
        const uniqueModuleAliases = new Set();
        moduleAliases?.forEach((alias) => {
            if (uniqueModuleAliases.has(alias)) {
                throw (0, errors_1.modulesWithSameAlias)(alias);
            }
            uniqueModuleAliases.add(alias);
        });
    }
}
exports.TerraformModule = TerraformModule;
_a = JSII_RTTI_SYMBOL_1;
TerraformModule[_a] = { fqn: "cdktf.TerraformModule", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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