"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformModuleAsset = void 0;
exports.findLowestCommonPath = findLowestCommonPath;
const constructs_1 = require("constructs");
const path = require("path");
const os = require("os");
const fs = require("fs");
const terraform_stack_1 = require("./terraform-stack");
const terraform_asset_1 = require("./terraform-asset");
const fs_1 = require("./private/fs");
const TERRAFORM_MODULE_ASSET_SYMBOL = Symbol.for("cdktf.TerraformModuleAsset");
/**
 * This Construct is being created as a singleton whenever the first
 * TerraformModule with a local source is created in a stack.
 *
 * When Terraform modules depend on local modules, the local modules
 * they depend on need to have the correct relative path to the module
 * in the asset. This only works if all modules are in one asset.
 *
 * (We could theoretically detect if a module uses relative path references, but this is easier)
 * @internal
 */
class TerraformModuleAsset extends constructs_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        const relativeModules = this.node.tryGetContext("cdktfRelativeModules");
        const staticModuleAssetHash = this.node.tryGetContext("cdktfStaticModuleAssetHash");
        if (!relativeModules) {
            throw new Error("You are trying to use a local module with a relative path, but the cdktfRelativeModules context is not set. It is expected to be an array of strings containing the relative paths to the relative modules your app is using. You either need to supply it in the Apps constructor via the context option or invoke the synthesis through the CLI. We need this information so that assets with relative paths are properly handled when used with assets, so you can also set the skipAssetCreationFromLocalModules to true on your relative modules.");
        }
        const moduleSources = relativeModules.map((module) => typeof module === "string" ? module : module.source);
        const relativeAssetPath = findLowestCommonPath(moduleSources);
        if (!relativeAssetPath) {
            throw new Error("Could not find lowest common path for relative modules. This should not happen, you might be overwriting the cdktfRelativeModules value of the context with something unexpected. We expect an array of string or objects with a source attribute where the string or the source attribute reference a relative path to a module. The context can be set in the context option of the App Constructor Options.");
        }
        this.relativeAssetPath = relativeAssetPath;
        // Create a tmp dir for the asset
        const tmpDir = fs.mkdtempSync(path.join(os.tmpdir(), "cdktf-module-asset-"));
        // Copy all modules into the tmp dir
        for (const module of moduleSources) {
            const target = path.join(tmpDir, this.relativeModulePath(module));
            copySync(module, target);
        }
        // Create asset based on tmp dir
        this.asset = new terraform_asset_1.TerraformAsset(this, "asset", {
            path: tmpDir,
            type: terraform_asset_1.AssetType.DIRECTORY,
            assetHash: staticModuleAssetHash ?? (0, fs_1.hashPath)(relativeAssetPath),
        });
    }
    static of(construct) {
        const stack = terraform_stack_1.TerraformStack.of(construct);
        let asset = stack[TERRAFORM_MODULE_ASSET_SYMBOL];
        if (!asset) {
            asset = new TerraformModuleAsset(stack, "__cdktf_module_asset");
            Object.defineProperty(stack, TERRAFORM_MODULE_ASSET_SYMBOL, {
                value: asset,
                configurable: false,
                enumerable: false,
            });
        }
        return asset;
    }
    /**
     * The input source is relative to cwd, but we want the value relative
     * to the assets target, so the common path prefix
     */
    relativeModulePath(source) {
        const absoluteSourcePath = path.resolve(source);
        const absoluteAssetPath = path.resolve(this.relativeAssetPath);
        return path.relative(absoluteAssetPath, absoluteSourcePath);
    }
    getAssetPathForModule(source) {
        return `./${path.join(this.asset.path, this.relativeModulePath(source))}`;
    }
}
exports.TerraformModuleAsset = TerraformModuleAsset;
/**
 * Finds the lowest common path of all relaticve paths in the given array
 */
function findLowestCommonPath(paths) {
    if (paths.length === 0) {
        return undefined;
    }
    // We first need to make the paths absolute so that we can compare them
    const absolutePaths = paths.map((p) => path.resolve(p));
    // We take the first path as the base and then remove parts from the end until we find the lowest common path
    let absolutePathPrefix = absolutePaths[0];
    while (!absolutePaths.every((p) => p.startsWith(absolutePathPrefix)) &&
        absolutePathPrefix !== "/") {
        absolutePathPrefix = path.dirname(absolutePathPrefix);
    }
    const relativePath = path.relative(process.cwd(), absolutePathPrefix);
    return relativePath === "" ? "." : relativePath;
}
/**
 * Copies a file or directory recursively
 * @param from
 * @param to
 */
function copySync(from, to) {
    if (fs.lstatSync(from).isDirectory()) {
        fs.mkdirSync(to, { recursive: true });
        for (const file of fs.readdirSync(from)) {
            copySync(path.join(from, file), path.join(to, file));
        }
    }
    else {
        fs.copyFileSync(from, to);
    }
}
//# sourceMappingURL=data:application/json;base64,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