"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamicListTerraformIterator = exports.ResourceTerraformIterator = exports.MapTerraformIterator = exports.ListTerraformIterator = exports.TerraformIterator = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const errors_1 = require("./errors");
const terraform_dynamic_expression_1 = require("./terraform-dynamic-expression");
const terraform_functions_1 = require("./terraform-functions");
const tfExpression_1 = require("./tfExpression");
const tokens_1 = require("./tokens");
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformIterator {
    /**
     * Creates a new iterator from a list
     */
    static fromList(list) {
        // TODO: this could return different iterators depending on the type of the list
        // for example it could return a NumberListIterator whose iterator.key would be a number
        return new ListTerraformIterator(list);
    }
    /**
     * Creates a new iterator from a complex list. One example for this would be a list of maps.
     * The list will be converted into a map with the mapKeyAttributeName as the key.
     * @param list the list to iterate over
     * @param mapKeyAttributeName the name of the attribute that should be used as the key in the map
     *
     * Visit https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-iterators-on-complex-lists for more information.
     *
     * @example
     * const cert = new AcmCertificate(this, "cert", {
     *    domainName: "example.com",
     *    validationMethod: "DNS",
     *  });
     *
     * const dvoIterator = TerraformIterator.fromComplexList(
     *   cert.domainValidationOptions,
     *   "domain_name"
     * );
     *
     * new Route53Record(this, "record", {
     *   allowOverwrite: true,
     *   name: dvoIterator.getString("name"),
     *   records: [dvoIterator.getString("record")],
     *   ttl: 60,
     *   type: dvoIterator.getString("type"),
     *   zoneId: Token.asString(dataAwsRoute53ZoneExample.zoneId),
     *   forEach: dvoIterator,
     * });
     */
    static fromComplexList(list, mapKeyAttributeName) {
        return new DynamicListTerraformIterator(list, mapKeyAttributeName);
    }
    /**
     * Creates a new iterator from a map
     */
    static fromMap(map) {
        return new MapTerraformIterator(map);
    }
    /**
     * Creates a new iterator from a resource that
     * has been created with the `for_each` argument.
     */
    static fromResources(resource) {
        return new ResourceTerraformIterator(resource);
    }
    /**
     * Creates a new iterator from a data source that
     * has been created with the `for_each` argument.
     */
    static fromDataSources(resource) {
        return new ResourceTerraformIterator(resource);
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a string
     */
    getString(attribute) {
        return tokens_1.Token.asString((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a number
     */
    getNumber(attribute) {
        return tokens_1.Token.asNumber((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a boolean
     */
    getBoolean(attribute) {
        return tokens_1.Token.asAny((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as any
     */
    getAny(attribute) {
        return tokens_1.Token.asAny((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a (string) list
     */
    getList(attribute) {
        return tokens_1.Token.asList((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a number list
     */
    getNumberList(attribute) {
        return tokens_1.Token.asNumberList((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map
     */
    getMap(attribute) {
        return tokens_1.Token.asAnyMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of strings
     */
    getStringMap(attribute) {
        return tokens_1.Token.asStringMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of numbers
     */
    getNumberMap(attribute) {
        return tokens_1.Token.asNumberMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of booleans
     */
    getBooleanMap(attribute) {
        return tokens_1.Token.asBooleanMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of any
     */
    getAnyMap(attribute) {
        return tokens_1.Token.asAnyMap((0, tfExpression_1.propertyAccess)(this._getValue(), [attribute]));
    }
    /**
     * @internal
     */
    _getValue() {
        // uses a Lazy value to be able to access the current TerraformStack and pass it to ref()
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                switch (context.iteratorContext) {
                    case "DYNAMIC_BLOCK":
                        return (0, tfExpression_1.ref)("each.value");
                    case "FOR_EXPRESSION":
                        return tfExpression_1.FOR_EXPRESSION_VALUE;
                    default:
                        // same as dynamic block, as this is the case when a iterator is passed to the root level of e.g. a resource
                        return (0, tfExpression_1.ref)("each.value");
                }
            },
        }, { displayHint: "<iterator value>" });
    }
    /**
     * @internal
     */
    _getKey() {
        // uses a Lazy value to be able to access the current TerraformStack and pass it to ref()
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                switch (context.iteratorContext) {
                    case "DYNAMIC_BLOCK":
                        return (0, tfExpression_1.ref)("each.key");
                    case "FOR_EXPRESSION":
                        return tfExpression_1.FOR_EXPRESSION_KEY;
                    default:
                        // same as dynamic block, as this is the case when a iterator is passed to the root level of e.g. a resource
                        return (0, tfExpression_1.ref)("each.key");
                }
            },
        }, { displayHint: "<iterator key>" });
    }
    /**
     * Creates a dynamic expression that can be used to loop over this iterator
     * in a dynamic block.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * See https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-iterators-for-list-attributes
     */
    dynamic(attributes) {
        return tokens_1.Token.asAny(new terraform_dynamic_expression_1.TerraformDynamicExpression({
            iterator: this,
            content: attributes,
        }));
    }
    /**
     * Creates a for expression that maps the iterators to its keys.
     * For lists these would be the indices, for maps the keys.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     */
    keys() {
        return tokens_1.Token.asAny((0, tfExpression_1.forExpression)(this._getForEachExpression(), tfExpression_1.FOR_EXPRESSION_KEY));
    }
    /**
     * Creates a for expression that maps the iterators to its value in case it is a map.
     * For lists these would stay the same.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     */
    values() {
        return tokens_1.Token.asAny((0, tfExpression_1.forExpression)(this._getForEachExpression(), tfExpression_1.FOR_EXPRESSION_VALUE));
    }
    /**
     * Creates a for expression that accesses the key on each element of the iterator.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * @param property The property of the iterators values to map to
     */
    pluckProperty(property) {
        return tokens_1.Token.asAny((0, tfExpression_1.forExpression)(this._getForEachExpression(), (0, tfExpression_1.propertyAccess)(tfExpression_1.FOR_EXPRESSION_VALUE, [property])));
    }
    /**
     * Creates a for expression that results in a list.
     * This method allows you to create every possible for expression, but requires more knowledge about
     * Terraform's for expression syntax.
     * For the most common use cases you can use keys(), values(), and pluckProperty() instead.
     *
     * You may write any valid Terraform for each expression, e.g.
     * `TerraformIterator.fromList(myIteratorSourceVar).forExpressionForList("val.foo if val.bar == true")`
     * will result in `[ for key, val in var.myIteratorSource: val.foo if val.bar == true ]`.
     *
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * @param expression The expression to use in the for mapping
     */
    forExpressionForList(expression) {
        return tokens_1.Token.asAny((0, tfExpression_1.forExpression)(this._getForEachExpression(), expression));
    }
    /**
     * Creates a for expression that results in a map.
     * This method allows you to create every possible for expression, but requires more knowledge about
     * Terraforms for expression syntax.
     * For the most common use cases you can use keys(), values(), and pluckProperty instead.
     *
     * You may write any valid Terraform for each expression, e.g.
     * `TerraformIterator.fromMap(myIteratorSourceVar).forExpressionForMap("key", "val.foo if val.bar == true")`
     * will result in `{ for key, val in var.myIteratorSource: key => val.foo if val.bar == true }`.
     *
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * @param keyExpression The expression to use as key in the for mapping
     * @param valueExpression The expression to use as value in the for mapping
     */
    forExpressionForMap(keyExpression, valueExpression) {
        return tokens_1.Token.asAny((0, tfExpression_1.forExpression)(this._getForEachExpression(), valueExpression, keyExpression));
    }
}
exports.TerraformIterator = TerraformIterator;
_a = JSII_RTTI_SYMBOL_1;
TerraformIterator[_a] = { fqn: "cdktf.TerraformIterator", version: "0.21.0" };
// eslint-disable-next-line jsdoc/require-jsdoc
class ListTerraformIterator extends TerraformIterator {
    constructor(list) {
        super();
        this.list = list;
    }
    /**
     * Returns the currently entry in the list or set that is being iterated over.
     * For lists this is the same as `iterator.value`. If you need the index,
     * use count via `TerraformCount`:
     * https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-count
     */
    get key() {
        return this._getKey();
    }
    /**
     * Returns the value of the current item iterated over.
     */
    get value() {
        return this._getValue();
    }
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression() {
        // needs to be wrapped in a set as Terraform only allows sets in for_each
        return terraform_functions_1.Fn.toset(this.list);
    }
}
exports.ListTerraformIterator = ListTerraformIterator;
_b = JSII_RTTI_SYMBOL_1;
ListTerraformIterator[_b] = { fqn: "cdktf.ListTerraformIterator", version: "0.21.0" };
// eslint-disable-next-line jsdoc/require-jsdoc
class MapTerraformIterator extends TerraformIterator {
    constructor(map) {
        super();
        this.map = map;
    }
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression() {
        // explicit wrapping to circumvent "Found an encoded map token in a scalar string context." error
        return tokens_1.Token.asString(this.map);
    }
    /**
     * Returns the key of the current entry in the map that is being iterated over.
     */
    get key() {
        return tokens_1.Token.asString(this._getKey());
    }
    /**
     * Returns the value of the current item iterated over.
     */
    get value() {
        return this._getValue();
    }
}
exports.MapTerraformIterator = MapTerraformIterator;
_c = JSII_RTTI_SYMBOL_1;
MapTerraformIterator[_c] = { fqn: "cdktf.MapTerraformIterator", version: "0.21.0" };
// eslint-disable-next-line jsdoc/require-jsdoc
class ResourceTerraformIterator extends TerraformIterator {
    constructor(element) {
        super();
        this.element = element;
        if (element.count) {
            throw (0, errors_1.iteratorOnResourceWithCount)();
        }
        if (!element.forEach) {
            throw (0, errors_1.iteratorOnResourceWithoutForEach)();
        }
    }
    /**
     * Returns the current entry in the list or set that is being iterated over.
     * For lists this is the same as `iterator.value`. If you need the index,
     * use count via `TerraformCount`:
     * https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-count
     */
    get key() {
        return this._getKey();
    }
    /**
     * Returns the value of the current item iterated over.
     */
    get value() {
        return this._getValue();
    }
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression() {
        return this.element.fqn; // no wrapping necessary for resources
    }
}
exports.ResourceTerraformIterator = ResourceTerraformIterator;
_d = JSII_RTTI_SYMBOL_1;
ResourceTerraformIterator[_d] = { fqn: "cdktf.ResourceTerraformIterator", version: "0.21.0" };
// eslint-disable-next-line jsdoc/require-jsdoc
class DynamicListTerraformIterator extends MapTerraformIterator {
    constructor(list, mapKeyAttributeName) {
        super(list);
        this.list = list;
        this.mapKeyAttributeName = mapKeyAttributeName;
    }
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression() {
        // uses a Lazy value to be able to render a conversion into a map in the context of a TerraformResource
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                switch (context.iteratorContext) {
                    case "FOR_EXPRESSION":
                        return this.list;
                    case "DYNAMIC_BLOCK": // fallthrough
                    default: // same as dynamic block, as this is the case when a iterator is passed to the root level of e.g. a resource
                        // Turn list into a map
                        // { for k,v in <input> : <keyExpression> => <valueExpression>}
                        return (0, tfExpression_1.forExpression)(this.list, // input
                        tfExpression_1.FOR_EXPRESSION_VALUE, // valueExpression
                        terraform_functions_1.Fn.lookupNested(tfExpression_1.FOR_EXPRESSION_VALUE, [
                            this.mapKeyAttributeName,
                        ]));
                }
            },
        }, { displayHint: "<iterator value>" });
    }
}
exports.DynamicListTerraformIterator = DynamicListTerraformIterator;
_e = JSII_RTTI_SYMBOL_1;
DynamicListTerraformIterator[_e] = { fqn: "cdktf.DynamicListTerraformIterator", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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