import { AnyMap, AnyMapList, BooleanMap, BooleanMapList, ComplexList, ComplexMap, NumberMap, NumberMapList, StringMap, StringMapList } from "./complex-computed-list";
import { ITerraformResource } from "./terraform-resource";
import { IResolvable } from "./tokens";
export interface ITerraformIterator {
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression(): any;
}
type ListType = Array<string> | Array<number> | Array<boolean | IResolvable> | IResolvable;
type ComplexListType = ComplexList | StringMapList | NumberMapList | BooleanMapList | AnyMapList | IResolvable;
type MapType = {
    [key: string]: any;
} | {
    [key: string]: string;
} | {
    [key: string]: number;
} | StringMap | NumberMap | BooleanMap | AnyMap | ComplexMap;
export declare abstract class TerraformIterator implements ITerraformIterator {
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    abstract _getForEachExpression(): any;
    /**
     * Creates a new iterator from a list
     */
    static fromList(list: ListType): ListTerraformIterator;
    /**
     * Creates a new iterator from a complex list. One example for this would be a list of maps.
     * The list will be converted into a map with the mapKeyAttributeName as the key.
     * @param list the list to iterate over
     * @param mapKeyAttributeName the name of the attribute that should be used as the key in the map
     *
     * Visit https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-iterators-on-complex-lists for more information.
     *
     * @example
     * const cert = new AcmCertificate(this, "cert", {
     *    domainName: "example.com",
     *    validationMethod: "DNS",
     *  });
     *
     * const dvoIterator = TerraformIterator.fromComplexList(
     *   cert.domainValidationOptions,
     *   "domain_name"
     * );
     *
     * new Route53Record(this, "record", {
     *   allowOverwrite: true,
     *   name: dvoIterator.getString("name"),
     *   records: [dvoIterator.getString("record")],
     *   ttl: 60,
     *   type: dvoIterator.getString("type"),
     *   zoneId: Token.asString(dataAwsRoute53ZoneExample.zoneId),
     *   forEach: dvoIterator,
     * });
     */
    static fromComplexList(list: ComplexListType, mapKeyAttributeName: string): DynamicListTerraformIterator;
    /**
     * Creates a new iterator from a map
     */
    static fromMap(map: ComplexMap | {
        [key: string]: any;
    } | {
        [key: string]: string;
    } | {
        [key: string]: number;
    } | {
        [key: string]: boolean;
    }): MapTerraformIterator;
    /**
     * Creates a new iterator from a resource that
     * has been created with the `for_each` argument.
     */
    static fromResources(resource: ITerraformResource): ResourceTerraformIterator;
    /**
     * Creates a new iterator from a data source that
     * has been created with the `for_each` argument.
     */
    static fromDataSources(resource: ITerraformResource): ResourceTerraformIterator;
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a string
     */
    getString(attribute: string): string;
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a number
     */
    getNumber(attribute: string): number;
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a boolean
     */
    getBoolean(attribute: string): IResolvable;
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as any
     */
    getAny(attribute: string): IResolvable;
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a (string) list
     */
    getList(attribute: string): string[];
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a number list
     */
    getNumberList(attribute: string): number[];
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map
     */
    getMap(attribute: string): {
        [key: string]: any;
    };
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of strings
     */
    getStringMap(attribute: string): {
        [key: string]: string;
    };
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of numbers
     */
    getNumberMap(attribute: string): {
        [key: string]: number;
    };
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of booleans
     */
    getBooleanMap(attribute: string): {
        [key: string]: boolean;
    };
    /**
     * @param attribute name of the property to retrieve
     * @returns the given attribute of the current item iterated over as a map of any
     */
    getAnyMap(attribute: string): {
        [key: string]: any;
    };
    /**
     * @internal
     */
    protected _getValue(): any;
    /**
     * @internal
     */
    protected _getKey(): any;
    /**
     * Creates a dynamic expression that can be used to loop over this iterator
     * in a dynamic block.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * See https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-iterators-for-list-attributes
     */
    dynamic(attributes: {
        [key: string]: any;
    }): IResolvable;
    /**
     * Creates a for expression that maps the iterators to its keys.
     * For lists these would be the indices, for maps the keys.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     */
    keys(): IResolvable;
    /**
     * Creates a for expression that maps the iterators to its value in case it is a map.
     * For lists these would stay the same.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     */
    values(): IResolvable;
    /**
     * Creates a for expression that accesses the key on each element of the iterator.
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * @param property The property of the iterators values to map to
     */
    pluckProperty(property: string): IResolvable;
    /**
     * Creates a for expression that results in a list.
     * This method allows you to create every possible for expression, but requires more knowledge about
     * Terraform's for expression syntax.
     * For the most common use cases you can use keys(), values(), and pluckProperty() instead.
     *
     * You may write any valid Terraform for each expression, e.g.
     * `TerraformIterator.fromList(myIteratorSourceVar).forExpressionForList("val.foo if val.bar == true")`
     * will result in `[ for key, val in var.myIteratorSource: val.foo if val.bar == true ]`.
     *
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * @param expression The expression to use in the for mapping
     */
    forExpressionForList(expression: string | IResolvable): IResolvable;
    /**
     * Creates a for expression that results in a map.
     * This method allows you to create every possible for expression, but requires more knowledge about
     * Terraforms for expression syntax.
     * For the most common use cases you can use keys(), values(), and pluckProperty instead.
     *
     * You may write any valid Terraform for each expression, e.g.
     * `TerraformIterator.fromMap(myIteratorSourceVar).forExpressionForMap("key", "val.foo if val.bar == true")`
     * will result in `{ for key, val in var.myIteratorSource: key => val.foo if val.bar == true }`.
     *
     * As this returns an IResolvable you might need to wrap the output in
     * a Token, e.g. `Token.asString`.
     * @param keyExpression The expression to use as key in the for mapping
     * @param valueExpression The expression to use as value in the for mapping
     */
    forExpressionForMap(keyExpression: string | IResolvable, valueExpression: string | IResolvable): IResolvable;
}
export declare class ListTerraformIterator extends TerraformIterator {
    private readonly list;
    constructor(list: ListType);
    /**
     * Returns the currently entry in the list or set that is being iterated over.
     * For lists this is the same as `iterator.value`. If you need the index,
     * use count via `TerraformCount`:
     * https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-count
     */
    get key(): any;
    /**
     * Returns the value of the current item iterated over.
     */
    get value(): any;
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression(): any;
}
export declare class MapTerraformIterator extends TerraformIterator {
    private readonly map;
    constructor(map: MapType);
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression(): any;
    /**
     * Returns the key of the current entry in the map that is being iterated over.
     */
    get key(): string;
    /**
     * Returns the value of the current item iterated over.
     */
    get value(): any;
}
export declare class ResourceTerraformIterator extends TerraformIterator {
    private readonly element;
    constructor(element: ITerraformResource);
    /**
     * Returns the current entry in the list or set that is being iterated over.
     * For lists this is the same as `iterator.value`. If you need the index,
     * use count via `TerraformCount`:
     * https://developer.hashicorp.com/terraform/cdktf/concepts/iterators#using-count
     */
    get key(): any;
    /**
     * Returns the value of the current item iterated over.
     */
    get value(): any;
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression(): any;
}
export declare class DynamicListTerraformIterator extends MapTerraformIterator {
    private readonly list;
    private readonly mapKeyAttributeName;
    constructor(list: ListType, mapKeyAttributeName: string);
    /**
     * @internal used by TerraformResource to set the for_each expression
     */
    _getForEachExpression(): any;
}
export {};
