"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformDynamicExpression = void 0;
const _1 = require(".");
const tokens_1 = require("./tokens");
const stack_trace_1 = require("./tokens/private/stack-trace");
const DYNAMIC_EXPRESSION_SYMBOL = Symbol.for("cdktf/TerraformDynamicExpression");
/**
 * A TerraformDynamicExpression is returned by TerraformIterator.dynamic
 * and can be passed to list attributes of e.g. TerraformResources
 * As it resolves to a for expression, it can be used directly on attributes of
 * a list(object) type (or similar) and in Terraform functions that take a list.
 * When used in attributes that are of a block type which need to be modeled as
 * a dynamic block, the listMapper for that attribute will change this to a
 * TerraformDynamicBlock, which in turn will be properly handled by the
 * processDynamicAttributes() utility function.
 */
class TerraformDynamicExpression {
    constructor(args) {
        Object.defineProperty(this, DYNAMIC_EXPRESSION_SYMBOL, { value: true });
        this.creationStack = (0, stack_trace_1.captureStackTrace)();
        this.iterator = args.iterator;
        this.content = args.content;
    }
    /**
     * resolves this as a Terraform for expression.
     * This allows using iterator.dynamic() on attributes that are not blocks but
     * lists of objects (which is not visible to the user when working with provider
     * mappings as it is abstracted away)
     */
    resolve() {
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                // resolve nested Lazy values (within this.content) for a for expression
                context.iteratorContext = "FOR_EXPRESSION";
                // returns [ for key, value in forEachExpr : content ]
                // context.resolve is required for the iteratorContext to be correctly passed
                // to Lazy values within this.content
                return context.resolve((0, _1.forExpression)(this.iterator._getForEachExpression(), this.content));
            },
        });
    }
    toString() {
        return tokens_1.Token.asString(this);
    }
    /**
     * Turn this Dynamic Expression into JSON
     *
     * Called automatically when JSON.stringify() is called on this.
     */
    toJSON() {
        return "<unresolved-terraform-dynamic-expression>";
    }
    static isTerraformDynamicExpression(x) {
        return (x !== null && typeof x === "object" && DYNAMIC_EXPRESSION_SYMBOL in x);
    }
}
exports.TerraformDynamicExpression = TerraformDynamicExpression;
//# sourceMappingURL=data:application/json;base64,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