"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformDynamicBlock = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const errors_1 = require("./errors");
const tokens_1 = require("./tokens");
const stack_trace_1 = require("./tokens/private/stack-trace");
const DYNAMIC_BLOCK_SYMBOL = Symbol.for("cdktf/TerraformDynamicBlock");
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformDynamicBlock {
    // labels: TODO: support labels, but they seem to be quite rare (issue: https://github.com/hashicorp/terraform-cdk/issues/1939)
    constructor(args) {
        this.iterator = "each"; // name of temporary variable used in iteration, needed in the future when allowing nesting iterators
        Object.defineProperty(this, DYNAMIC_BLOCK_SYMBOL, { value: true });
        this.creationStack = (0, stack_trace_1.captureStackTrace)();
        this.forEach = args.forEach;
        this.content = args.content;
    }
    static fromDynamicExpression(expr) {
        return new TerraformDynamicBlock({
            forEach: expr.iterator,
            content: expr.content,
        });
    }
    resolve() {
        throw (0, errors_1.dynamicBlockNotSupported)(this.forEach._getForEachExpression());
    }
    toTerraformDynamicBlockJson() {
        return tokens_1.Lazy.anyValue({
            produce: (context) => {
                context.iteratorContext = "DYNAMIC_BLOCK"; // resolve nested Lazy values (within this.content) as dynamic block expressions
                return {
                    for_each: this.forEach._getForEachExpression(),
                    iterator: this.iterator,
                    content: context.resolve(this.content),
                };
            },
        });
    }
    toString() {
        return tokens_1.Token.asString(this);
    }
    /**
     * Turn this Dynamic Block into JSON
     *
     * Called automatically when JSON.stringify() is called on this.
     */
    toJSON() {
        return "<unresolved-terraform-dynamic-block>";
    }
    static isTerraformDynamicBlock(x) {
        return x !== null && typeof x === "object" && DYNAMIC_BLOCK_SYMBOL in x;
    }
}
exports.TerraformDynamicBlock = TerraformDynamicBlock;
//# sourceMappingURL=data:application/json;base64,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