"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformAsset = exports.AssetType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const constructs_1 = require("constructs");
const fs = require("fs");
const path = require("path");
const fs_1 = require("./private/fs");
const synthesizer_1 = require("./synthesize/synthesizer");
const terraform_stack_1 = require("./terraform-stack");
const errors_1 = require("./errors");
var AssetType;
(function (AssetType) {
    AssetType[AssetType["FILE"] = 0] = "FILE";
    AssetType[AssetType["DIRECTORY"] = 1] = "DIRECTORY";
    AssetType[AssetType["ARCHIVE"] = 2] = "ARCHIVE";
})(AssetType || (exports.AssetType = AssetType = {}));
const ARCHIVE_NAME = "archive.zip";
const ASSETS_DIRECTORY = "assets";
// eslint-disable-next-line jsdoc/require-jsdoc
class TerraformAsset extends constructs_1.Construct {
    /**
     * A Terraform Asset takes a file or directory outside of the CDK for Terraform context and moves it into it.
     * Assets copy referenced files into the stacks context for further usage in other resources.
     * @param scope
     * @param id
     * @param config
     */
    constructor(scope, id, config) {
        super(scope, id);
        this.stack = terraform_stack_1.TerraformStack.of(this);
        if (path.isAbsolute(config.path)) {
            this.sourcePath = config.path;
        }
        else {
            const cdktfJsonPath = scope.node.tryGetContext("cdktfJsonPath") ??
                (0, fs_1.findFileAboveCwd)("cdktf.json");
            if (cdktfJsonPath) {
                // Relative paths are always considered to be relative to cdktf.json, but operations are performed relative to process.cwd
                const absolutePath = path.resolve(path.dirname(cdktfJsonPath), config.path);
                this.sourcePath = path.relative(process.cwd(), absolutePath);
            }
            else {
                throw (0, errors_1.assetOutOfScopeOfCDKTFJson)(id, config.path);
            }
        }
        const stat = fs.statSync(this.sourcePath);
        const inferredType = stat.isFile() ? AssetType.FILE : AssetType.DIRECTORY;
        this.type = config.type ?? inferredType;
        this.assetHash = config.assetHash || (0, fs_1.hashPath)(this.sourcePath);
        if (stat.isFile() && this.type !== AssetType.FILE) {
            throw (0, errors_1.assetExpectsDirectory)(id, config.path);
        }
        if (!stat.isFile() && this.type === AssetType.FILE) {
            throw (0, errors_1.assetExpectsDirectory)(id, config.path);
        }
        (0, synthesizer_1.addCustomSynthesis)(this, {
            onSynthesize: this._onSynthesize.bind(this),
        });
    }
    get namedFolder() {
        return path.posix.join(ASSETS_DIRECTORY, this.stack.getLogicalId(this.node));
    }
    /**
     * The path relative to the root of the terraform directory in posix format
     * Use this property to reference the asset
     */
    get path() {
        return path.posix.join(this.namedFolder, // readable name
        this.assetHash, // hash depending on content so that path changes if content changes
        this.type === AssetType.DIRECTORY ? "" : this.fileName);
    }
    /**
     * Name of the asset
     */
    get fileName() {
        switch (this.type) {
            case AssetType.ARCHIVE:
                return ARCHIVE_NAME;
            default:
                return path.basename(this.sourcePath);
        }
    }
    _onSynthesize(session) {
        const stackManifest = session.manifest.forStack(this.stack);
        const basePath = path.join(session.manifest.outdir, stackManifest.synthesizedStackPath, "..");
        // Cleanup existing assets
        const previousVersionsFolder = path.join(basePath, this.namedFolder);
        if (fs.existsSync(previousVersionsFolder)) {
            fs.rmSync(previousVersionsFolder, { recursive: true });
        }
        const targetPath = path.join(basePath, this.path);
        if (this.type === AssetType.DIRECTORY) {
            fs.mkdirSync(targetPath, { recursive: true });
        }
        else {
            fs.mkdirSync(path.dirname(targetPath), { recursive: true });
        }
        switch (this.type) {
            case AssetType.FILE:
                fs.copyFileSync(this.sourcePath, targetPath);
                break;
            case AssetType.DIRECTORY:
                (0, fs_1.copySync)(this.sourcePath, targetPath);
                break;
            case AssetType.ARCHIVE:
                (0, fs_1.archiveSync)(this.sourcePath, targetPath);
                break;
            default:
                throw (0, errors_1.assetTypeNotImplemented)();
        }
    }
}
exports.TerraformAsset = TerraformAsset;
_a = JSII_RTTI_SYMBOL_1;
TerraformAsset[_a] = { fqn: "cdktf.TerraformAsset", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVycmFmb3JtLWFzc2V0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVycmFmb3JtLWFzc2V0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsK0JBQStCO0FBQy9CLG1DQUFtQztBQUNuQywyQ0FBdUM7QUFDdkMseUJBQXlCO0FBQ3pCLDZCQUE2QjtBQUM3QixxQ0FLc0I7QUFFdEIsMERBQThEO0FBQzlELHVEQUFtRDtBQUNuRCxxQ0FJa0I7QUFXbEIsSUFBWSxTQUlYO0FBSkQsV0FBWSxTQUFTO0lBQ25CLHlDQUFJLENBQUE7SUFDSixtREFBUyxDQUFBO0lBQ1QsK0NBQU8sQ0FBQTtBQUNULENBQUMsRUFKVyxTQUFTLHlCQUFULFNBQVMsUUFJcEI7QUFFRCxNQUFNLFlBQVksR0FBRyxhQUFhLENBQUM7QUFDbkMsTUFBTSxnQkFBZ0IsR0FBRyxRQUFRLENBQUM7QUFFbEMsK0NBQStDO0FBQy9DLE1BQWEsY0FBZSxTQUFRLHNCQUFTO0lBUTNDOzs7Ozs7T0FNRztJQUNILFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsTUFBNEI7UUFDcEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsS0FBSyxHQUFHLGdDQUFjLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXJDLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUNqQyxJQUFJLENBQUMsVUFBVSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUM7UUFDaEMsQ0FBQzthQUFNLENBQUM7WUFDTixNQUFNLGFBQWEsR0FDakIsS0FBSyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUFDO2dCQUN6QyxJQUFBLHFCQUFnQixFQUFDLFlBQVksQ0FBQyxDQUFDO1lBQ2pDLElBQUksYUFBYSxFQUFFLENBQUM7Z0JBQ2xCLDBIQUEwSDtnQkFDMUgsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDL0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxhQUFhLENBQUMsRUFDM0IsTUFBTSxDQUFDLElBQUksQ0FDWixDQUFDO2dCQUNGLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsWUFBWSxDQUFDLENBQUM7WUFDL0QsQ0FBQztpQkFBTSxDQUFDO2dCQUNOLE1BQU0sSUFBQSxtQ0FBMEIsRUFBQyxFQUFFLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3BELENBQUM7UUFDSCxDQUFDO1FBRUQsTUFBTSxJQUFJLEdBQUcsRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDMUMsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDO1FBQzFFLElBQUksQ0FBQyxJQUFJLEdBQUcsTUFBTSxDQUFDLElBQUksSUFBSSxZQUFZLENBQUM7UUFDeEMsSUFBSSxDQUFDLFNBQVMsR0FBRyxNQUFNLENBQUMsU0FBUyxJQUFJLElBQUEsYUFBUSxFQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUUvRCxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNsRCxNQUFNLElBQUEsOEJBQXFCLEVBQUMsRUFBRSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQyxDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNuRCxNQUFNLElBQUEsOEJBQXFCLEVBQUMsRUFBRSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvQyxDQUFDO1FBRUQsSUFBQSxnQ0FBa0IsRUFBQyxJQUFJLEVBQUU7WUFDdkIsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztTQUM1QyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQsSUFBWSxXQUFXO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQ3BCLGdCQUFnQixFQUNoQixJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQ25DLENBQUM7SUFDSixDQUFDO0lBRUQ7OztPQUdHO0lBQ0gsSUFBVyxJQUFJO1FBQ2IsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FDcEIsSUFBSSxDQUFDLFdBQVcsRUFBRSxnQkFBZ0I7UUFDbEMsSUFBSSxDQUFDLFNBQVMsRUFBRSxvRUFBb0U7UUFDcEYsSUFBSSxDQUFDLElBQUksS0FBSyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQ3ZELENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLFFBQVE7UUFDakIsUUFBUSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDbEIsS0FBSyxTQUFTLENBQUMsT0FBTztnQkFDcEIsT0FBTyxZQUFZLENBQUM7WUFDdEI7Z0JBQ0UsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMxQyxDQUFDO0lBQ0gsQ0FBQztJQUVPLGFBQWEsQ0FBQyxPQUEwQjtRQUM5QyxNQUFNLGFBQWEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDNUQsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDeEIsT0FBTyxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQ3ZCLGFBQWEsQ0FBQyxvQkFBb0IsRUFDbEMsSUFBSSxDQUNMLENBQUM7UUFFRiwwQkFBMEI7UUFDMUIsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDckUsSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsQ0FBQztZQUMxQyxFQUFFLENBQUMsTUFBTSxDQUFDLHNCQUFzQixFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDekQsQ0FBQztRQUVELE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVsRCxJQUFJLElBQUksQ0FBQyxJQUFJLEtBQUssU0FBUyxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBQ3RDLEVBQUUsQ0FBQyxTQUFTLENBQUMsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFDaEQsQ0FBQzthQUFNLENBQUM7WUFDTixFQUFFLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUM5RCxDQUFDO1FBRUQsUUFBUSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDbEIsS0FBSyxTQUFTLENBQUMsSUFBSTtnQkFDakIsRUFBRSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDO2dCQUM3QyxNQUFNO1lBRVIsS0FBSyxTQUFTLENBQUMsU0FBUztnQkFDdEIsSUFBQSxhQUFRLEVBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxVQUFVLENBQUMsQ0FBQztnQkFDdEMsTUFBTTtZQUVSLEtBQUssU0FBUyxDQUFDLE9BQU87Z0JBQ3BCLElBQUEsZ0JBQVcsRUFBQyxJQUFJLENBQUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDO2dCQUN6QyxNQUFNO1lBQ1I7Z0JBQ0UsTUFBTSxJQUFBLGdDQUF1QixHQUFFLENBQUM7UUFDcEMsQ0FBQztJQUNILENBQUM7O0FBNUhILHdDQTZIQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmNcbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNUEwtMi4wXG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tIFwiY29uc3RydWN0c1wiO1xuaW1wb3J0ICogYXMgZnMgZnJvbSBcImZzXCI7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQge1xuICBjb3B5U3luYyxcbiAgYXJjaGl2ZVN5bmMsXG4gIGhhc2hQYXRoLFxuICBmaW5kRmlsZUFib3ZlQ3dkLFxufSBmcm9tIFwiLi9wcml2YXRlL2ZzXCI7XG5pbXBvcnQgeyBJU3ludGhlc2lzU2Vzc2lvbiB9IGZyb20gXCIuL3N5bnRoZXNpemVcIjtcbmltcG9ydCB7IGFkZEN1c3RvbVN5bnRoZXNpcyB9IGZyb20gXCIuL3N5bnRoZXNpemUvc3ludGhlc2l6ZXJcIjtcbmltcG9ydCB7IFRlcnJhZm9ybVN0YWNrIH0gZnJvbSBcIi4vdGVycmFmb3JtLXN0YWNrXCI7XG5pbXBvcnQge1xuICBhc3NldEV4cGVjdHNEaXJlY3RvcnksXG4gIGFzc2V0T3V0T2ZTY29wZU9mQ0RLVEZKc29uLFxuICBhc3NldFR5cGVOb3RJbXBsZW1lbnRlZCxcbn0gZnJvbSBcIi4vZXJyb3JzXCI7XG5cbmV4cG9ydCBpbnRlcmZhY2UgVGVycmFmb3JtQXNzZXRDb25maWcge1xuICAvLyBwYXRoIHRvIHRoZSBmaWxlIG9yIGZvbGRlciBjb25maWd1cmVkLiBJZiByZWxhdGl2ZSwgdGhlIHBhdGggaXMgcmVzb2x2ZWQgZnJvbSB0aGUgbG9jYXRpb24gb2YgY2RrdGYuanNvblxuICByZWFkb25seSBwYXRoOiBzdHJpbmc7XG4gIC8vIGZpbGUgdHlwZSBvZiB0aGUgYXNzZXQsIGVpdGhlciBBc3NldFR5cGUuRklMRSwgQXNzZXRUeXBlLkRJUkVDVE9SWSwgQXNzZXRUeXBlLkFSQ0hJVkVcbiAgcmVhZG9ubHkgdHlwZT86IEFzc2V0VHlwZTtcbiAgLy8gaGFzaCB2YWx1ZSBvZiB0aGUgYXNzZXQsIGlmIHBhc3NlZCB3aWxsIGJlIHVzZWQgYXMgcmV0dXJuZWQgYXNzZXRIYXNoXG4gIHJlYWRvbmx5IGFzc2V0SGFzaD86IHN0cmluZztcbn1cblxuZXhwb3J0IGVudW0gQXNzZXRUeXBlIHtcbiAgRklMRSxcbiAgRElSRUNUT1JZLFxuICBBUkNISVZFLFxufVxuXG5jb25zdCBBUkNISVZFX05BTUUgPSBcImFyY2hpdmUuemlwXCI7XG5jb25zdCBBU1NFVFNfRElSRUNUT1JZID0gXCJhc3NldHNcIjtcblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmV4cG9ydCBjbGFzcyBUZXJyYWZvcm1Bc3NldCBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHByaXZhdGUgc3RhY2s6IFRlcnJhZm9ybVN0YWNrO1xuICBwcml2YXRlIHNvdXJjZVBhdGg6IHN0cmluZztcbiAgLy8gaGFzaCB2YWx1ZSBvZiB0aGUgYXNzZXQgdGhhdCBjYW4gYmUgcGFzc2VkIHRvIGNvbnN1bWluZyBjb25zdHJ1Y3RzIChlLmcuIHRvIG5vdCByZWNyZWF0ZSBhIGxhbWJkYSBmdW5jdGlvbiBpbiBjYXNlIHRoZSB1bmRlcmx5aW5nIGZpbGVzIGRpZCBub3QgY2hhbmdlKVxuICBwdWJsaWMgYXNzZXRIYXNoOiBzdHJpbmc7XG4gIC8vIGZpbGUgdHlwZSBvZiB0aGUgYXNzZXQsIGVpdGhlciBBc3NldFR5cGUuRklMRSwgQXNzZXRUeXBlLkRJUkVDVE9SWSwgQXNzZXRUeXBlLkFSQ0hJVkVcbiAgcHVibGljIHR5cGU6IEFzc2V0VHlwZTtcblxuICAvKipcbiAgICogQSBUZXJyYWZvcm0gQXNzZXQgdGFrZXMgYSBmaWxlIG9yIGRpcmVjdG9yeSBvdXRzaWRlIG9mIHRoZSBDREsgZm9yIFRlcnJhZm9ybSBjb250ZXh0IGFuZCBtb3ZlcyBpdCBpbnRvIGl0LlxuICAgKiBBc3NldHMgY29weSByZWZlcmVuY2VkIGZpbGVzIGludG8gdGhlIHN0YWNrcyBjb250ZXh0IGZvciBmdXJ0aGVyIHVzYWdlIGluIG90aGVyIHJlc291cmNlcy5cbiAgICogQHBhcmFtIHNjb3BlXG4gICAqIEBwYXJhbSBpZFxuICAgKiBAcGFyYW0gY29uZmlnXG4gICAqL1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBjb25maWc6IFRlcnJhZm9ybUFzc2V0Q29uZmlnKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuc3RhY2sgPSBUZXJyYWZvcm1TdGFjay5vZih0aGlzKTtcblxuICAgIGlmIChwYXRoLmlzQWJzb2x1dGUoY29uZmlnLnBhdGgpKSB7XG4gICAgICB0aGlzLnNvdXJjZVBhdGggPSBjb25maWcucGF0aDtcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgY2RrdGZKc29uUGF0aCA9XG4gICAgICAgIHNjb3BlLm5vZGUudHJ5R2V0Q29udGV4dChcImNka3RmSnNvblBhdGhcIikgPz9cbiAgICAgICAgZmluZEZpbGVBYm92ZUN3ZChcImNka3RmLmpzb25cIik7XG4gICAgICBpZiAoY2RrdGZKc29uUGF0aCkge1xuICAgICAgICAvLyBSZWxhdGl2ZSBwYXRocyBhcmUgYWx3YXlzIGNvbnNpZGVyZWQgdG8gYmUgcmVsYXRpdmUgdG8gY2RrdGYuanNvbiwgYnV0IG9wZXJhdGlvbnMgYXJlIHBlcmZvcm1lZCByZWxhdGl2ZSB0byBwcm9jZXNzLmN3ZFxuICAgICAgICBjb25zdCBhYnNvbHV0ZVBhdGggPSBwYXRoLnJlc29sdmUoXG4gICAgICAgICAgcGF0aC5kaXJuYW1lKGNka3RmSnNvblBhdGgpLFxuICAgICAgICAgIGNvbmZpZy5wYXRoLFxuICAgICAgICApO1xuICAgICAgICB0aGlzLnNvdXJjZVBhdGggPSBwYXRoLnJlbGF0aXZlKHByb2Nlc3MuY3dkKCksIGFic29sdXRlUGF0aCk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICB0aHJvdyBhc3NldE91dE9mU2NvcGVPZkNES1RGSnNvbihpZCwgY29uZmlnLnBhdGgpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IHN0YXQgPSBmcy5zdGF0U3luYyh0aGlzLnNvdXJjZVBhdGgpO1xuICAgIGNvbnN0IGluZmVycmVkVHlwZSA9IHN0YXQuaXNGaWxlKCkgPyBBc3NldFR5cGUuRklMRSA6IEFzc2V0VHlwZS5ESVJFQ1RPUlk7XG4gICAgdGhpcy50eXBlID0gY29uZmlnLnR5cGUgPz8gaW5mZXJyZWRUeXBlO1xuICAgIHRoaXMuYXNzZXRIYXNoID0gY29uZmlnLmFzc2V0SGFzaCB8fCBoYXNoUGF0aCh0aGlzLnNvdXJjZVBhdGgpO1xuXG4gICAgaWYgKHN0YXQuaXNGaWxlKCkgJiYgdGhpcy50eXBlICE9PSBBc3NldFR5cGUuRklMRSkge1xuICAgICAgdGhyb3cgYXNzZXRFeHBlY3RzRGlyZWN0b3J5KGlkLCBjb25maWcucGF0aCk7XG4gICAgfVxuXG4gICAgaWYgKCFzdGF0LmlzRmlsZSgpICYmIHRoaXMudHlwZSA9PT0gQXNzZXRUeXBlLkZJTEUpIHtcbiAgICAgIHRocm93IGFzc2V0RXhwZWN0c0RpcmVjdG9yeShpZCwgY29uZmlnLnBhdGgpO1xuICAgIH1cblxuICAgIGFkZEN1c3RvbVN5bnRoZXNpcyh0aGlzLCB7XG4gICAgICBvblN5bnRoZXNpemU6IHRoaXMuX29uU3ludGhlc2l6ZS5iaW5kKHRoaXMpLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXQgbmFtZWRGb2xkZXIoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gcGF0aC5wb3NpeC5qb2luKFxuICAgICAgQVNTRVRTX0RJUkVDVE9SWSxcbiAgICAgIHRoaXMuc3RhY2suZ2V0TG9naWNhbElkKHRoaXMubm9kZSksXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgcGF0aCByZWxhdGl2ZSB0byB0aGUgcm9vdCBvZiB0aGUgdGVycmFmb3JtIGRpcmVjdG9yeSBpbiBwb3NpeCBmb3JtYXRcbiAgICogVXNlIHRoaXMgcHJvcGVydHkgdG8gcmVmZXJlbmNlIHRoZSBhc3NldFxuICAgKi9cbiAgcHVibGljIGdldCBwYXRoKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHBhdGgucG9zaXguam9pbihcbiAgICAgIHRoaXMubmFtZWRGb2xkZXIsIC8vIHJlYWRhYmxlIG5hbWVcbiAgICAgIHRoaXMuYXNzZXRIYXNoLCAvLyBoYXNoIGRlcGVuZGluZyBvbiBjb250ZW50IHNvIHRoYXQgcGF0aCBjaGFuZ2VzIGlmIGNvbnRlbnQgY2hhbmdlc1xuICAgICAgdGhpcy50eXBlID09PSBBc3NldFR5cGUuRElSRUNUT1JZID8gXCJcIiA6IHRoaXMuZmlsZU5hbWUsXG4gICAgKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSBhc3NldFxuICAgKi9cbiAgcHVibGljIGdldCBmaWxlTmFtZSgpOiBzdHJpbmcge1xuICAgIHN3aXRjaCAodGhpcy50eXBlKSB7XG4gICAgICBjYXNlIEFzc2V0VHlwZS5BUkNISVZFOlxuICAgICAgICByZXR1cm4gQVJDSElWRV9OQU1FO1xuICAgICAgZGVmYXVsdDpcbiAgICAgICAgcmV0dXJuIHBhdGguYmFzZW5hbWUodGhpcy5zb3VyY2VQYXRoKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIF9vblN5bnRoZXNpemUoc2Vzc2lvbjogSVN5bnRoZXNpc1Nlc3Npb24pIHtcbiAgICBjb25zdCBzdGFja01hbmlmZXN0ID0gc2Vzc2lvbi5tYW5pZmVzdC5mb3JTdGFjayh0aGlzLnN0YWNrKTtcbiAgICBjb25zdCBiYXNlUGF0aCA9IHBhdGguam9pbihcbiAgICAgIHNlc3Npb24ubWFuaWZlc3Qub3V0ZGlyLFxuICAgICAgc3RhY2tNYW5pZmVzdC5zeW50aGVzaXplZFN0YWNrUGF0aCxcbiAgICAgIFwiLi5cIixcbiAgICApO1xuXG4gICAgLy8gQ2xlYW51cCBleGlzdGluZyBhc3NldHNcbiAgICBjb25zdCBwcmV2aW91c1ZlcnNpb25zRm9sZGVyID0gcGF0aC5qb2luKGJhc2VQYXRoLCB0aGlzLm5hbWVkRm9sZGVyKTtcbiAgICBpZiAoZnMuZXhpc3RzU3luYyhwcmV2aW91c1ZlcnNpb25zRm9sZGVyKSkge1xuICAgICAgZnMucm1TeW5jKHByZXZpb3VzVmVyc2lvbnNGb2xkZXIsIHsgcmVjdXJzaXZlOiB0cnVlIH0pO1xuICAgIH1cblxuICAgIGNvbnN0IHRhcmdldFBhdGggPSBwYXRoLmpvaW4oYmFzZVBhdGgsIHRoaXMucGF0aCk7XG5cbiAgICBpZiAodGhpcy50eXBlID09PSBBc3NldFR5cGUuRElSRUNUT1JZKSB7XG4gICAgICBmcy5ta2RpclN5bmModGFyZ2V0UGF0aCwgeyByZWN1cnNpdmU6IHRydWUgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGZzLm1rZGlyU3luYyhwYXRoLmRpcm5hbWUodGFyZ2V0UGF0aCksIHsgcmVjdXJzaXZlOiB0cnVlIH0pO1xuICAgIH1cblxuICAgIHN3aXRjaCAodGhpcy50eXBlKSB7XG4gICAgICBjYXNlIEFzc2V0VHlwZS5GSUxFOlxuICAgICAgICBmcy5jb3B5RmlsZVN5bmModGhpcy5zb3VyY2VQYXRoLCB0YXJnZXRQYXRoKTtcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgQXNzZXRUeXBlLkRJUkVDVE9SWTpcbiAgICAgICAgY29weVN5bmModGhpcy5zb3VyY2VQYXRoLCB0YXJnZXRQYXRoKTtcbiAgICAgICAgYnJlYWs7XG5cbiAgICAgIGNhc2UgQXNzZXRUeXBlLkFSQ0hJVkU6XG4gICAgICAgIGFyY2hpdmVTeW5jKHRoaXMuc291cmNlUGF0aCwgdGFyZ2V0UGF0aCk7XG4gICAgICAgIGJyZWFrO1xuICAgICAgZGVmYXVsdDpcbiAgICAgICAgdGhyb3cgYXNzZXRUeXBlTm90SW1wbGVtZW50ZWQoKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==