"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackSynthesizer = void 0;
exports.invokeAspects = invokeAspects;
exports.addCustomSynthesis = addCustomSynthesis;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const path = require("path");
const fs = require("fs");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const stringify = require("json-stable-stringify");
const annotations_1 = require("../annotations");
const constructs_1 = require("constructs");
const aspect_1 = require("../aspect");
const validate_terraform_version_1 = require("../validations/validate-terraform-version");
const errors_1 = require("../errors");
// eslint-disable-next-line jsdoc/require-jsdoc
class StackSynthesizer {
    /**
     * @param stack the stack to synthesize
     * @param continueOnErrorAnnotations if set to true, `synthesize()` will not throw an error
     * if it encounters an error Annotation. However, it will still add all errors to the manifest
     * file. The cdktf-cli uses this flag to print the errors itself.
     */
    constructor(stack, continueOnErrorAnnotations = false, hclOutput = false) {
        this.stack = stack;
        this.continueOnErrorAnnotations = continueOnErrorAnnotations;
        this.hclOutput = hclOutput;
    }
    synthesize(session) {
        invokeAspects(this.stack);
        if (this.stack.hasResourceMove()) {
            this.stack.node.addValidation(new validate_terraform_version_1.ValidateTerraformVersion(">=1.5", `Resource move functionality is only supported for Terraform >=1.5. Please upgrade your Terraform version.`));
        }
        if (!session.skipValidation) {
            this.stack.runAllValidations();
        }
        const manifest = session.manifest;
        const stackManifest = manifest.forStack(this.stack);
        const workingDirectory = path.join(session.outdir, stackManifest.workingDirectory);
        if (!fs.existsSync(workingDirectory))
            fs.mkdirSync(workingDirectory);
        // call custom synthesis on child nodes (leafs first)
        this.stack.node
            .findAll(constructs_1.ConstructOrder.POSTORDER)
            .forEach((node) => getCustomSynthesis(node)?.onSynthesize(session));
        // collect Annotations into Manifest
        const annotations = this.stack.node
            .findAll()
            .map((node) => ({
            node,
            metadatas: node.node.metadata.filter(isAnnotationMetadata),
        }))
            .map(({ node, metadatas }) => metadatas.map((metadata) => ({
            constructPath: node.node.path,
            level: metadata.type,
            message: metadata.data,
            stacktrace: metadata.trace,
        })))
            .reduce((list, metadatas) => [...list, ...metadatas], []); // Array.flat()
        // it is readonly but this is the place where we are allowed to write to it
        stackManifest.annotations = annotations;
        // abort if one or more error annotations have been encountered
        if (!this.continueOnErrorAnnotations &&
            annotations.some(isErrorAnnotation)) {
            throw (0, errors_1.encounteredAnnotationWithLevelError)(annotations
                .filter(isErrorAnnotation)
                .map((a) => `[${a.constructPath}] ${a.message}`)
                .join("\n"));
        }
        if (this.hclOutput) {
            const hcl = this.stack.toHclTerraform();
            fs.writeFileSync(path.join(session.outdir, stackManifest.synthesizedStackPath), hcl.hcl);
            fs.writeFileSync(path.join(session.outdir, stackManifest.stackMetadataPath), stringify(hcl.metadata, { space: 2 }));
            return;
        }
        const jsonTfConfig = this.stack.toTerraform();
        fs.writeFileSync(path.join(session.outdir, stackManifest.synthesizedStackPath), stringify(jsonTfConfig, { space: 2 }));
    }
}
exports.StackSynthesizer = StackSynthesizer;
/**
 * Invoke aspects on the given construct tree.
 *
 * originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/@aws-cdk/core/lib/private/synthesis.ts#L99-L137
 */
function invokeAspects(root) {
    const invokedByPath = {};
    let nestedAspectWarning = false;
    recurse(root, []);
    // eslint-disable-next-line jsdoc/require-jsdoc
    function recurse(construct, inheritedAspects) {
        const node = construct.node;
        const aspects = aspect_1.Aspects.of(construct);
        const allAspectsHere = [...(inheritedAspects ?? []), ...aspects.all];
        const nodeAspectsCount = aspects.all.length;
        for (const aspect of allAspectsHere) {
            let invoked = invokedByPath[node.path];
            if (!invoked) {
                invoked = invokedByPath[node.path] = [];
            }
            if (invoked.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            // if an aspect was added to the node while invoking another aspect it will not be invoked, emit a warning
            // the `nestedAspectWarning` flag is used to prevent the warning from being emitted for every child
            if (!nestedAspectWarning && nodeAspectsCount !== aspects.all.length) {
                annotations_1.Annotations.of(construct).addWarning("We detected an Aspect was added via another Aspect, and will not be applied");
                nestedAspectWarning = true;
            }
            // mark as invoked for this node
            invoked.push(aspect);
        }
        for (const child of construct.node.children) {
            recurse(child, allAspectsHere);
        }
    }
}
const annotationMetadataEntryTypes = [
    annotations_1.AnnotationMetadataEntryType.INFO,
    annotations_1.AnnotationMetadataEntryType.WARN,
    annotations_1.AnnotationMetadataEntryType.ERROR,
];
// eslint-disable-next-line jsdoc/require-jsdoc
function isAnnotationMetadata(metadata) {
    return annotationMetadataEntryTypes.includes(metadata.type);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function isErrorAnnotation(annotation) {
    return annotation.level === annotations_1.AnnotationMetadataEntryType.ERROR;
}
// originally from https://github.com/aws/aws-cdk/blob/dcae3eead0dbf9acb1ed80ba95bb104c64cb1bd7/packages/%40aws-cdk/core/lib/private/synthesis.ts#L52
const CUSTOM_SYNTHESIS_SYM = Symbol.for("cdktf/customSynthesis");
// eslint-disable-next-line jsdoc/require-jsdoc
function addCustomSynthesis(construct, synthesis) {
    Object.defineProperty(construct, CUSTOM_SYNTHESIS_SYM, {
        value: synthesis,
        enumerable: false,
    });
}
// eslint-disable-next-line jsdoc/require-jsdoc
function getCustomSynthesis(construct) {
    return construct[CUSTOM_SYNTHESIS_SYM];
}
//# sourceMappingURL=data:application/json;base64,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