"use strict";
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// ----------------------------------------------------------------------
// PROPERTY MAPPERS
//
// These are used while converting generated classes/property bags to Terraform JSON objects
//
// We use identity mappers for the primitive types. These don't do anything but are there to make the code
// generation work out nicely (so the code generator doesn't need to emit different code for primitive
Object.defineProperty(exports, "__esModule", { value: true });
exports.isComplexElement = exports.numberToHclTerraform = exports.anyToHclTerraform = exports.booleanToHclTerraform = exports.stringToHclTerraform = exports.numberToTerraform = exports.anyToTerraform = exports.booleanToTerraform = exports.stringToTerraform = void 0;
exports.listMapper = listMapper;
exports.listMapperHcl = listMapperHcl;
exports.hashMapper = hashMapper;
exports.hashMapperHcl = hashMapperHcl;
exports.canInspect = canInspect;
const encoding_1 = require("./tokens/private/encoding");
Object.defineProperty(exports, "isComplexElement", { enumerable: true, get: function () { return encoding_1.isComplexElement; } });
const tokens_1 = require("./tokens");
const terraform_dynamic_block_1 = require("./terraform-dynamic-block");
const terraform_dynamic_expression_1 = require("./terraform-dynamic-expression");
// eslint-disable-next-line jsdoc/require-jsdoc
function identity(x) {
    return x;
}
exports.stringToTerraform = identity;
exports.booleanToTerraform = identity;
exports.anyToTerraform = identity;
exports.numberToTerraform = identity;
exports.stringToHclTerraform = identity;
exports.booleanToHclTerraform = identity;
exports.anyToHclTerraform = identity;
exports.numberToHclTerraform = identity;
/**
 * @param isBlockType blocks and lists of objects need to be handled different in Terraform
 *                    but are represented exactly the same in generated bindings
 *                    currently this is used to resolve TerraformDynamicBlocks if not within
 *                    a block type (will be resolved to a for expression for those instead of
 *                    a dynamic block)
 */
function listMapper(elementMapper, isBlockType) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        // replace dynamic expressions for block types so they can be detected and replaced properly by processDynamicAttributes()
        // which also relocates them to dynamic.attributeName (something we can't do when resolving a value, put it somewhere else)
        // if isBlockType is false, the TerraformDynamicExpression that is present will be resolved (it implements IResolvable) to a
        // for expression directly (which can be used e.g. within Terraform functions or for attributes that are not of a block type)
        if (terraform_dynamic_expression_1.TerraformDynamicExpression.isTerraformDynamicExpression(x) &&
            isBlockType) {
            return terraform_dynamic_block_1.TerraformDynamicBlock.fromDynamicExpression(x);
        }
        if (!Array.isArray(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
/**
 * @param isBlockType blocks and lists of objects need to be handled different in Terraform
 *                    but are represented exactly the same in generated bindings
 *                    currently this is used to resolve TerraformDynamicBlocks if not within
 *                    a block type (will be resolved to a for expression for those instead of
 *                    a dynamic block)
 */
function listMapperHcl(elementMapper, isBlockType) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        // replace dynamic expressions for block types so they can be detected and replaced properly by processDynamicAttributes()
        // which also relocates them to dynamic.attributeName (something we can't do when resolving a value, put it somewhere else)
        // if isBlockType is false, the TerraformDynamicExpression that is present will be resolved (it implements IResolvable) to a
        // for expression directly (which can be used e.g. within Terraform functions or for attributes that are not of a block type)
        if (terraform_dynamic_expression_1.TerraformDynamicExpression.isTerraformDynamicExpression(x) &&
            isBlockType) {
            return terraform_dynamic_block_1.TerraformDynamicBlock.fromDynamicExpression(x);
        }
        if (!Array.isArray(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
// eslint-disable-next-line jsdoc/require-jsdoc
function hashMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        // Check if it's a token object
        if ((0, encoding_1.containsComplexElement)(x)) {
            return x;
        }
        if (tokens_1.Tokenization.isResolvable(x)) {
            return x;
        }
        // We can't treat strings as hashes (likely a token or a misconfiguration)
        if (typeof x === "string") {
            return x;
        }
        if ((0, encoding_1.containsMapToken)(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
/**
 *
 */
function hashMapperHcl(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        // Check if it's a token object
        if ((0, encoding_1.containsComplexElement)(x)) {
            return x;
        }
        if (tokens_1.Tokenization.isResolvable(x)) {
            return x;
        }
        // We can't treat strings as hashes (likely a token or a misconfiguration)
        if (typeof x === "string") {
            return x;
        }
        if ((0, encoding_1.containsMapToken)(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
/**
 * Return whether this object can be validated at all
 *
 * True unless it's undefined
 */
function canInspect(x) {
    // Note: using weak equality on purpose, we also want to catch undefined
    return x != null;
}
//# sourceMappingURL=data:application/json;base64,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