"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.makeUniqueId = makeUniqueId;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const errors_1 = require("../errors");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = "Resource";
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = "Default";
const PATH_SEP = "/";
const UNIQUE_SEP = "_";
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
const MAX_ID_LEN = 255;
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter((x) => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw (0, errors_1.cannotCalcIdForEmptySetOfComponents)();
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        // we filter out non-alpha characters but that is actually a bad idea
        // because it could create conflicts ("A-B" and "AB" will render the same
        // logical ID). sadly, changing it in the 1.x version line is impossible
        // because it will be a breaking change. we should consider for v2.0.
        // https://github.com/aws/aws-cdk/issues/6421
        const candidate = removeDisallowedCharacters(components[0]);
        // if our candidate is short enough, use it as is. otherwise, fall back to
        // the normal mode.
        if (candidate.length <= MAX_ID_LEN) {
            return candidate;
        }
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter((x) => x !== HIDDEN_FROM_HUMAN_ID)
        .map((s) => removeDisallowedCharacters(s))
        .join(UNIQUE_SEP)
        .slice(0, MAX_HUMAN_LEN);
    return human + UNIQUE_SEP + hash;
}
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto
        .createHash("md5")
        .update(path.join(PATH_SEP))
        .digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
// eslint-disable-next-line jsdoc/require-jsdoc
function removeDisallowedCharacters(s) {
    return removeNonAlphanumericSep(s);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function removeNonAlphanumericSep(s) {
    return s.replace(/[^A-Za-z0-9_-]/g, "");
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidW5pcXVlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidW5pcXVlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBbUNBLG9DQWlDQztBQXBFRCwrQkFBK0I7QUFDL0IsbUNBQW1DO0FBQ25DLDJDQUEyQztBQUMzQyxpQ0FBaUM7QUFDakMsc0NBQWdFO0FBRWhFOzs7OztHQUtHO0FBQ0gsTUFBTSxvQkFBb0IsR0FBRyxVQUFVLENBQUM7QUFFeEM7O0dBRUc7QUFDSCxNQUFNLFNBQVMsR0FBRyxTQUFTLENBQUM7QUFFNUIsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDO0FBQ3JCLE1BQU0sVUFBVSxHQUFHLEdBQUcsQ0FBQztBQUV2QixNQUFNLFFBQVEsR0FBRyxDQUFDLENBQUM7QUFDbkIsTUFBTSxhQUFhLEdBQUcsR0FBRyxDQUFDLENBQUMsb0JBQW9CO0FBQy9DLE1BQU0sVUFBVSxHQUFHLEdBQUcsQ0FBQztBQUV2Qjs7Ozs7Ozs7R0FRRztBQUNILFNBQWdCLFlBQVksQ0FBQyxVQUFvQjtJQUMvQyxVQUFVLEdBQUcsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO0lBRXZELElBQUksVUFBVSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUM1QixNQUFNLElBQUEsNENBQW1DLEdBQUUsQ0FBQztJQUM5QyxDQUFDO0lBRUQsMkVBQTJFO0lBQzNFLDJFQUEyRTtJQUMzRSxnQ0FBZ0M7SUFDaEMsSUFBSSxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQzVCLHFFQUFxRTtRQUNyRSx5RUFBeUU7UUFDekUsd0VBQXdFO1FBQ3hFLHFFQUFxRTtRQUNyRSw2Q0FBNkM7UUFDN0MsTUFBTSxTQUFTLEdBQUcsMEJBQTBCLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFNUQsMEVBQTBFO1FBQzFFLG1CQUFtQjtRQUNuQixJQUFJLFNBQVMsQ0FBQyxNQUFNLElBQUksVUFBVSxFQUFFLENBQUM7WUFDbkMsT0FBTyxTQUFTLENBQUM7UUFDbkIsQ0FBQztJQUNILENBQUM7SUFFRCxNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7SUFDbEMsTUFBTSxLQUFLLEdBQUcsV0FBVyxDQUFDLFVBQVUsQ0FBQztTQUNsQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsS0FBSyxvQkFBb0IsQ0FBQztTQUN6QyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUMsQ0FBQyxDQUFDO1NBQ3pDLElBQUksQ0FBQyxVQUFVLENBQUM7U0FDaEIsS0FBSyxDQUFDLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUUzQixPQUFPLEtBQUssR0FBRyxVQUFVLEdBQUcsSUFBSSxDQUFDO0FBQ25DLENBQUM7QUFFRDs7OztHQUlHO0FBQ0gsU0FBUyxRQUFRLENBQUMsSUFBYztJQUM5QixNQUFNLEdBQUcsR0FBRyxNQUFNO1NBQ2YsVUFBVSxDQUFDLEtBQUssQ0FBQztTQUNqQixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUMzQixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDakIsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUM5QyxDQUFDO0FBRUQsK0NBQStDO0FBQy9DLFNBQVMsMEJBQTBCLENBQUMsQ0FBUztJQUMzQyxPQUFPLHdCQUF3QixDQUFDLENBQUMsQ0FBQyxDQUFDO0FBQ3JDLENBQUM7QUFFRCwrQ0FBK0M7QUFDL0MsU0FBUyx3QkFBd0IsQ0FBQyxDQUFTO0lBQ3pDLE9BQU8sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxFQUFFLENBQUMsQ0FBQztBQUMxQyxDQUFDO0FBRUQ7Ozs7O0dBS0c7QUFDSCxTQUFTLFdBQVcsQ0FBQyxJQUFjO0lBQ2pDLE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7SUFFaEMsS0FBSyxNQUFNLFNBQVMsSUFBSSxJQUFJLEVBQUUsQ0FBQztRQUM3QixJQUFJLEdBQUcsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxFQUFFLENBQUM7WUFDakUsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUN0QixDQUFDO0lBQ0gsQ0FBQztJQUVELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmNcbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNUEwtMi4wXG4vLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bm8tdmFyLXJlcXVpcmVzXG5pbXBvcnQgKiBhcyBjcnlwdG8gZnJvbSBcImNyeXB0b1wiO1xuaW1wb3J0IHsgY2Fubm90Q2FsY0lkRm9yRW1wdHlTZXRPZkNvbXBvbmVudHMgfSBmcm9tIFwiLi4vZXJyb3JzXCI7XG5cbi8qKlxuICogUmVzb3VyY2VzIHdpdGggdGhpcyBJRCBhcmUgaGlkZGVuIGZyb20gaHVtYW5zXG4gKlxuICogVGhleSBkbyBub3QgYXBwZWFyIGluIHRoZSBodW1hbi1yZWFkYWJsZSBwYXJ0IG9mIHRoZSBsb2dpY2FsIElELFxuICogYnV0IHRoZXkgYXJlIGluY2x1ZGVkIGluIHRoZSBoYXNoIGNhbGN1bGF0aW9uLlxuICovXG5jb25zdCBISURERU5fRlJPTV9IVU1BTl9JRCA9IFwiUmVzb3VyY2VcIjtcblxuLyoqXG4gKiBSZXNvdXJjZXMgd2l0aCB0aGlzIElEIGFyZSBjb21wbGV0ZSBoaWRkZW4gZnJvbSB0aGUgbG9naWNhbCBJRCBjYWxjdWxhdGlvbi5cbiAqL1xuY29uc3QgSElEREVOX0lEID0gXCJEZWZhdWx0XCI7XG5cbmNvbnN0IFBBVEhfU0VQID0gXCIvXCI7XG5jb25zdCBVTklRVUVfU0VQID0gXCJfXCI7XG5cbmNvbnN0IEhBU0hfTEVOID0gODtcbmNvbnN0IE1BWF9IVU1BTl9MRU4gPSAyNDA7IC8vIG1heCBJRCBsZW4gaXMgMjU1XG5jb25zdCBNQVhfSURfTEVOID0gMjU1O1xuXG4vKipcbiAqIENhbGN1bGF0ZXMgYSB1bmlxdWUgSUQgZm9yIGEgc2V0IG9mIHRleHR1YWwgY29tcG9uZW50cy5cbiAqXG4gKiBUaGlzIGlzIGRvbmUgYnkgY2FsY3VsYXRpbmcgYSBoYXNoIG9uIHRoZSBmdWxsIHBhdGggYW5kIHVzaW5nIGl0IGFzIGEgc3VmZml4XG4gKiBvZiBhIGxlbmd0aC1saW1pdGVkIFwiaHVtYW5cIiByZW5kaXRpb24gb2YgdGhlIHBhdGggY29tcG9uZW50cy5cbiAqXG4gKiBAcGFyYW0gY29tcG9uZW50cyBUaGUgcGF0aCBjb21wb25lbnRzXG4gKiBAcmV0dXJucyBhIHVuaXF1ZSBhbHBoYS1udW1lcmljIGlkZW50aWZpZXIgd2l0aCBhIG1heGltdW0gbGVuZ3RoIG9mIDI1NVxuICovXG5leHBvcnQgZnVuY3Rpb24gbWFrZVVuaXF1ZUlkKGNvbXBvbmVudHM6IHN0cmluZ1tdKSB7XG4gIGNvbXBvbmVudHMgPSBjb21wb25lbnRzLmZpbHRlcigoeCkgPT4geCAhPT0gSElEREVOX0lEKTtcblxuICBpZiAoY29tcG9uZW50cy5sZW5ndGggPT09IDApIHtcbiAgICB0aHJvdyBjYW5ub3RDYWxjSWRGb3JFbXB0eVNldE9mQ29tcG9uZW50cygpO1xuICB9XG5cbiAgLy8gdG9wLWxldmVsIHJlc291cmNlcyB3aWxsIHNpbXBseSB1c2UgdGhlIGBuYW1lYCBhcy1pcyBpbiBvcmRlciB0byBzdXBwb3J0XG4gIC8vIHRyYW5zcGFyZW50IG1pZ3JhdGlvbiBvZiBjbG91ZGZvcm1hdGlvbiB0ZW1wbGF0ZXMgdG8gdGhlIENESyB3aXRob3V0IHRoZVxuICAvLyBuZWVkIHRvIHJlbmFtZSBhbGwgcmVzb3VyY2VzLlxuICBpZiAoY29tcG9uZW50cy5sZW5ndGggPT09IDEpIHtcbiAgICAvLyB3ZSBmaWx0ZXIgb3V0IG5vbi1hbHBoYSBjaGFyYWN0ZXJzIGJ1dCB0aGF0IGlzIGFjdHVhbGx5IGEgYmFkIGlkZWFcbiAgICAvLyBiZWNhdXNlIGl0IGNvdWxkIGNyZWF0ZSBjb25mbGljdHMgKFwiQS1CXCIgYW5kIFwiQUJcIiB3aWxsIHJlbmRlciB0aGUgc2FtZVxuICAgIC8vIGxvZ2ljYWwgSUQpLiBzYWRseSwgY2hhbmdpbmcgaXQgaW4gdGhlIDEueCB2ZXJzaW9uIGxpbmUgaXMgaW1wb3NzaWJsZVxuICAgIC8vIGJlY2F1c2UgaXQgd2lsbCBiZSBhIGJyZWFraW5nIGNoYW5nZS4gd2Ugc2hvdWxkIGNvbnNpZGVyIGZvciB2Mi4wLlxuICAgIC8vIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvNjQyMVxuICAgIGNvbnN0IGNhbmRpZGF0ZSA9IHJlbW92ZURpc2FsbG93ZWRDaGFyYWN0ZXJzKGNvbXBvbmVudHNbMF0pO1xuXG4gICAgLy8gaWYgb3VyIGNhbmRpZGF0ZSBpcyBzaG9ydCBlbm91Z2gsIHVzZSBpdCBhcyBpcy4gb3RoZXJ3aXNlLCBmYWxsIGJhY2sgdG9cbiAgICAvLyB0aGUgbm9ybWFsIG1vZGUuXG4gICAgaWYgKGNhbmRpZGF0ZS5sZW5ndGggPD0gTUFYX0lEX0xFTikge1xuICAgICAgcmV0dXJuIGNhbmRpZGF0ZTtcbiAgICB9XG4gIH1cblxuICBjb25zdCBoYXNoID0gcGF0aEhhc2goY29tcG9uZW50cyk7XG4gIGNvbnN0IGh1bWFuID0gcmVtb3ZlRHVwZXMoY29tcG9uZW50cylcbiAgICAuZmlsdGVyKCh4KSA9PiB4ICE9PSBISURERU5fRlJPTV9IVU1BTl9JRClcbiAgICAubWFwKChzKSA9PiByZW1vdmVEaXNhbGxvd2VkQ2hhcmFjdGVycyhzKSlcbiAgICAuam9pbihVTklRVUVfU0VQKVxuICAgIC5zbGljZSgwLCBNQVhfSFVNQU5fTEVOKTtcblxuICByZXR1cm4gaHVtYW4gKyBVTklRVUVfU0VQICsgaGFzaDtcbn1cblxuLyoqXG4gKiBUYWtlIGEgaGFzaCBvZiB0aGUgZ2l2ZW4gcGF0aC5cbiAqXG4gKiBUaGUgaGFzaCBpcyBsaW1pdGVkIGluIHNpemUuXG4gKi9cbmZ1bmN0aW9uIHBhdGhIYXNoKHBhdGg6IHN0cmluZ1tdKTogc3RyaW5nIHtcbiAgY29uc3QgbWQ1ID0gY3J5cHRvXG4gICAgLmNyZWF0ZUhhc2goXCJtZDVcIilcbiAgICAudXBkYXRlKHBhdGguam9pbihQQVRIX1NFUCkpXG4gICAgLmRpZ2VzdChcImhleFwiKTtcbiAgcmV0dXJuIG1kNS5zbGljZSgwLCBIQVNIX0xFTikudG9VcHBlckNhc2UoKTtcbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGpzZG9jL3JlcXVpcmUtanNkb2NcbmZ1bmN0aW9uIHJlbW92ZURpc2FsbG93ZWRDaGFyYWN0ZXJzKHM6IHN0cmluZykge1xuICByZXR1cm4gcmVtb3ZlTm9uQWxwaGFudW1lcmljU2VwKHMpO1xufVxuXG4vLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUganNkb2MvcmVxdWlyZS1qc2RvY1xuZnVuY3Rpb24gcmVtb3ZlTm9uQWxwaGFudW1lcmljU2VwKHM6IHN0cmluZykge1xuICByZXR1cm4gcy5yZXBsYWNlKC9bXkEtWmEtejAtOV8tXS9nLCBcIlwiKTtcbn1cblxuLyoqXG4gKiBSZW1vdmUgZHVwbGljYXRlIFwidGVybXNcIiBmcm9tIHRoZSBwYXRoIGxpc3RcbiAqXG4gKiBJZiB0aGUgcHJldmlvdXMgcGF0aCBjb21wb25lbnQgbmFtZSBlbmRzIHdpdGggdGhpcyBjb21wb25lbnQgbmFtZSwgc2tpcCB0aGVcbiAqIGN1cnJlbnQgY29tcG9uZW50LlxuICovXG5mdW5jdGlvbiByZW1vdmVEdXBlcyhwYXRoOiBzdHJpbmdbXSk6IHN0cmluZ1tdIHtcbiAgY29uc3QgcmV0ID0gbmV3IEFycmF5PHN0cmluZz4oKTtcblxuICBmb3IgKGNvbnN0IGNvbXBvbmVudCBvZiBwYXRoKSB7XG4gICAgaWYgKHJldC5sZW5ndGggPT09IDAgfHwgIXJldFtyZXQubGVuZ3RoIC0gMV0uZW5kc1dpdGgoY29tcG9uZW50KSkge1xuICAgICAgcmV0LnB1c2goY29tcG9uZW50KTtcbiAgICB9XG4gIH1cblxuICByZXR1cm4gcmV0O1xufVxuIl19