"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.copySync = copySync;
exports.archiveSync = archiveSync;
exports.hashPath = hashPath;
exports.findFileAboveCwd = findFileAboveCwd;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const fs = require("fs");
const path = require("path");
const crypto = require("crypto");
const archiver = require("archiver");
const errors_1 = require("../errors");
const child_process_1 = require("child_process");
const HASH_LEN = 32;
// Full implementation at https://github.com/jprichardson/node-fs-extra/blob/master/lib/copy/copy-sync.js
/**
 * Copy a file or directory. The directory can have contents and subfolders.
 * @param {string} src
 * @param {string} dest
 */
function copySync(src, dest) {
    /**
     * Copies file if present otherwise walks subfolder
     * @param {string} p
     */
    function copyItem(p) {
        const sourcePath = path.resolve(src, p);
        const stat = fs.statSync(sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(sourcePath, path.resolve(dest, p));
        }
        if (stat.isDirectory()) {
            walkSubfolder(p);
        }
    }
    /**
     * Copies contents of subfolder
     * @param {string} p
     */
    function walkSubfolder(p) {
        const sourceDir = path.resolve(src, p);
        fs.mkdirSync(path.resolve(dest, p), { recursive: true });
        fs.readdirSync(sourceDir).forEach((item) => copyItem(path.join(p, item)));
    }
    walkSubfolder(".");
}
/**
 * Zips contents at src and places zip archive at dest
 * @param {string} src
 * @param {string} dest
 */
function archiveSync(src, dest) {
    // Run this module as a CLI to get around the synchronous limitation
    try {
        (0, child_process_1.execSync)(`node ${__filename} ${src} ${dest}`, { encoding: "utf-8" });
    }
    catch (err) {
        throw (0, errors_1.assetCanNotCreateZipArchive)(src, dest, err);
    }
}
/**
 *
 * @param src
 * @param dest
 */
async function runArchive(src, dest) {
    return new Promise((resolve, reject) => {
        const output = fs.createWriteStream(dest);
        const archive = archiver("zip", {
            zlib: { level: 9 }, // Sets the compression level.
        });
        archive.pipe(output);
        archive.on("error", (err) => {
            reject(err);
        });
        output.on("close", () => {
            resolve();
        });
        archive.directory(src, false);
        archive.finalize();
    });
}
// If this file is executed as a CLI we run archive directly
// It's a bit of a hack due to us being restricted to synchronous functions
// when there is no sync way to create a zip archive.
// We get around this by using execSync and invoking this file as the CLI.
// This only works for one function, but we only have this use-case once.
if (require.main === module) {
    const src = process.argv[2];
    const dest = process.argv[3];
    runArchive(src, dest)
        .then(() => {
        process.exit(0);
    })
        .catch((err) => {
        console.error(err);
        process.exit(1);
    });
}
// eslint-disable-next-line jsdoc/require-jsdoc
function hashPath(src) {
    const hash = crypto.createHash("md5");
    // eslint-disable-next-line jsdoc/require-jsdoc
    function hashRecursion(p) {
        const stat = fs.statSync(p);
        if (stat.isFile()) {
            hash.update(fs.readFileSync(p));
        }
        else if (stat.isDirectory()) {
            fs.readdirSync(p).forEach((filename) => hashRecursion(path.resolve(p, filename)));
        }
    }
    hashRecursion(src);
    return hash.digest("hex").slice(0, HASH_LEN).toUpperCase();
}
// eslint-disable-next-line jsdoc/require-jsdoc
function findFileAboveCwd(file, rootPath = process.cwd()) {
    const fullPath = path.resolve(rootPath, file);
    if (fs.existsSync(fullPath)) {
        return fullPath;
    }
    const parentDir = path.resolve(rootPath, "..");
    if (fs.existsSync(parentDir) && parentDir !== rootPath) {
        return findFileAboveCwd(file, parentDir);
    }
    return null;
}
//# sourceMappingURL=data:application/json;base64,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