"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.cleanForMetadata = cleanForMetadata;
exports.renderList = renderList;
exports.renderListValue = renderListValue;
exports.renderMap = renderMap;
exports.renderMapValue = renderMapValue;
exports.renderResource = renderResource;
exports.renderDatasource = renderDatasource;
exports.renderProvisionerBlock = renderProvisionerBlock;
exports.renderDynamicBlocks = renderDynamicBlocks;
exports.renderProvider = renderProvider;
exports.renderModule = renderModule;
exports.renderOutput = renderOutput;
exports.renderLocals = renderLocals;
exports.renderVariable = renderVariable;
exports.renderMoved = renderMoved;
exports.renderImport = renderImport;
exports.renderTerraform = renderTerraform;
exports.renderBlock = renderBlock;
exports.renderSimpleAttributes = renderSimpleAttributes;
exports.renderAttributes = renderAttributes;
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
const token_1 = require("../tokens/token");
/**
 *
 */
function escapeQuotes(str) {
    // Don't attempt to escape if we have an expression
    if (str.match(/\$\{/)) {
        return str;
    }
    return str.replace(/(?<!\\)"/g, '\\"');
}
/**
 *
 */
function wrapIdentifierInQuotesIfNeeded(key) {
    return /(^\d)|[^A-Za-z0-9_-]/.test(key) ? `"${key}"` : key;
}
/**
 *
 */
function renderString(str) {
    if (str === "") {
        return `""`;
    }
    if (!str) {
        return str;
    }
    if (typeof str !== "string") {
        throw new Error("Unable to process attribute that should have been a string, but isn't");
    }
    const lines = str.split(/\r\n|[\n\r]/);
    if (lines.length === 1)
        return `"${escapeQuotes(str)}"`;
    return `<<EOF\n${lines.join("\n")}\nEOF`;
}
/**
 *
 */
function cleanForMetadata(block) {
    const res = {};
    if (typeof block !== "object" && !Array.isArray(block)) {
        return block;
    }
    if (Array.isArray(block)) {
        return block.map((i) => cleanForMetadata(i));
    }
    const keys = Object.keys(block);
    if (keys.includes("value")) {
        return cleanForMetadata(block.value);
    }
    for (const key of keys) {
        const value = block[key];
        if (typeof value === "object" && value != null && !Array.isArray(value)) {
            // if the value is a resolvable we don't want to recurse into it
            if (token_1.Tokenization.isResolvable(value)) {
                res[key] = {};
                continue;
            }
            res[key] = cleanForMetadata(value);
            continue;
        }
        else if (typeof value === "object" &&
            value != null &&
            Array.isArray(value)) {
            if (Array.isArray(res[key])) {
                res[key] = [...res[key], ...value];
            }
            else {
                res[key] = value;
            }
        }
        else if (value === undefined) {
            delete res[key];
        }
        else {
            res[key] = value;
        }
    }
    return res;
}
/**
 *
 */
function renderList(v, name) {
    const { isBlock, is_block, value, storage_class_type, storageClassType } = v;
    const block = isBlock || is_block;
    const classType = storageClassType || storage_class_type;
    if (value === undefined) {
        return "";
    }
    if (typeof value === "string") {
        // this could be an expression, so we don't need to do anything here
        return renderString(value);
    }
    if (name) {
        if (block) {
            if (!Array.isArray(value)) {
                return `${name} {` + `\n${renderAttributes(value)}\n}`;
            }
            return value
                .map((v) => {
                return `${name} {` + `\n${renderAttributes(v)}\n}`;
            })
                .join("\n");
        }
        if (!Array.isArray(value)) {
            return `${name} = ${renderListValue(value, classType)}`;
        }
        return `${name} = [
${value.map((v) => renderListValue(v, classType)).join(",\n")}
]`;
    }
    if (!Array.isArray(value)) {
        // A list type of only a single item
        return renderListValue(value, classType);
    }
    if (block) {
        return value.map((v) => renderListValue(v, classType)).join("\n");
    }
    return `[
${value.map((v) => renderListValue(v, classType)).join(",\n")}
]`;
}
/**
 *
 */
function renderListValue(value, storageClassType) {
    if (storageClassType === "stringList") {
        return renderString(value);
    }
    if (storageClassType === "numberList" || storageClassType === "booleanList") {
        return `${value}`;
    }
    return renderBlock(value, storageClassType);
}
/**
 *
 */
function renderMap(map) {
    if (typeof map === "string") {
        // this could be an expression, so we don't need to do anything here
        return `"${escapeQuotes(map)}"`;
    }
    return `{
${Object.entries(map)
        .map(([k, v]) => {
        const wrappedKey = wrapIdentifierInQuotesIfNeeded(k);
        return `${wrappedKey} = ${renderMapValue(v)}`;
    })
        .join("\n")}
}`;
}
/**
 *
 */
function renderMapValue(value) {
    if (typeof value === "string") {
        return renderString(value);
    }
    if (typeof value === "number") {
        return `${value}`;
    }
    if (typeof value === "boolean") {
        return `${value}`;
    }
    if (typeof value === "object") {
        if (Array.isArray(value)) {
            return renderListValue(value, "unknownStorageClass");
        }
        if (value === null) {
            return "null";
        }
        return renderMap(value);
    }
    return `${value}`;
}
/**
 *
 */
function renderResource(resource) {
    const resourceType = Object.keys(resource)[0];
    const resourcesWithType = resource[resourceType];
    const resourceName = Object.keys(resourcesWithType)[0];
    const resourceAttributes = resourcesWithType[resourceName];
    if (resourceAttributes.connection) {
        resourceAttributes.connection = {
            value: resourceAttributes.connection,
            isBlock: true,
        };
    }
    const { provisioner, dynamic, ...otherAttrs } = resourceAttributes;
    const hcl = [`resource "${resourceType}" "${resourceName}" {`];
    const attrs = renderAttributes(otherAttrs);
    if (attrs)
        hcl.push(attrs);
    if (provisioner)
        hcl.push(renderProvisionerBlock(provisioner));
    if (dynamic)
        hcl.push(...renderDynamicBlocks(dynamic));
    hcl.push("}");
    return {
        hcl: hcl.join("\n"),
        metadata: {
            resource: {
                [resourceType]: {
                    [resourceName]: {
                        "//": resourceAttributes["//"],
                    },
                },
            },
        },
    };
}
/**
 *
 */
function renderDatasource(dataSource) {
    const dataSourceType = Object.keys(dataSource)[0];
    const dataSourcesWithType = dataSource[dataSourceType];
    const dataSourceName = Object.keys(dataSourcesWithType)[0];
    const dataSourceAttributes = dataSourcesWithType[dataSourceName];
    const { dynamic, ...otherAttrs } = dataSourceAttributes;
    const hcl = [`data "${dataSourceType}" "${dataSourceName}" {`];
    const attrs = renderAttributes(otherAttrs);
    if (attrs)
        hcl.push(attrs);
    if (dynamic)
        hcl.push(...renderDynamicBlocks(dynamic));
    hcl.push("}");
    return {
        hcl: hcl.join("\n"),
        metadata: {
            data: {
                [dataSourceType]: {
                    [dataSourceName]: {
                        "//": dataSourceAttributes["//"],
                    },
                },
            },
        },
    };
}
/**
 *
 */
function renderProvisionerBlock(provisioners) {
    return provisioners
        .map((provisioner) => {
        const provisionerType = Object.keys(provisioner)[0];
        const provisionerAttrs = provisioner[provisionerType];
        if (provisionerAttrs.connection) {
            provisionerAttrs.connection = {
                value: provisionerAttrs.connection,
                isBlock: true,
            };
        }
        const { dynamic, ...otherAttrs } = provisionerAttrs;
        const hcl = [`provisioner "${provisionerType}" {`];
        const attrs = renderAttributes(otherAttrs.value || otherAttrs);
        if (attrs)
            hcl.push(attrs);
        if (dynamic)
            hcl.push(...renderDynamicBlocks(dynamic));
        hcl.push("}");
        return hcl.join("\n");
    })
        .join("\n\n");
}
/**
 *
 */
function renderDynamicBlocks(dynamic) {
    return Object.entries(dynamic).map(([dynamicName, dynamicAttrs]) => {
        const res = [`dynamic "${dynamicName}" {`];
        res.push(`for_each = ${renderFuzzyJsonExpression(dynamicAttrs?.for_each)}`);
        if (dynamicAttrs?.iterator) {
            res.push(`iterator = ${dynamicAttrs?.iterator}`);
        }
        if (dynamicAttrs?.labels) {
            res.push(`labels = ${renderFuzzyJsonExpression(dynamicAttrs?.labels)}`);
        }
        res.push(`content {
  ${renderAttributes(dynamicAttrs.content)}
  }`);
        res.push(`}`);
        return res.join("\n");
    });
}
/**
 *
 */
function renderProvider(provider) {
    const providerName = Object.keys(provider)[0];
    const providerAttributes = provider[providerName];
    return providerAttributes
        .map((providerInstance) => {
        const { dynamic, ...otherAttrs } = providerInstance;
        const hcl = [`provider "${providerName}" {`];
        const attrs = renderAttributes(otherAttrs);
        if (attrs)
            hcl.push(attrs);
        if (dynamic)
            hcl.push(...renderDynamicBlocks(dynamic));
        hcl.push("}");
        return hcl.join("\n");
    })
        .join("\n\n");
}
/**
 *
 */
function renderModule(module) {
    const moduleName = Object.keys(module)[0];
    const moduleAttributes = module[moduleName];
    return {
        hcl: `module "${moduleName}" {
${renderAttributes(moduleAttributes)}
}`,
        metadata: {
            module: {
                [moduleName]: {
                    "//": moduleAttributes["//"],
                },
            },
        },
    };
}
/**
 *
 */
function renderOutput(output) {
    const outputName = Object.keys(output)[0];
    const outputAttributes = output[outputName];
    return `output "${outputName}" {
${renderAttributes(outputAttributes)}
}`;
}
/**
 *
 */
function renderLocals(locals) {
    if (!locals) {
        return "";
    }
    const localNames = Object.keys(locals);
    if (localNames.length === 0) {
        return "";
    }
    return `locals {
    ${localNames
        .filter((name) => !!locals[name].value)
        .map((name) => {
        const value = renderFuzzyJsonExpression(locals[name].value);
        return `${name} = ${value}`;
    })
        .join("\n")}
}`;
}
/**
 *
 */
function renderVariable(variable) {
    const variableName = Object.keys(variable)[0];
    const variableAttributes = variable[variableName];
    return `variable "${variableName}" {
${renderAttributes(variableAttributes)}
}`;
}
/**
 *
 */
function renderMoved(move) {
    const movedBlocks = move.map((moveBlock) => {
        return `moved {
${renderAttributes(moveBlock)}
}`;
    });
    return movedBlocks.join("\n");
}
/**
 *
 */
function renderImport(imports) {
    const importBlocks = imports.map((importBlock) => {
        return `import {
${renderAttributes(importBlock)}
}`;
    });
    return importBlocks.join("\n");
}
/**
 *
 */
function renderTerraform(terraform) {
    const blockAttributes = ["required_providers", "backend", "cloud"];
    const requiredProviders = `required_providers {
${renderSimpleAttributes(terraform.required_providers)}
}`;
    const backends = Object.keys(terraform.backend || {}).map((backendName) => {
        return `backend "${backendName}" {
${renderAttributes(terraform.backend[backendName])}
}`;
    });
    const cloudBackend = terraform.cloud
        ? `cloud {
${renderAttributes(terraform.cloud)}
}`
        : undefined;
    const otherAttributes = Object.keys(terraform).filter((key) => !blockAttributes.includes(key));
    return `terraform {
${requiredProviders}
${[...backends, cloudBackend].join("\n")}
${renderSimpleAttributes(otherAttributes)}
}`;
}
/**
 *
 */
function renderBlock(block, _storageClassType) {
    return `{
${renderAttributes(block)}
}`;
}
/**
 *
 */
function renderFuzzyJsonObject(jsonObject) {
    return [
        "{",
        ...Object.entries(jsonObject).map(([name, value]) => {
            const wrappedKey = wrapIdentifierInQuotesIfNeeded(name);
            return `${wrappedKey} = ${renderFuzzyJsonExpression(value)}`;
        }),
        "}",
    ].join("\n");
}
/**
 *
 */
function renderFuzzyJsonExpression(jsonExpression) {
    if (Array.isArray(jsonExpression)) {
        return [
            "[",
            ...jsonExpression.map((value) => `${renderFuzzyJsonExpression(value)},`),
            "]",
        ].join("\n");
    }
    if (typeof jsonExpression === "object") {
        return renderFuzzyJsonObject(jsonExpression);
    }
    if (typeof jsonExpression === "string") {
        if (jsonExpression === "") {
            return `""`;
        }
        if (jsonExpression.includes("${")) {
            return `"${jsonExpression}"`;
        }
        if (parseInt(jsonExpression, 10).toString() === jsonExpression) {
            return jsonExpression;
        }
        if (jsonExpression.startsWith("[")) {
            return jsonExpression;
        }
        if (jsonExpression.startsWith("{")) {
            return jsonExpression;
        }
        if (jsonExpression.startsWith('"')) {
            return jsonExpression;
        }
        return `"${escapeQuotes(jsonExpression)}"`;
    }
    if (jsonExpression === "true" || jsonExpression === "false") {
        return jsonExpression;
    }
    if (jsonExpression === "null") {
        return jsonExpression;
    }
    return `${jsonExpression}`;
}
/**
 *
 */
function renderSimpleAttributes(attributes) {
    if (attributes === undefined) {
        return "";
    }
    return Object.entries(attributes)
        .map(([name, value]) => `  ${name} = ${renderFuzzyJsonExpression(value)}`)
        .join("\n");
}
/**
 *
 */
function renderAttributes(attributes) {
    if (attributes === undefined || attributes === null) {
        return "";
    }
    return Object.entries(attributes)
        .map(([name, v]) => {
        if (name === "//") {
            return undefined;
        }
        const metaBlocks = ["lifecycle", "provisioner"];
        //
        // We might have some attributes that don't have type information
        // just try to guess them
        if (typeof v === "string" || typeof v === "number") {
            return `${name} = ${renderFuzzyJsonExpression(v)}`;
        }
        else if (typeof v === "boolean") {
            return `${name} = ${v}`;
        }
        else if (Array.isArray(v)) {
            return `${name} = ${renderFuzzyJsonExpression(v)}`;
        }
        else if (v === null) {
            return `${name} = null`;
        }
        else if (typeof v === "object" &&
            // eslint-disable-next-line no-prototype-builtins
            !v.hasOwnProperty("value") &&
            // eslint-disable-next-line no-prototype-builtins
            !v.hasOwnProperty("dynamic")) {
            if (metaBlocks.includes(name)) {
                return `${name} {
${renderSimpleAttributes(v)}
}`;
            }
            return `${name} = ${renderFuzzyJsonExpression(v)}`;
        }
        else if (v === undefined) {
            return undefined;
        }
        // Referencing both isBlock and is_block, because sometimes we pass through a snake case filter
        // within attributes.
        const { dynamic, value, type, isBlock, storageClassType, is_block, storage_class_type, } = v;
        const block = isBlock || is_block || metaBlocks.includes(name);
        const classType = storageClassType || storage_class_type;
        if (dynamic) {
            return renderDynamicBlocks({
                [name]: dynamic.value,
            });
        }
        // Short circuit type checking if value is an expression
        if (typeof value === "string" && value.includes("${")) {
            return `${name} = ${renderString(value)}`;
        }
        if (block && type !== "list" && type !== "set") {
            return `${name} {
${renderAttributes(value)}
}`;
        }
        if (type === "list" || type === "set") {
            if (block) {
                return renderList(v, name);
            }
            return `${name} = ${renderList(v)}`;
        }
        if (type === "map") {
            return `${name} = ${renderMap(value)}`;
        }
        if (type === "simple") {
            if (classType === "string") {
                return `${name} = ${renderString(value)}`;
            }
            if (classType === "number" || classType === "boolean") {
                return `${name} = ${value}`;
            }
            // Not sure why we're here, but there's either a bug in the provider
            // or we have skipped the attribute to reduce the size of our provider
            // In either case, we should try to not output [object Object] here
            // and try a best approximation. Though, it would not work for
            // blocks
            if (typeof value === "object") {
                return `
# Warning: The following attribute is of an unexpected type. Either there's a problem with the provider
# or CDKTF has chosen to skip generating a detailed type for this because it increases the size of the provider library significantly. 
# Please check if this resource is included in our skip list: 
# https://github.com/hashicorp/terraform-cdk/blob/1fb8588095a55d475dc4be28882ec4f42006ec8d/packages/%40cdktf/provider-generator/lib/get/generator/skipped-attributes.ts
# and if not, please check with the provider authors to see if the provider schema is accurate.
# 
# We understand this is not ideal, so we suggest using JSON synthesis instead of HCL synthesis for this particular case.

${name} = ${renderFuzzyJsonExpression(value)}
`;
            }
        }
        if (type === "any") {
            return `${name} = ${renderFuzzyJsonExpression(value)}`;
        }
        if (type === "reference") {
            return `${name} = ${value}`;
        }
        return `${name} = ${value}`;
    })
        .filter((v) => v !== undefined)
        .join("\n");
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVuZGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmVuZGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBb0RBLDRDQTZDQztBQUtELGdDQWdEQztBQUtELDBDQVNDO0FBS0QsOEJBY0M7QUFLRCx3Q0FzQkM7QUFLRCx3Q0FxQ0M7QUFLRCw0Q0EyQkM7QUFLRCx3REF5QkM7QUFLRCxrREFxQkM7QUFLRCx3Q0FtQkM7QUFLRCxvQ0FnQkM7QUFLRCxvQ0FPQztBQUtELG9DQW9CQztBQUtELHdDQU9DO0FBS0Qsa0NBUUM7QUFLRCxvQ0FRQztBQUtELDBDQTBCQztBQUtELGtDQUlDO0FBeUVELHdEQVdDO0FBSUQsNENBc0hDO0FBbHNCRDs7O0dBR0c7QUFDSCwyQ0FBK0M7QUFFL0M7O0dBRUc7QUFDSCxTQUFTLFlBQVksQ0FBQyxHQUFXO0lBQy9CLG1EQUFtRDtJQUNuRCxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQztRQUN0QixPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFDRCxPQUFPLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBQ3pDLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsOEJBQThCLENBQUMsR0FBVztJQUNqRCxPQUFPLHNCQUFzQixDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxHQUFHLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO0FBQzdELENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMsWUFBWSxDQUFDLEdBQVc7SUFDL0IsSUFBSSxHQUFHLEtBQUssRUFBRSxFQUFFLENBQUM7UUFDZixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRCxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDVCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7SUFFRCxJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsRUFBRSxDQUFDO1FBQzVCLE1BQU0sSUFBSSxLQUFLLENBQ2IsdUVBQXVFLENBQ3hFLENBQUM7SUFDSixDQUFDO0lBRUQsTUFBTSxLQUFLLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUV2QyxJQUFJLEtBQUssQ0FBQyxNQUFNLEtBQUssQ0FBQztRQUFFLE9BQU8sSUFBSSxZQUFZLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQztJQUV4RCxPQUFPLFVBQVUsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDO0FBQzNDLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLGdCQUFnQixDQUFDLEtBQVU7SUFDekMsTUFBTSxHQUFHLEdBQTRCLEVBQUUsQ0FBQztJQUV4QyxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUN2RCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7SUFFRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUN6QixPQUFPLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDL0MsQ0FBQztJQUVELE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFaEMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7UUFDM0IsT0FBTyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVELEtBQUssTUFBTSxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7UUFDdkIsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3pCLElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxJQUFJLEtBQUssSUFBSSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDeEUsZ0VBQWdFO1lBQ2hFLElBQUksb0JBQVksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztnQkFDckMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEVBQUUsQ0FBQztnQkFDZCxTQUFTO1lBQ1gsQ0FBQztZQUVELEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNuQyxTQUFTO1FBQ1gsQ0FBQzthQUFNLElBQ0wsT0FBTyxLQUFLLEtBQUssUUFBUTtZQUN6QixLQUFLLElBQUksSUFBSTtZQUNiLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQ3BCLENBQUM7WUFDRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsQ0FBQztnQkFDNUIsR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUUsR0FBRyxLQUFLLENBQUMsQ0FBQztZQUNyQyxDQUFDO2lCQUFNLENBQUM7Z0JBQ04sR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztZQUNuQixDQUFDO1FBQ0gsQ0FBQzthQUFNLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRSxDQUFDO1lBQy9CLE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2xCLENBQUM7YUFBTSxDQUFDO1lBQ04sR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLEtBQUssQ0FBQztRQUNuQixDQUFDO0lBQ0gsQ0FBQztJQUNELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsVUFBVSxDQUFDLENBQU0sRUFBRSxJQUFhO0lBQzlDLE1BQU0sRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxrQkFBa0IsRUFBRSxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUM3RSxNQUFNLEtBQUssR0FBRyxPQUFPLElBQUksUUFBUSxDQUFDO0lBQ2xDLE1BQU0sU0FBUyxHQUFHLGdCQUFnQixJQUFJLGtCQUFrQixDQUFDO0lBRXpELElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRSxDQUFDO1FBQ3hCLE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztJQUVELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxFQUFFLENBQUM7UUFDOUIsb0VBQW9FO1FBQ3BFLE9BQU8sWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFFRCxJQUFJLElBQUksRUFBRSxDQUFDO1FBQ1QsSUFBSSxLQUFLLEVBQUUsQ0FBQztZQUNWLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7Z0JBQzFCLE9BQU8sR0FBRyxJQUFJLElBQUksR0FBRyxLQUFLLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUM7WUFDekQsQ0FBQztZQUVELE9BQU8sS0FBSztpQkFDVCxHQUFHLENBQUMsQ0FBQyxDQUFNLEVBQUUsRUFBRTtnQkFDZCxPQUFPLEdBQUcsSUFBSSxJQUFJLEdBQUcsS0FBSyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDO1lBQ3JELENBQUMsQ0FBQztpQkFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDaEIsQ0FBQztRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDMUIsT0FBTyxHQUFHLElBQUksTUFBTSxlQUFlLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxFQUFFLENBQUM7UUFDMUQsQ0FBQztRQUVELE9BQU8sR0FBRyxJQUFJO0VBQ2hCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFNLEVBQUUsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDLEVBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDO0VBQ2hFLENBQUM7SUFDRCxDQUFDO0lBRUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUMxQixvQ0FBb0M7UUFDcEMsT0FBTyxlQUFlLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFRCxJQUFJLEtBQUssRUFBRSxDQUFDO1FBQ1YsT0FBTyxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBTSxFQUFFLEVBQUUsQ0FBQyxlQUFlLENBQUMsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRCxPQUFPO0VBQ1AsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQU0sRUFBRSxFQUFFLENBQUMsZUFBZSxDQUFDLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUM7RUFDaEUsQ0FBQztBQUNILENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLGVBQWUsQ0FBQyxLQUFVLEVBQUUsZ0JBQXdCO0lBQ2xFLElBQUksZ0JBQWdCLEtBQUssWUFBWSxFQUFFLENBQUM7UUFDdEMsT0FBTyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0IsQ0FBQztJQUNELElBQUksZ0JBQWdCLEtBQUssWUFBWSxJQUFJLGdCQUFnQixLQUFLLGFBQWEsRUFBRSxDQUFDO1FBQzVFLE9BQU8sR0FBRyxLQUFLLEVBQUUsQ0FBQztJQUNwQixDQUFDO0lBRUQsT0FBTyxXQUFXLENBQUMsS0FBSyxFQUFFLGdCQUFnQixDQUFDLENBQUM7QUFDOUMsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsU0FBUyxDQUFDLEdBQVE7SUFDaEMsSUFBSSxPQUFPLEdBQUcsS0FBSyxRQUFRLEVBQUUsQ0FBQztRQUM1QixvRUFBb0U7UUFDcEUsT0FBTyxJQUFJLFlBQVksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDO0lBQ2xDLENBQUM7SUFDRCxPQUFPO0VBQ1AsTUFBTSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7U0FDbEIsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRTtRQUNkLE1BQU0sVUFBVSxHQUFHLDhCQUE4QixDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXJELE9BQU8sR0FBRyxVQUFVLE1BQU0sY0FBYyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFDaEQsQ0FBQyxDQUFDO1NBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQztFQUNYLENBQUM7QUFDSCxDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixjQUFjLENBQUMsS0FBVTtJQUN2QyxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRSxDQUFDO1FBQzlCLE9BQU8sWUFBWSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzdCLENBQUM7SUFDRCxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRSxDQUFDO1FBQzlCLE9BQU8sR0FBRyxLQUFLLEVBQUUsQ0FBQztJQUNwQixDQUFDO0lBQ0QsSUFBSSxPQUFPLEtBQUssS0FBSyxTQUFTLEVBQUUsQ0FBQztRQUMvQixPQUFPLEdBQUcsS0FBSyxFQUFFLENBQUM7SUFDcEIsQ0FBQztJQUVELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxFQUFFLENBQUM7UUFDOUIsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7WUFDekIsT0FBTyxlQUFlLENBQUMsS0FBSyxFQUFFLHFCQUFxQixDQUFDLENBQUM7UUFDdkQsQ0FBQztRQUNELElBQUksS0FBSyxLQUFLLElBQUksRUFBRSxDQUFDO1lBQ25CLE9BQU8sTUFBTSxDQUFDO1FBQ2hCLENBQUM7UUFDRCxPQUFPLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMxQixDQUFDO0lBRUQsT0FBTyxHQUFHLEtBQUssRUFBRSxDQUFDO0FBQ3BCLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLGNBQWMsQ0FBQyxRQUFhO0lBQzFDLE1BQU0sWUFBWSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDOUMsTUFBTSxpQkFBaUIsR0FBRyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDakQsTUFBTSxZQUFZLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3ZELE1BQU0sa0JBQWtCLEdBQUcsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUM7SUFFM0QsSUFBSSxrQkFBa0IsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUNsQyxrQkFBa0IsQ0FBQyxVQUFVLEdBQUc7WUFDOUIsS0FBSyxFQUFFLGtCQUFrQixDQUFDLFVBQVU7WUFDcEMsT0FBTyxFQUFFLElBQUk7U0FDZCxDQUFDO0lBQ0osQ0FBQztJQUVELE1BQU0sRUFBRSxXQUFXLEVBQUUsT0FBTyxFQUFFLEdBQUcsVUFBVSxFQUFFLEdBQUcsa0JBQWtCLENBQUM7SUFFbkUsTUFBTSxHQUFHLEdBQUcsQ0FBQyxhQUFhLFlBQVksTUFBTSxZQUFZLEtBQUssQ0FBQyxDQUFDO0lBRS9ELE1BQU0sS0FBSyxHQUFHLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBRTNDLElBQUksS0FBSztRQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0IsSUFBSSxXQUFXO1FBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO0lBQy9ELElBQUksT0FBTztRQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBRXZELEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFZCxPQUFPO1FBQ0wsR0FBRyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1FBQ25CLFFBQVEsRUFBRTtZQUNSLFFBQVEsRUFBRTtnQkFDUixDQUFDLFlBQVksQ0FBQyxFQUFFO29CQUNkLENBQUMsWUFBWSxDQUFDLEVBQUU7d0JBQ2QsSUFBSSxFQUFFLGtCQUFrQixDQUFDLElBQUksQ0FBQztxQkFDL0I7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQztBQUNKLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLGdCQUFnQixDQUFDLFVBQWU7SUFDOUMsTUFBTSxjQUFjLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNsRCxNQUFNLG1CQUFtQixHQUFHLFVBQVUsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUN2RCxNQUFNLGNBQWMsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLG1CQUFtQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDM0QsTUFBTSxvQkFBb0IsR0FBRyxtQkFBbUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUVqRSxNQUFNLEVBQUUsT0FBTyxFQUFFLEdBQUcsVUFBVSxFQUFFLEdBQUcsb0JBQW9CLENBQUM7SUFFeEQsTUFBTSxHQUFHLEdBQUcsQ0FBQyxTQUFTLGNBQWMsTUFBTSxjQUFjLEtBQUssQ0FBQyxDQUFDO0lBRS9ELE1BQU0sS0FBSyxHQUFHLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQzNDLElBQUksS0FBSztRQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0IsSUFBSSxPQUFPO1FBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLG1CQUFtQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFDdkQsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUVkLE9BQU87UUFDTCxHQUFHLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDbkIsUUFBUSxFQUFFO1lBQ1IsSUFBSSxFQUFFO2dCQUNKLENBQUMsY0FBYyxDQUFDLEVBQUU7b0JBQ2hCLENBQUMsY0FBYyxDQUFDLEVBQUU7d0JBQ2hCLElBQUksRUFBRSxvQkFBb0IsQ0FBQyxJQUFJLENBQUM7cUJBQ2pDO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUM7QUFDSixDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixzQkFBc0IsQ0FBQyxZQUFpQjtJQUN0RCxPQUFPLFlBQVk7U0FDaEIsR0FBRyxDQUFDLENBQUMsV0FBZ0IsRUFBRSxFQUFFO1FBQ3hCLE1BQU0sZUFBZSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEQsTUFBTSxnQkFBZ0IsR0FBRyxXQUFXLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFdEQsSUFBSSxnQkFBZ0IsQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNoQyxnQkFBZ0IsQ0FBQyxVQUFVLEdBQUc7Z0JBQzVCLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxVQUFVO2dCQUNsQyxPQUFPLEVBQUUsSUFBSTthQUNkLENBQUM7UUFDSixDQUFDO1FBRUQsTUFBTSxFQUFFLE9BQU8sRUFBRSxHQUFHLFVBQVUsRUFBRSxHQUFHLGdCQUFnQixDQUFDO1FBRXBELE1BQU0sR0FBRyxHQUFHLENBQUMsZ0JBQWdCLGVBQWUsS0FBSyxDQUFDLENBQUM7UUFFbkQsTUFBTSxLQUFLLEdBQUcsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLEtBQUssSUFBSSxVQUFVLENBQUMsQ0FBQztRQUMvRCxJQUFJLEtBQUs7WUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNCLElBQUksT0FBTztZQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFZCxPQUFPLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDeEIsQ0FBQyxDQUFDO1NBQ0QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQ2xCLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLG1CQUFtQixDQUFDLE9BQVk7SUFDOUMsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsQ0FDaEMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxZQUFZLENBQWdCLEVBQUUsRUFBRTtRQUM3QyxNQUFNLEdBQUcsR0FBRyxDQUFDLFlBQVksV0FBVyxLQUFLLENBQUMsQ0FBQztRQUMzQyxHQUFHLENBQUMsSUFBSSxDQUNOLGNBQWMseUJBQXlCLENBQUMsWUFBWSxFQUFFLFFBQVEsQ0FBQyxFQUFFLENBQ2xFLENBQUM7UUFDRixJQUFJLFlBQVksRUFBRSxRQUFRLEVBQUUsQ0FBQztZQUMzQixHQUFHLENBQUMsSUFBSSxDQUFDLGNBQWMsWUFBWSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDbkQsQ0FBQztRQUNELElBQUksWUFBWSxFQUFFLE1BQU0sRUFBRSxDQUFDO1lBQ3pCLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBWSx5QkFBeUIsQ0FBQyxZQUFZLEVBQUUsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQzFFLENBQUM7UUFDRCxHQUFHLENBQUMsSUFBSSxDQUFDO0lBQ1gsZ0JBQWdCLENBQUMsWUFBWSxDQUFDLE9BQU8sQ0FBQztJQUN0QyxDQUFDLENBQUM7UUFDQSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBRWQsT0FBTyxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3hCLENBQUMsQ0FDRixDQUFDO0FBQ0osQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsY0FBYyxDQUFDLFFBQWE7SUFDMUMsTUFBTSxZQUFZLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM5QyxNQUFNLGtCQUFrQixHQUFHLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUVsRCxPQUFPLGtCQUFrQjtTQUN0QixHQUFHLENBQUMsQ0FBQyxnQkFBcUIsRUFBRSxFQUFFO1FBQzdCLE1BQU0sRUFBRSxPQUFPLEVBQUUsR0FBRyxVQUFVLEVBQUUsR0FBRyxnQkFBZ0IsQ0FBQztRQUVwRCxNQUFNLEdBQUcsR0FBRyxDQUFDLGFBQWEsWUFBWSxLQUFLLENBQUMsQ0FBQztRQUU3QyxNQUFNLEtBQUssR0FBRyxnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMzQyxJQUFJLEtBQUs7WUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzNCLElBQUksT0FBTztZQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxtQkFBbUIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBRXZELEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7UUFFZCxPQUFPLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDeEIsQ0FBQyxDQUFDO1NBQ0QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0FBQ2xCLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLFlBQVksQ0FBQyxNQUFXO0lBQ3RDLE1BQU0sVUFBVSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDMUMsTUFBTSxnQkFBZ0IsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUM7SUFFNUMsT0FBTztRQUNMLEdBQUcsRUFBRSxXQUFXLFVBQVU7RUFDNUIsZ0JBQWdCLENBQUMsZ0JBQWdCLENBQUM7RUFDbEM7UUFDRSxRQUFRLEVBQUU7WUFDUixNQUFNLEVBQUU7Z0JBQ04sQ0FBQyxVQUFVLENBQUMsRUFBRTtvQkFDWixJQUFJLEVBQUUsZ0JBQWdCLENBQUMsSUFBSSxDQUFDO2lCQUM3QjthQUNGO1NBQ0Y7S0FDRixDQUFDO0FBQ0osQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsWUFBWSxDQUFDLE1BQVc7SUFDdEMsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMxQyxNQUFNLGdCQUFnQixHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUU1QyxPQUFPLFdBQVcsVUFBVTtFQUM1QixnQkFBZ0IsQ0FBQyxnQkFBZ0IsQ0FBQztFQUNsQyxDQUFDO0FBQ0gsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsWUFBWSxDQUFDLE1BQVc7SUFDdEMsSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBQ1osT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0lBRUQsTUFBTSxVQUFVLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUV2QyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFLENBQUM7UUFDNUIsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0lBRUQsT0FBTztNQUNILFVBQVU7U0FDVCxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsS0FBSyxDQUFDO1NBQ3RDLEdBQUcsQ0FBQyxDQUFDLElBQVksRUFBRSxFQUFFO1FBQ3BCLE1BQU0sS0FBSyxHQUFHLHlCQUF5QixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM1RCxPQUFPLEdBQUcsSUFBSSxNQUFNLEtBQUssRUFBRSxDQUFDO0lBQzlCLENBQUMsQ0FBQztTQUNELElBQUksQ0FBQyxJQUFJLENBQUM7RUFDZixDQUFDO0FBQ0gsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsY0FBYyxDQUFDLFFBQWE7SUFDMUMsTUFBTSxZQUFZLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUM5QyxNQUFNLGtCQUFrQixHQUFHLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUVsRCxPQUFPLGFBQWEsWUFBWTtFQUNoQyxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQztFQUNwQyxDQUFDO0FBQ0gsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsV0FBVyxDQUFDLElBQVM7SUFDbkMsTUFBTSxXQUFXLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFNBQWMsRUFBRSxFQUFFO1FBQzlDLE9BQU87RUFDVCxnQkFBZ0IsQ0FBQyxTQUFTLENBQUM7RUFDM0IsQ0FBQztJQUNELENBQUMsQ0FBQyxDQUFDO0lBRUgsT0FBTyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ2hDLENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLFlBQVksQ0FBQyxPQUFZO0lBQ3ZDLE1BQU0sWUFBWSxHQUFHLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxXQUFnQixFQUFFLEVBQUU7UUFDcEQsT0FBTztFQUNULGdCQUFnQixDQUFDLFdBQVcsQ0FBQztFQUM3QixDQUFDO0lBQ0QsQ0FBQyxDQUFDLENBQUM7SUFFSCxPQUFPLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDakMsQ0FBQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsZUFBZSxDQUFDLFNBQWM7SUFDNUMsTUFBTSxlQUFlLEdBQUcsQ0FBQyxvQkFBb0IsRUFBRSxTQUFTLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDbkUsTUFBTSxpQkFBaUIsR0FBRztFQUMxQixzQkFBc0IsQ0FBQyxTQUFTLENBQUMsa0JBQWtCLENBQUM7RUFDcEQsQ0FBQztJQUVELE1BQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sSUFBSSxFQUFFLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRTtRQUN4RSxPQUFPLFlBQVksV0FBVztFQUNoQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDO0VBQ2hELENBQUM7SUFDRCxDQUFDLENBQUMsQ0FBQztJQUVILE1BQU0sWUFBWSxHQUFHLFNBQVMsQ0FBQyxLQUFLO1FBQ2xDLENBQUMsQ0FBQztFQUNKLGdCQUFnQixDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUM7RUFDakM7UUFDRSxDQUFDLENBQUMsU0FBUyxDQUFDO0lBRWQsTUFBTSxlQUFlLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxNQUFNLENBQ25ELENBQUMsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQ3hDLENBQUM7SUFDRixPQUFPO0VBQ1AsaUJBQWlCO0VBQ2pCLENBQUMsR0FBRyxRQUFRLEVBQUUsWUFBWSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztFQUN0QyxzQkFBc0IsQ0FBQyxlQUFlLENBQUM7RUFDdkMsQ0FBQztBQUNILENBQUM7QUFFRDs7R0FFRztBQUNILFNBQWdCLFdBQVcsQ0FBQyxLQUFVLEVBQUUsaUJBQXlCO0lBQy9ELE9BQU87RUFDUCxnQkFBZ0IsQ0FBQyxLQUFLLENBQUM7RUFDdkIsQ0FBQztBQUNILENBQUM7QUFFRDs7R0FFRztBQUNILFNBQVMscUJBQXFCLENBQUMsVUFBZTtJQUM1QyxPQUFPO1FBQ0wsR0FBRztRQUNILEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsRUFBRSxFQUFFO1lBQ2xELE1BQU0sVUFBVSxHQUFHLDhCQUE4QixDQUFDLElBQUksQ0FBQyxDQUFDO1lBQ3hELE9BQU8sR0FBRyxVQUFVLE1BQU0seUJBQXlCLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUMvRCxDQUFDLENBQUM7UUFDRixHQUFHO0tBQ0osQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7QUFDZixDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLHlCQUF5QixDQUFDLGNBQW1CO0lBQ3BELElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxjQUFjLENBQUMsRUFBRSxDQUFDO1FBQ2xDLE9BQU87WUFDTCxHQUFHO1lBQ0gsR0FBRyxjQUFjLENBQUMsR0FBRyxDQUFDLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxHQUFHLHlCQUF5QixDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUM7WUFDeEUsR0FBRztTQUNKLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2YsQ0FBQztJQUVELElBQUksT0FBTyxjQUFjLEtBQUssUUFBUSxFQUFFLENBQUM7UUFDdkMsT0FBTyxxQkFBcUIsQ0FBQyxjQUFjLENBQUMsQ0FBQztJQUMvQyxDQUFDO0lBRUQsSUFBSSxPQUFPLGNBQWMsS0FBSyxRQUFRLEVBQUUsQ0FBQztRQUN2QyxJQUFJLGNBQWMsS0FBSyxFQUFFLEVBQUUsQ0FBQztZQUMxQixPQUFPLElBQUksQ0FBQztRQUNkLENBQUM7UUFFRCxJQUFJLGNBQWMsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUNsQyxPQUFPLElBQUksY0FBYyxHQUFHLENBQUM7UUFDL0IsQ0FBQztRQUVELElBQUksUUFBUSxDQUFDLGNBQWMsRUFBRSxFQUFFLENBQUMsQ0FBQyxRQUFRLEVBQUUsS0FBSyxjQUFjLEVBQUUsQ0FBQztZQUMvRCxPQUFPLGNBQWMsQ0FBQztRQUN4QixDQUFDO1FBRUQsSUFBSSxjQUFjLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUM7WUFDbkMsT0FBTyxjQUFjLENBQUM7UUFDeEIsQ0FBQztRQUVELElBQUksY0FBYyxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQ25DLE9BQU8sY0FBYyxDQUFDO1FBQ3hCLENBQUM7UUFFRCxJQUFJLGNBQWMsQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUNuQyxPQUFPLGNBQWMsQ0FBQztRQUN4QixDQUFDO1FBRUQsT0FBTyxJQUFJLFlBQVksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDO0lBQzdDLENBQUM7SUFFRCxJQUFJLGNBQWMsS0FBSyxNQUFNLElBQUksY0FBYyxLQUFLLE9BQU8sRUFBRSxDQUFDO1FBQzVELE9BQU8sY0FBd0IsQ0FBQztJQUNsQyxDQUFDO0lBRUQsSUFBSSxjQUFjLEtBQUssTUFBTSxFQUFFLENBQUM7UUFDOUIsT0FBTyxjQUFjLENBQUM7SUFDeEIsQ0FBQztJQUVELE9BQU8sR0FBRyxjQUFjLEVBQUUsQ0FBQztBQUM3QixDQUFDO0FBRUQ7O0dBRUc7QUFDSCxTQUFnQixzQkFBc0IsQ0FBQyxVQUFlO0lBQ3BELElBQUksVUFBVSxLQUFLLFNBQVMsRUFBRSxDQUFDO1FBQzdCLE9BQU8sRUFBRSxDQUFDO0lBQ1osQ0FBQztJQUVELE9BQU8sTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7U0FDOUIsR0FBRyxDQUNGLENBQUMsQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLEVBQUUsRUFBRSxDQUNoQixLQUFLLElBQUksTUFBTSx5QkFBeUIsQ0FBQyxLQUFZLENBQUMsRUFBRSxDQUMzRDtTQUNBLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUNoQixDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFnQixnQkFBZ0IsQ0FBQyxVQUFlO0lBQzlDLElBQUksVUFBVSxLQUFLLFNBQVMsSUFBSSxVQUFVLEtBQUssSUFBSSxFQUFFLENBQUM7UUFDcEQsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0lBQ0QsT0FBTyxNQUFNLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQztTQUM5QixHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsRUFBRSxFQUFFO1FBQ2pCLElBQUksSUFBSSxLQUFLLElBQUksRUFBRSxDQUFDO1lBQ2xCLE9BQU8sU0FBUyxDQUFDO1FBQ25CLENBQUM7UUFDRCxNQUFNLFVBQVUsR0FBRyxDQUFDLFdBQVcsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUVoRCxFQUFFO1FBQ0YsaUVBQWlFO1FBQ2pFLHlCQUF5QjtRQUN6QixJQUFJLE9BQU8sQ0FBQyxLQUFLLFFBQVEsSUFBSSxPQUFPLENBQUMsS0FBSyxRQUFRLEVBQUUsQ0FBQztZQUNuRCxPQUFPLEdBQUcsSUFBSSxNQUFNLHlCQUF5QixDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDckQsQ0FBQzthQUFNLElBQUksT0FBTyxDQUFDLEtBQUssU0FBUyxFQUFFLENBQUM7WUFDbEMsT0FBTyxHQUFHLElBQUksTUFBTSxDQUFDLEVBQUUsQ0FBQztRQUMxQixDQUFDO2FBQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDNUIsT0FBTyxHQUFHLElBQUksTUFBTSx5QkFBeUIsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3JELENBQUM7YUFBTSxJQUFJLENBQUMsS0FBSyxJQUFJLEVBQUUsQ0FBQztZQUN0QixPQUFPLEdBQUcsSUFBSSxTQUFTLENBQUM7UUFDMUIsQ0FBQzthQUFNLElBQ0wsT0FBTyxDQUFDLEtBQUssUUFBUTtZQUNyQixpREFBaUQ7WUFDakQsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQztZQUMxQixpREFBaUQ7WUFDakQsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLFNBQVMsQ0FBQyxFQUM1QixDQUFDO1lBQ0QsSUFBSSxVQUFVLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7Z0JBQzlCLE9BQU8sR0FBRyxJQUFJO0VBQ3RCLHNCQUFzQixDQUFDLENBQUMsQ0FBQztFQUN6QixDQUFDO1lBQ0ssQ0FBQztZQUNELE9BQU8sR0FBRyxJQUFJLE1BQU0seUJBQXlCLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNyRCxDQUFDO2FBQU0sSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFLENBQUM7WUFDM0IsT0FBTyxTQUFTLENBQUM7UUFDbkIsQ0FBQztRQUVELCtGQUErRjtRQUMvRixxQkFBcUI7UUFDckIsTUFBTSxFQUNKLE9BQU8sRUFDUCxLQUFLLEVBQ0wsSUFBSSxFQUNKLE9BQU8sRUFDUCxnQkFBZ0IsRUFDaEIsUUFBUSxFQUNSLGtCQUFrQixHQUNuQixHQUFHLENBQVEsQ0FBQztRQUNiLE1BQU0sS0FBSyxHQUFHLE9BQU8sSUFBSSxRQUFRLElBQUksVUFBVSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMvRCxNQUFNLFNBQVMsR0FBRyxnQkFBZ0IsSUFBSSxrQkFBa0IsQ0FBQztRQUV6RCxJQUFJLE9BQU8sRUFBRSxDQUFDO1lBQ1osT0FBTyxtQkFBbUIsQ0FBQztnQkFDekIsQ0FBQyxJQUFJLENBQUMsRUFBRSxPQUFPLENBQUMsS0FBSzthQUN0QixDQUFDLENBQUM7UUFDTCxDQUFDO1FBQ0Qsd0RBQXdEO1FBQ3hELElBQUksT0FBTyxLQUFLLEtBQUssUUFBUSxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUN0RCxPQUFPLEdBQUcsSUFBSSxNQUFNLFlBQVksQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQzVDLENBQUM7UUFFRCxJQUFJLEtBQUssSUFBSSxJQUFJLEtBQUssTUFBTSxJQUFJLElBQUksS0FBSyxLQUFLLEVBQUUsQ0FBQztZQUMvQyxPQUFPLEdBQUcsSUFBSTtFQUNwQixnQkFBZ0IsQ0FBQyxLQUFLLENBQUM7RUFDdkIsQ0FBQztRQUNHLENBQUM7UUFDRCxJQUFJLElBQUksS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLEtBQUssRUFBRSxDQUFDO1lBQ3RDLElBQUksS0FBSyxFQUFFLENBQUM7Z0JBQ1YsT0FBTyxVQUFVLENBQUMsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO1lBQzdCLENBQUM7WUFDRCxPQUFPLEdBQUcsSUFBSSxNQUFNLFVBQVUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3RDLENBQUM7UUFDRCxJQUFJLElBQUksS0FBSyxLQUFLLEVBQUUsQ0FBQztZQUNuQixPQUFPLEdBQUcsSUFBSSxNQUFNLFNBQVMsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ3pDLENBQUM7UUFFRCxJQUFJLElBQUksS0FBSyxRQUFRLEVBQUUsQ0FBQztZQUN0QixJQUFJLFNBQVMsS0FBSyxRQUFRLEVBQUUsQ0FBQztnQkFDM0IsT0FBTyxHQUFHLElBQUksTUFBTSxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUM1QyxDQUFDO1lBQ0QsSUFBSSxTQUFTLEtBQUssUUFBUSxJQUFJLFNBQVMsS0FBSyxTQUFTLEVBQUUsQ0FBQztnQkFDdEQsT0FBTyxHQUFHLElBQUksTUFBTSxLQUFLLEVBQUUsQ0FBQztZQUM5QixDQUFDO1lBRUQsb0VBQW9FO1lBQ3BFLHNFQUFzRTtZQUN0RSxtRUFBbUU7WUFDbkUsOERBQThEO1lBQzlELFNBQVM7WUFDVCxJQUFJLE9BQU8sS0FBSyxLQUFLLFFBQVEsRUFBRSxDQUFDO2dCQUM5QixPQUFPOzs7Ozs7Ozs7RUFTZixJQUFJLE1BQU0seUJBQXlCLENBQUMsS0FBSyxDQUFDO0NBQzNDLENBQUM7WUFDTSxDQUFDO1FBQ0gsQ0FBQztRQUVELElBQUksSUFBSSxLQUFLLEtBQUssRUFBRSxDQUFDO1lBQ25CLE9BQU8sR0FBRyxJQUFJLE1BQU0seUJBQXlCLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUN6RCxDQUFDO1FBRUQsSUFBSSxJQUFJLEtBQUssV0FBVyxFQUFFLENBQUM7WUFDekIsT0FBTyxHQUFHLElBQUksTUFBTSxLQUFLLEVBQUUsQ0FBQztRQUM5QixDQUFDO1FBRUQsT0FBTyxHQUFHLElBQUksTUFBTSxLQUFLLEVBQUUsQ0FBQztJQUM5QixDQUFDLENBQUM7U0FDRCxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsS0FBSyxTQUFTLENBQUM7U0FDOUIsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0FBQ2hCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmMuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTVBMLTIuMFxuICovXG5pbXBvcnQgeyBUb2tlbml6YXRpb24gfSBmcm9tIFwiLi4vdG9rZW5zL3Rva2VuXCI7XG5cbi8qKlxuICpcbiAqL1xuZnVuY3Rpb24gZXNjYXBlUXVvdGVzKHN0cjogc3RyaW5nKTogc3RyaW5nIHtcbiAgLy8gRG9uJ3QgYXR0ZW1wdCB0byBlc2NhcGUgaWYgd2UgaGF2ZSBhbiBleHByZXNzaW9uXG4gIGlmIChzdHIubWF0Y2goL1xcJFxcey8pKSB7XG4gICAgcmV0dXJuIHN0cjtcbiAgfVxuICByZXR1cm4gc3RyLnJlcGxhY2UoLyg/PCFcXFxcKVwiL2csICdcXFxcXCInKTtcbn1cblxuLyoqXG4gKlxuICovXG5mdW5jdGlvbiB3cmFwSWRlbnRpZmllckluUXVvdGVzSWZOZWVkZWQoa2V5OiBzdHJpbmcpOiBzdHJpbmcge1xuICByZXR1cm4gLyheXFxkKXxbXkEtWmEtejAtOV8tXS8udGVzdChrZXkpID8gYFwiJHtrZXl9XCJgIDoga2V5O1xufVxuXG4vKipcbiAqXG4gKi9cbmZ1bmN0aW9uIHJlbmRlclN0cmluZyhzdHI6IHN0cmluZyk6IHN0cmluZyB7XG4gIGlmIChzdHIgPT09IFwiXCIpIHtcbiAgICByZXR1cm4gYFwiXCJgO1xuICB9XG5cbiAgaWYgKCFzdHIpIHtcbiAgICByZXR1cm4gc3RyO1xuICB9XG5cbiAgaWYgKHR5cGVvZiBzdHIgIT09IFwic3RyaW5nXCIpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICBcIlVuYWJsZSB0byBwcm9jZXNzIGF0dHJpYnV0ZSB0aGF0IHNob3VsZCBoYXZlIGJlZW4gYSBzdHJpbmcsIGJ1dCBpc24ndFwiLFxuICAgICk7XG4gIH1cblxuICBjb25zdCBsaW5lcyA9IHN0ci5zcGxpdCgvXFxyXFxufFtcXG5cXHJdLyk7XG5cbiAgaWYgKGxpbmVzLmxlbmd0aCA9PT0gMSkgcmV0dXJuIGBcIiR7ZXNjYXBlUXVvdGVzKHN0cil9XCJgO1xuXG4gIHJldHVybiBgPDxFT0ZcXG4ke2xpbmVzLmpvaW4oXCJcXG5cIil9XFxuRU9GYDtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gY2xlYW5Gb3JNZXRhZGF0YShibG9jazogYW55KTogYW55IHtcbiAgY29uc3QgcmVzOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfSA9IHt9O1xuXG4gIGlmICh0eXBlb2YgYmxvY2sgIT09IFwib2JqZWN0XCIgJiYgIUFycmF5LmlzQXJyYXkoYmxvY2spKSB7XG4gICAgcmV0dXJuIGJsb2NrO1xuICB9XG5cbiAgaWYgKEFycmF5LmlzQXJyYXkoYmxvY2spKSB7XG4gICAgcmV0dXJuIGJsb2NrLm1hcCgoaSkgPT4gY2xlYW5Gb3JNZXRhZGF0YShpKSk7XG4gIH1cblxuICBjb25zdCBrZXlzID0gT2JqZWN0LmtleXMoYmxvY2spO1xuXG4gIGlmIChrZXlzLmluY2x1ZGVzKFwidmFsdWVcIikpIHtcbiAgICByZXR1cm4gY2xlYW5Gb3JNZXRhZGF0YShibG9jay52YWx1ZSk7XG4gIH1cblxuICBmb3IgKGNvbnN0IGtleSBvZiBrZXlzKSB7XG4gICAgY29uc3QgdmFsdWUgPSBibG9ja1trZXldO1xuICAgIGlmICh0eXBlb2YgdmFsdWUgPT09IFwib2JqZWN0XCIgJiYgdmFsdWUgIT0gbnVsbCAmJiAhQXJyYXkuaXNBcnJheSh2YWx1ZSkpIHtcbiAgICAgIC8vIGlmIHRoZSB2YWx1ZSBpcyBhIHJlc29sdmFibGUgd2UgZG9uJ3Qgd2FudCB0byByZWN1cnNlIGludG8gaXRcbiAgICAgIGlmIChUb2tlbml6YXRpb24uaXNSZXNvbHZhYmxlKHZhbHVlKSkge1xuICAgICAgICByZXNba2V5XSA9IHt9O1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cblxuICAgICAgcmVzW2tleV0gPSBjbGVhbkZvck1ldGFkYXRhKHZhbHVlKTtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH0gZWxzZSBpZiAoXG4gICAgICB0eXBlb2YgdmFsdWUgPT09IFwib2JqZWN0XCIgJiZcbiAgICAgIHZhbHVlICE9IG51bGwgJiZcbiAgICAgIEFycmF5LmlzQXJyYXkodmFsdWUpXG4gICAgKSB7XG4gICAgICBpZiAoQXJyYXkuaXNBcnJheShyZXNba2V5XSkpIHtcbiAgICAgICAgcmVzW2tleV0gPSBbLi4ucmVzW2tleV0sIC4uLnZhbHVlXTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHJlc1trZXldID0gdmFsdWU7XG4gICAgICB9XG4gICAgfSBlbHNlIGlmICh2YWx1ZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICBkZWxldGUgcmVzW2tleV07XG4gICAgfSBlbHNlIHtcbiAgICAgIHJlc1trZXldID0gdmFsdWU7XG4gICAgfVxuICB9XG4gIHJldHVybiByZXM7XG59XG5cbi8qKlxuICpcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlbmRlckxpc3QodjogYW55LCBuYW1lPzogc3RyaW5nKTogc3RyaW5nIHtcbiAgY29uc3QgeyBpc0Jsb2NrLCBpc19ibG9jaywgdmFsdWUsIHN0b3JhZ2VfY2xhc3NfdHlwZSwgc3RvcmFnZUNsYXNzVHlwZSB9ID0gdjtcbiAgY29uc3QgYmxvY2sgPSBpc0Jsb2NrIHx8IGlzX2Jsb2NrO1xuICBjb25zdCBjbGFzc1R5cGUgPSBzdG9yYWdlQ2xhc3NUeXBlIHx8IHN0b3JhZ2VfY2xhc3NfdHlwZTtcblxuICBpZiAodmFsdWUgPT09IHVuZGVmaW5lZCkge1xuICAgIHJldHVybiBcIlwiO1xuICB9XG5cbiAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJzdHJpbmdcIikge1xuICAgIC8vIHRoaXMgY291bGQgYmUgYW4gZXhwcmVzc2lvbiwgc28gd2UgZG9uJ3QgbmVlZCB0byBkbyBhbnl0aGluZyBoZXJlXG4gICAgcmV0dXJuIHJlbmRlclN0cmluZyh2YWx1ZSk7XG4gIH1cblxuICBpZiAobmFtZSkge1xuICAgIGlmIChibG9jaykge1xuICAgICAgaWYgKCFBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgICAgICByZXR1cm4gYCR7bmFtZX0ge2AgKyBgXFxuJHtyZW5kZXJBdHRyaWJ1dGVzKHZhbHVlKX1cXG59YDtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIHZhbHVlXG4gICAgICAgIC5tYXAoKHY6IGFueSkgPT4ge1xuICAgICAgICAgIHJldHVybiBgJHtuYW1lfSB7YCArIGBcXG4ke3JlbmRlckF0dHJpYnV0ZXModil9XFxufWA7XG4gICAgICAgIH0pXG4gICAgICAgIC5qb2luKFwiXFxuXCIpO1xuICAgIH1cblxuICAgIGlmICghQXJyYXkuaXNBcnJheSh2YWx1ZSkpIHtcbiAgICAgIHJldHVybiBgJHtuYW1lfSA9ICR7cmVuZGVyTGlzdFZhbHVlKHZhbHVlLCBjbGFzc1R5cGUpfWA7XG4gICAgfVxuXG4gICAgcmV0dXJuIGAke25hbWV9ID0gW1xuJHt2YWx1ZS5tYXAoKHY6IGFueSkgPT4gcmVuZGVyTGlzdFZhbHVlKHYsIGNsYXNzVHlwZSkpLmpvaW4oXCIsXFxuXCIpfVxuXWA7XG4gIH1cblxuICBpZiAoIUFycmF5LmlzQXJyYXkodmFsdWUpKSB7XG4gICAgLy8gQSBsaXN0IHR5cGUgb2Ygb25seSBhIHNpbmdsZSBpdGVtXG4gICAgcmV0dXJuIHJlbmRlckxpc3RWYWx1ZSh2YWx1ZSwgY2xhc3NUeXBlKTtcbiAgfVxuXG4gIGlmIChibG9jaykge1xuICAgIHJldHVybiB2YWx1ZS5tYXAoKHY6IGFueSkgPT4gcmVuZGVyTGlzdFZhbHVlKHYsIGNsYXNzVHlwZSkpLmpvaW4oXCJcXG5cIik7XG4gIH1cblxuICByZXR1cm4gYFtcbiR7dmFsdWUubWFwKCh2OiBhbnkpID0+IHJlbmRlckxpc3RWYWx1ZSh2LCBjbGFzc1R5cGUpKS5qb2luKFwiLFxcblwiKX1cbl1gO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJMaXN0VmFsdWUodmFsdWU6IGFueSwgc3RvcmFnZUNsYXNzVHlwZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgaWYgKHN0b3JhZ2VDbGFzc1R5cGUgPT09IFwic3RyaW5nTGlzdFwiKSB7XG4gICAgcmV0dXJuIHJlbmRlclN0cmluZyh2YWx1ZSk7XG4gIH1cbiAgaWYgKHN0b3JhZ2VDbGFzc1R5cGUgPT09IFwibnVtYmVyTGlzdFwiIHx8IHN0b3JhZ2VDbGFzc1R5cGUgPT09IFwiYm9vbGVhbkxpc3RcIikge1xuICAgIHJldHVybiBgJHt2YWx1ZX1gO1xuICB9XG5cbiAgcmV0dXJuIHJlbmRlckJsb2NrKHZhbHVlLCBzdG9yYWdlQ2xhc3NUeXBlKTtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyTWFwKG1hcDogYW55KTogc3RyaW5nIHtcbiAgaWYgKHR5cGVvZiBtYXAgPT09IFwic3RyaW5nXCIpIHtcbiAgICAvLyB0aGlzIGNvdWxkIGJlIGFuIGV4cHJlc3Npb24sIHNvIHdlIGRvbid0IG5lZWQgdG8gZG8gYW55dGhpbmcgaGVyZVxuICAgIHJldHVybiBgXCIke2VzY2FwZVF1b3RlcyhtYXApfVwiYDtcbiAgfVxuICByZXR1cm4gYHtcbiR7T2JqZWN0LmVudHJpZXMobWFwKVxuICAubWFwKChbaywgdl0pID0+IHtcbiAgICBjb25zdCB3cmFwcGVkS2V5ID0gd3JhcElkZW50aWZpZXJJblF1b3Rlc0lmTmVlZGVkKGspO1xuXG4gICAgcmV0dXJuIGAke3dyYXBwZWRLZXl9ID0gJHtyZW5kZXJNYXBWYWx1ZSh2KX1gO1xuICB9KVxuICAuam9pbihcIlxcblwiKX1cbn1gO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJNYXBWYWx1ZSh2YWx1ZTogYW55KTogc3RyaW5nIHtcbiAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJzdHJpbmdcIikge1xuICAgIHJldHVybiByZW5kZXJTdHJpbmcodmFsdWUpO1xuICB9XG4gIGlmICh0eXBlb2YgdmFsdWUgPT09IFwibnVtYmVyXCIpIHtcbiAgICByZXR1cm4gYCR7dmFsdWV9YDtcbiAgfVxuICBpZiAodHlwZW9mIHZhbHVlID09PSBcImJvb2xlYW5cIikge1xuICAgIHJldHVybiBgJHt2YWx1ZX1gO1xuICB9XG5cbiAgaWYgKHR5cGVvZiB2YWx1ZSA9PT0gXCJvYmplY3RcIikge1xuICAgIGlmIChBcnJheS5pc0FycmF5KHZhbHVlKSkge1xuICAgICAgcmV0dXJuIHJlbmRlckxpc3RWYWx1ZSh2YWx1ZSwgXCJ1bmtub3duU3RvcmFnZUNsYXNzXCIpO1xuICAgIH1cbiAgICBpZiAodmFsdWUgPT09IG51bGwpIHtcbiAgICAgIHJldHVybiBcIm51bGxcIjtcbiAgICB9XG4gICAgcmV0dXJuIHJlbmRlck1hcCh2YWx1ZSk7XG4gIH1cblxuICByZXR1cm4gYCR7dmFsdWV9YDtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyUmVzb3VyY2UocmVzb3VyY2U6IGFueSkge1xuICBjb25zdCByZXNvdXJjZVR5cGUgPSBPYmplY3Qua2V5cyhyZXNvdXJjZSlbMF07XG4gIGNvbnN0IHJlc291cmNlc1dpdGhUeXBlID0gcmVzb3VyY2VbcmVzb3VyY2VUeXBlXTtcbiAgY29uc3QgcmVzb3VyY2VOYW1lID0gT2JqZWN0LmtleXMocmVzb3VyY2VzV2l0aFR5cGUpWzBdO1xuICBjb25zdCByZXNvdXJjZUF0dHJpYnV0ZXMgPSByZXNvdXJjZXNXaXRoVHlwZVtyZXNvdXJjZU5hbWVdO1xuXG4gIGlmIChyZXNvdXJjZUF0dHJpYnV0ZXMuY29ubmVjdGlvbikge1xuICAgIHJlc291cmNlQXR0cmlidXRlcy5jb25uZWN0aW9uID0ge1xuICAgICAgdmFsdWU6IHJlc291cmNlQXR0cmlidXRlcy5jb25uZWN0aW9uLFxuICAgICAgaXNCbG9jazogdHJ1ZSxcbiAgICB9O1xuICB9XG5cbiAgY29uc3QgeyBwcm92aXNpb25lciwgZHluYW1pYywgLi4ub3RoZXJBdHRycyB9ID0gcmVzb3VyY2VBdHRyaWJ1dGVzO1xuXG4gIGNvbnN0IGhjbCA9IFtgcmVzb3VyY2UgXCIke3Jlc291cmNlVHlwZX1cIiBcIiR7cmVzb3VyY2VOYW1lfVwiIHtgXTtcblxuICBjb25zdCBhdHRycyA9IHJlbmRlckF0dHJpYnV0ZXMob3RoZXJBdHRycyk7XG5cbiAgaWYgKGF0dHJzKSBoY2wucHVzaChhdHRycyk7XG4gIGlmIChwcm92aXNpb25lcikgaGNsLnB1c2gocmVuZGVyUHJvdmlzaW9uZXJCbG9jayhwcm92aXNpb25lcikpO1xuICBpZiAoZHluYW1pYykgaGNsLnB1c2goLi4ucmVuZGVyRHluYW1pY0Jsb2NrcyhkeW5hbWljKSk7XG5cbiAgaGNsLnB1c2goXCJ9XCIpO1xuXG4gIHJldHVybiB7XG4gICAgaGNsOiBoY2wuam9pbihcIlxcblwiKSxcbiAgICBtZXRhZGF0YToge1xuICAgICAgcmVzb3VyY2U6IHtcbiAgICAgICAgW3Jlc291cmNlVHlwZV06IHtcbiAgICAgICAgICBbcmVzb3VyY2VOYW1lXToge1xuICAgICAgICAgICAgXCIvL1wiOiByZXNvdXJjZUF0dHJpYnV0ZXNbXCIvL1wiXSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9LFxuICB9O1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJEYXRhc291cmNlKGRhdGFTb3VyY2U6IGFueSkge1xuICBjb25zdCBkYXRhU291cmNlVHlwZSA9IE9iamVjdC5rZXlzKGRhdGFTb3VyY2UpWzBdO1xuICBjb25zdCBkYXRhU291cmNlc1dpdGhUeXBlID0gZGF0YVNvdXJjZVtkYXRhU291cmNlVHlwZV07XG4gIGNvbnN0IGRhdGFTb3VyY2VOYW1lID0gT2JqZWN0LmtleXMoZGF0YVNvdXJjZXNXaXRoVHlwZSlbMF07XG4gIGNvbnN0IGRhdGFTb3VyY2VBdHRyaWJ1dGVzID0gZGF0YVNvdXJjZXNXaXRoVHlwZVtkYXRhU291cmNlTmFtZV07XG5cbiAgY29uc3QgeyBkeW5hbWljLCAuLi5vdGhlckF0dHJzIH0gPSBkYXRhU291cmNlQXR0cmlidXRlcztcblxuICBjb25zdCBoY2wgPSBbYGRhdGEgXCIke2RhdGFTb3VyY2VUeXBlfVwiIFwiJHtkYXRhU291cmNlTmFtZX1cIiB7YF07XG5cbiAgY29uc3QgYXR0cnMgPSByZW5kZXJBdHRyaWJ1dGVzKG90aGVyQXR0cnMpO1xuICBpZiAoYXR0cnMpIGhjbC5wdXNoKGF0dHJzKTtcbiAgaWYgKGR5bmFtaWMpIGhjbC5wdXNoKC4uLnJlbmRlckR5bmFtaWNCbG9ja3MoZHluYW1pYykpO1xuICBoY2wucHVzaChcIn1cIik7XG5cbiAgcmV0dXJuIHtcbiAgICBoY2w6IGhjbC5qb2luKFwiXFxuXCIpLFxuICAgIG1ldGFkYXRhOiB7XG4gICAgICBkYXRhOiB7XG4gICAgICAgIFtkYXRhU291cmNlVHlwZV06IHtcbiAgICAgICAgICBbZGF0YVNvdXJjZU5hbWVdOiB7XG4gICAgICAgICAgICBcIi8vXCI6IGRhdGFTb3VyY2VBdHRyaWJ1dGVzW1wiLy9cIl0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgfTtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyUHJvdmlzaW9uZXJCbG9jayhwcm92aXNpb25lcnM6IGFueSkge1xuICByZXR1cm4gcHJvdmlzaW9uZXJzXG4gICAgLm1hcCgocHJvdmlzaW9uZXI6IGFueSkgPT4ge1xuICAgICAgY29uc3QgcHJvdmlzaW9uZXJUeXBlID0gT2JqZWN0LmtleXMocHJvdmlzaW9uZXIpWzBdO1xuICAgICAgY29uc3QgcHJvdmlzaW9uZXJBdHRycyA9IHByb3Zpc2lvbmVyW3Byb3Zpc2lvbmVyVHlwZV07XG5cbiAgICAgIGlmIChwcm92aXNpb25lckF0dHJzLmNvbm5lY3Rpb24pIHtcbiAgICAgICAgcHJvdmlzaW9uZXJBdHRycy5jb25uZWN0aW9uID0ge1xuICAgICAgICAgIHZhbHVlOiBwcm92aXNpb25lckF0dHJzLmNvbm5lY3Rpb24sXG4gICAgICAgICAgaXNCbG9jazogdHJ1ZSxcbiAgICAgICAgfTtcbiAgICAgIH1cblxuICAgICAgY29uc3QgeyBkeW5hbWljLCAuLi5vdGhlckF0dHJzIH0gPSBwcm92aXNpb25lckF0dHJzO1xuXG4gICAgICBjb25zdCBoY2wgPSBbYHByb3Zpc2lvbmVyIFwiJHtwcm92aXNpb25lclR5cGV9XCIge2BdO1xuXG4gICAgICBjb25zdCBhdHRycyA9IHJlbmRlckF0dHJpYnV0ZXMob3RoZXJBdHRycy52YWx1ZSB8fCBvdGhlckF0dHJzKTtcbiAgICAgIGlmIChhdHRycykgaGNsLnB1c2goYXR0cnMpO1xuICAgICAgaWYgKGR5bmFtaWMpIGhjbC5wdXNoKC4uLnJlbmRlckR5bmFtaWNCbG9ja3MoZHluYW1pYykpO1xuICAgICAgaGNsLnB1c2goXCJ9XCIpO1xuXG4gICAgICByZXR1cm4gaGNsLmpvaW4oXCJcXG5cIik7XG4gICAgfSlcbiAgICAuam9pbihcIlxcblxcblwiKTtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyRHluYW1pY0Jsb2NrcyhkeW5hbWljOiBhbnkpIHtcbiAgcmV0dXJuIE9iamVjdC5lbnRyaWVzKGR5bmFtaWMpLm1hcChcbiAgICAoW2R5bmFtaWNOYW1lLCBkeW5hbWljQXR0cnNdOiBbc3RyaW5nLCBhbnldKSA9PiB7XG4gICAgICBjb25zdCByZXMgPSBbYGR5bmFtaWMgXCIke2R5bmFtaWNOYW1lfVwiIHtgXTtcbiAgICAgIHJlcy5wdXNoKFxuICAgICAgICBgZm9yX2VhY2ggPSAke3JlbmRlckZ1enp5SnNvbkV4cHJlc3Npb24oZHluYW1pY0F0dHJzPy5mb3JfZWFjaCl9YCxcbiAgICAgICk7XG4gICAgICBpZiAoZHluYW1pY0F0dHJzPy5pdGVyYXRvcikge1xuICAgICAgICByZXMucHVzaChgaXRlcmF0b3IgPSAke2R5bmFtaWNBdHRycz8uaXRlcmF0b3J9YCk7XG4gICAgICB9XG4gICAgICBpZiAoZHluYW1pY0F0dHJzPy5sYWJlbHMpIHtcbiAgICAgICAgcmVzLnB1c2goYGxhYmVscyA9ICR7cmVuZGVyRnV6enlKc29uRXhwcmVzc2lvbihkeW5hbWljQXR0cnM/LmxhYmVscyl9YCk7XG4gICAgICB9XG4gICAgICByZXMucHVzaChgY29udGVudCB7XG4gICR7cmVuZGVyQXR0cmlidXRlcyhkeW5hbWljQXR0cnMuY29udGVudCl9XG4gIH1gKTtcbiAgICAgIHJlcy5wdXNoKGB9YCk7XG5cbiAgICAgIHJldHVybiByZXMuam9pbihcIlxcblwiKTtcbiAgICB9LFxuICApO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJQcm92aWRlcihwcm92aWRlcjogYW55KSB7XG4gIGNvbnN0IHByb3ZpZGVyTmFtZSA9IE9iamVjdC5rZXlzKHByb3ZpZGVyKVswXTtcbiAgY29uc3QgcHJvdmlkZXJBdHRyaWJ1dGVzID0gcHJvdmlkZXJbcHJvdmlkZXJOYW1lXTtcblxuICByZXR1cm4gcHJvdmlkZXJBdHRyaWJ1dGVzXG4gICAgLm1hcCgocHJvdmlkZXJJbnN0YW5jZTogYW55KSA9PiB7XG4gICAgICBjb25zdCB7IGR5bmFtaWMsIC4uLm90aGVyQXR0cnMgfSA9IHByb3ZpZGVySW5zdGFuY2U7XG5cbiAgICAgIGNvbnN0IGhjbCA9IFtgcHJvdmlkZXIgXCIke3Byb3ZpZGVyTmFtZX1cIiB7YF07XG5cbiAgICAgIGNvbnN0IGF0dHJzID0gcmVuZGVyQXR0cmlidXRlcyhvdGhlckF0dHJzKTtcbiAgICAgIGlmIChhdHRycykgaGNsLnB1c2goYXR0cnMpO1xuICAgICAgaWYgKGR5bmFtaWMpIGhjbC5wdXNoKC4uLnJlbmRlckR5bmFtaWNCbG9ja3MoZHluYW1pYykpO1xuXG4gICAgICBoY2wucHVzaChcIn1cIik7XG5cbiAgICAgIHJldHVybiBoY2wuam9pbihcIlxcblwiKTtcbiAgICB9KVxuICAgIC5qb2luKFwiXFxuXFxuXCIpO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJNb2R1bGUobW9kdWxlOiBhbnkpIHtcbiAgY29uc3QgbW9kdWxlTmFtZSA9IE9iamVjdC5rZXlzKG1vZHVsZSlbMF07XG4gIGNvbnN0IG1vZHVsZUF0dHJpYnV0ZXMgPSBtb2R1bGVbbW9kdWxlTmFtZV07XG5cbiAgcmV0dXJuIHtcbiAgICBoY2w6IGBtb2R1bGUgXCIke21vZHVsZU5hbWV9XCIge1xuJHtyZW5kZXJBdHRyaWJ1dGVzKG1vZHVsZUF0dHJpYnV0ZXMpfVxufWAsXG4gICAgbWV0YWRhdGE6IHtcbiAgICAgIG1vZHVsZToge1xuICAgICAgICBbbW9kdWxlTmFtZV06IHtcbiAgICAgICAgICBcIi8vXCI6IG1vZHVsZUF0dHJpYnV0ZXNbXCIvL1wiXSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgfTtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyT3V0cHV0KG91dHB1dDogYW55KSB7XG4gIGNvbnN0IG91dHB1dE5hbWUgPSBPYmplY3Qua2V5cyhvdXRwdXQpWzBdO1xuICBjb25zdCBvdXRwdXRBdHRyaWJ1dGVzID0gb3V0cHV0W291dHB1dE5hbWVdO1xuXG4gIHJldHVybiBgb3V0cHV0IFwiJHtvdXRwdXROYW1lfVwiIHtcbiR7cmVuZGVyQXR0cmlidXRlcyhvdXRwdXRBdHRyaWJ1dGVzKX1cbn1gO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJMb2NhbHMobG9jYWxzOiBhbnkpIHtcbiAgaWYgKCFsb2NhbHMpIHtcbiAgICByZXR1cm4gXCJcIjtcbiAgfVxuXG4gIGNvbnN0IGxvY2FsTmFtZXMgPSBPYmplY3Qua2V5cyhsb2NhbHMpO1xuXG4gIGlmIChsb2NhbE5hbWVzLmxlbmd0aCA9PT0gMCkge1xuICAgIHJldHVybiBcIlwiO1xuICB9XG5cbiAgcmV0dXJuIGBsb2NhbHMge1xuICAgICR7bG9jYWxOYW1lc1xuICAgICAgLmZpbHRlcigobmFtZSkgPT4gISFsb2NhbHNbbmFtZV0udmFsdWUpXG4gICAgICAubWFwKChuYW1lOiBzdHJpbmcpID0+IHtcbiAgICAgICAgY29uc3QgdmFsdWUgPSByZW5kZXJGdXp6eUpzb25FeHByZXNzaW9uKGxvY2Fsc1tuYW1lXS52YWx1ZSk7XG4gICAgICAgIHJldHVybiBgJHtuYW1lfSA9ICR7dmFsdWV9YDtcbiAgICAgIH0pXG4gICAgICAuam9pbihcIlxcblwiKX1cbn1gO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJWYXJpYWJsZSh2YXJpYWJsZTogYW55KSB7XG4gIGNvbnN0IHZhcmlhYmxlTmFtZSA9IE9iamVjdC5rZXlzKHZhcmlhYmxlKVswXTtcbiAgY29uc3QgdmFyaWFibGVBdHRyaWJ1dGVzID0gdmFyaWFibGVbdmFyaWFibGVOYW1lXTtcblxuICByZXR1cm4gYHZhcmlhYmxlIFwiJHt2YXJpYWJsZU5hbWV9XCIge1xuJHtyZW5kZXJBdHRyaWJ1dGVzKHZhcmlhYmxlQXR0cmlidXRlcyl9XG59YDtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyTW92ZWQobW92ZTogYW55KSB7XG4gIGNvbnN0IG1vdmVkQmxvY2tzID0gbW92ZS5tYXAoKG1vdmVCbG9jazogYW55KSA9PiB7XG4gICAgcmV0dXJuIGBtb3ZlZCB7XG4ke3JlbmRlckF0dHJpYnV0ZXMobW92ZUJsb2NrKX1cbn1gO1xuICB9KTtcblxuICByZXR1cm4gbW92ZWRCbG9ja3Muam9pbihcIlxcblwiKTtcbn1cblxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVySW1wb3J0KGltcG9ydHM6IGFueSkge1xuICBjb25zdCBpbXBvcnRCbG9ja3MgPSBpbXBvcnRzLm1hcCgoaW1wb3J0QmxvY2s6IGFueSkgPT4ge1xuICAgIHJldHVybiBgaW1wb3J0IHtcbiR7cmVuZGVyQXR0cmlidXRlcyhpbXBvcnRCbG9jayl9XG59YDtcbiAgfSk7XG5cbiAgcmV0dXJuIGltcG9ydEJsb2Nrcy5qb2luKFwiXFxuXCIpO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJUZXJyYWZvcm0odGVycmFmb3JtOiBhbnkpIHtcbiAgY29uc3QgYmxvY2tBdHRyaWJ1dGVzID0gW1wicmVxdWlyZWRfcHJvdmlkZXJzXCIsIFwiYmFja2VuZFwiLCBcImNsb3VkXCJdO1xuICBjb25zdCByZXF1aXJlZFByb3ZpZGVycyA9IGByZXF1aXJlZF9wcm92aWRlcnMge1xuJHtyZW5kZXJTaW1wbGVBdHRyaWJ1dGVzKHRlcnJhZm9ybS5yZXF1aXJlZF9wcm92aWRlcnMpfVxufWA7XG5cbiAgY29uc3QgYmFja2VuZHMgPSBPYmplY3Qua2V5cyh0ZXJyYWZvcm0uYmFja2VuZCB8fCB7fSkubWFwKChiYWNrZW5kTmFtZSkgPT4ge1xuICAgIHJldHVybiBgYmFja2VuZCBcIiR7YmFja2VuZE5hbWV9XCIge1xuJHtyZW5kZXJBdHRyaWJ1dGVzKHRlcnJhZm9ybS5iYWNrZW5kW2JhY2tlbmROYW1lXSl9XG59YDtcbiAgfSk7XG5cbiAgY29uc3QgY2xvdWRCYWNrZW5kID0gdGVycmFmb3JtLmNsb3VkXG4gICAgPyBgY2xvdWQge1xuJHtyZW5kZXJBdHRyaWJ1dGVzKHRlcnJhZm9ybS5jbG91ZCl9XG59YFxuICAgIDogdW5kZWZpbmVkO1xuXG4gIGNvbnN0IG90aGVyQXR0cmlidXRlcyA9IE9iamVjdC5rZXlzKHRlcnJhZm9ybSkuZmlsdGVyKFxuICAgIChrZXkpID0+ICFibG9ja0F0dHJpYnV0ZXMuaW5jbHVkZXMoa2V5KSxcbiAgKTtcbiAgcmV0dXJuIGB0ZXJyYWZvcm0ge1xuJHtyZXF1aXJlZFByb3ZpZGVyc31cbiR7Wy4uLmJhY2tlbmRzLCBjbG91ZEJhY2tlbmRdLmpvaW4oXCJcXG5cIil9XG4ke3JlbmRlclNpbXBsZUF0dHJpYnV0ZXMob3RoZXJBdHRyaWJ1dGVzKX1cbn1gO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJCbG9jayhibG9jazogYW55LCBfc3RvcmFnZUNsYXNzVHlwZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgcmV0dXJuIGB7XG4ke3JlbmRlckF0dHJpYnV0ZXMoYmxvY2spfVxufWA7XG59XG5cbi8qKlxuICpcbiAqL1xuZnVuY3Rpb24gcmVuZGVyRnV6enlKc29uT2JqZWN0KGpzb25PYmplY3Q6IGFueSk6IHN0cmluZyB7XG4gIHJldHVybiBbXG4gICAgXCJ7XCIsXG4gICAgLi4uT2JqZWN0LmVudHJpZXMoanNvbk9iamVjdCkubWFwKChbbmFtZSwgdmFsdWVdKSA9PiB7XG4gICAgICBjb25zdCB3cmFwcGVkS2V5ID0gd3JhcElkZW50aWZpZXJJblF1b3Rlc0lmTmVlZGVkKG5hbWUpO1xuICAgICAgcmV0dXJuIGAke3dyYXBwZWRLZXl9ID0gJHtyZW5kZXJGdXp6eUpzb25FeHByZXNzaW9uKHZhbHVlKX1gO1xuICAgIH0pLFxuICAgIFwifVwiLFxuICBdLmpvaW4oXCJcXG5cIik7XG59XG4vKipcbiAqXG4gKi9cbmZ1bmN0aW9uIHJlbmRlckZ1enp5SnNvbkV4cHJlc3Npb24oanNvbkV4cHJlc3Npb246IGFueSk6IHN0cmluZyB7XG4gIGlmIChBcnJheS5pc0FycmF5KGpzb25FeHByZXNzaW9uKSkge1xuICAgIHJldHVybiBbXG4gICAgICBcIltcIixcbiAgICAgIC4uLmpzb25FeHByZXNzaW9uLm1hcCgodmFsdWUpID0+IGAke3JlbmRlckZ1enp5SnNvbkV4cHJlc3Npb24odmFsdWUpfSxgKSxcbiAgICAgIFwiXVwiLFxuICAgIF0uam9pbihcIlxcblwiKTtcbiAgfVxuXG4gIGlmICh0eXBlb2YganNvbkV4cHJlc3Npb24gPT09IFwib2JqZWN0XCIpIHtcbiAgICByZXR1cm4gcmVuZGVyRnV6enlKc29uT2JqZWN0KGpzb25FeHByZXNzaW9uKTtcbiAgfVxuXG4gIGlmICh0eXBlb2YganNvbkV4cHJlc3Npb24gPT09IFwic3RyaW5nXCIpIHtcbiAgICBpZiAoanNvbkV4cHJlc3Npb24gPT09IFwiXCIpIHtcbiAgICAgIHJldHVybiBgXCJcImA7XG4gICAgfVxuXG4gICAgaWYgKGpzb25FeHByZXNzaW9uLmluY2x1ZGVzKFwiJHtcIikpIHtcbiAgICAgIHJldHVybiBgXCIke2pzb25FeHByZXNzaW9ufVwiYDtcbiAgICB9XG5cbiAgICBpZiAocGFyc2VJbnQoanNvbkV4cHJlc3Npb24sIDEwKS50b1N0cmluZygpID09PSBqc29uRXhwcmVzc2lvbikge1xuICAgICAgcmV0dXJuIGpzb25FeHByZXNzaW9uO1xuICAgIH1cblxuICAgIGlmIChqc29uRXhwcmVzc2lvbi5zdGFydHNXaXRoKFwiW1wiKSkge1xuICAgICAgcmV0dXJuIGpzb25FeHByZXNzaW9uO1xuICAgIH1cblxuICAgIGlmIChqc29uRXhwcmVzc2lvbi5zdGFydHNXaXRoKFwie1wiKSkge1xuICAgICAgcmV0dXJuIGpzb25FeHByZXNzaW9uO1xuICAgIH1cblxuICAgIGlmIChqc29uRXhwcmVzc2lvbi5zdGFydHNXaXRoKCdcIicpKSB7XG4gICAgICByZXR1cm4ganNvbkV4cHJlc3Npb247XG4gICAgfVxuXG4gICAgcmV0dXJuIGBcIiR7ZXNjYXBlUXVvdGVzKGpzb25FeHByZXNzaW9uKX1cImA7XG4gIH1cblxuICBpZiAoanNvbkV4cHJlc3Npb24gPT09IFwidHJ1ZVwiIHx8IGpzb25FeHByZXNzaW9uID09PSBcImZhbHNlXCIpIHtcbiAgICByZXR1cm4ganNvbkV4cHJlc3Npb24gYXMgc3RyaW5nO1xuICB9XG5cbiAgaWYgKGpzb25FeHByZXNzaW9uID09PSBcIm51bGxcIikge1xuICAgIHJldHVybiBqc29uRXhwcmVzc2lvbjtcbiAgfVxuXG4gIHJldHVybiBgJHtqc29uRXhwcmVzc2lvbn1gO1xufVxuXG4vKipcbiAqXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiByZW5kZXJTaW1wbGVBdHRyaWJ1dGVzKGF0dHJpYnV0ZXM6IGFueSk6IHN0cmluZyB7XG4gIGlmIChhdHRyaWJ1dGVzID09PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4gXCJcIjtcbiAgfVxuXG4gIHJldHVybiBPYmplY3QuZW50cmllcyhhdHRyaWJ1dGVzKVxuICAgIC5tYXAoXG4gICAgICAoW25hbWUsIHZhbHVlXSkgPT5cbiAgICAgICAgYCAgJHtuYW1lfSA9ICR7cmVuZGVyRnV6enlKc29uRXhwcmVzc2lvbih2YWx1ZSBhcyBhbnkpfWAsXG4gICAgKVxuICAgIC5qb2luKFwiXFxuXCIpO1xufVxuLyoqXG4gKlxuICovXG5leHBvcnQgZnVuY3Rpb24gcmVuZGVyQXR0cmlidXRlcyhhdHRyaWJ1dGVzOiBhbnkpOiBzdHJpbmcge1xuICBpZiAoYXR0cmlidXRlcyA9PT0gdW5kZWZpbmVkIHx8IGF0dHJpYnV0ZXMgPT09IG51bGwpIHtcbiAgICByZXR1cm4gXCJcIjtcbiAgfVxuICByZXR1cm4gT2JqZWN0LmVudHJpZXMoYXR0cmlidXRlcylcbiAgICAubWFwKChbbmFtZSwgdl0pID0+IHtcbiAgICAgIGlmIChuYW1lID09PSBcIi8vXCIpIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgIH1cbiAgICAgIGNvbnN0IG1ldGFCbG9ja3MgPSBbXCJsaWZlY3ljbGVcIiwgXCJwcm92aXNpb25lclwiXTtcblxuICAgICAgLy9cbiAgICAgIC8vIFdlIG1pZ2h0IGhhdmUgc29tZSBhdHRyaWJ1dGVzIHRoYXQgZG9uJ3QgaGF2ZSB0eXBlIGluZm9ybWF0aW9uXG4gICAgICAvLyBqdXN0IHRyeSB0byBndWVzcyB0aGVtXG4gICAgICBpZiAodHlwZW9mIHYgPT09IFwic3RyaW5nXCIgfHwgdHlwZW9mIHYgPT09IFwibnVtYmVyXCIpIHtcbiAgICAgICAgcmV0dXJuIGAke25hbWV9ID0gJHtyZW5kZXJGdXp6eUpzb25FeHByZXNzaW9uKHYpfWA7XG4gICAgICB9IGVsc2UgaWYgKHR5cGVvZiB2ID09PSBcImJvb2xlYW5cIikge1xuICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSAke3Z9YDtcbiAgICAgIH0gZWxzZSBpZiAoQXJyYXkuaXNBcnJheSh2KSkge1xuICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSAke3JlbmRlckZ1enp5SnNvbkV4cHJlc3Npb24odil9YDtcbiAgICAgIH0gZWxzZSBpZiAodiA9PT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSBudWxsYDtcbiAgICAgIH0gZWxzZSBpZiAoXG4gICAgICAgIHR5cGVvZiB2ID09PSBcIm9iamVjdFwiICYmXG4gICAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBuby1wcm90b3R5cGUtYnVpbHRpbnNcbiAgICAgICAgIXYuaGFzT3duUHJvcGVydHkoXCJ2YWx1ZVwiKSAmJlxuICAgICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbm8tcHJvdG90eXBlLWJ1aWx0aW5zXG4gICAgICAgICF2Lmhhc093blByb3BlcnR5KFwiZHluYW1pY1wiKVxuICAgICAgKSB7XG4gICAgICAgIGlmIChtZXRhQmxvY2tzLmluY2x1ZGVzKG5hbWUpKSB7XG4gICAgICAgICAgcmV0dXJuIGAke25hbWV9IHtcbiR7cmVuZGVyU2ltcGxlQXR0cmlidXRlcyh2KX1cbn1gO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBgJHtuYW1lfSA9ICR7cmVuZGVyRnV6enlKc29uRXhwcmVzc2lvbih2KX1gO1xuICAgICAgfSBlbHNlIGlmICh2ID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgIH1cblxuICAgICAgLy8gUmVmZXJlbmNpbmcgYm90aCBpc0Jsb2NrIGFuZCBpc19ibG9jaywgYmVjYXVzZSBzb21ldGltZXMgd2UgcGFzcyB0aHJvdWdoIGEgc25ha2UgY2FzZSBmaWx0ZXJcbiAgICAgIC8vIHdpdGhpbiBhdHRyaWJ1dGVzLlxuICAgICAgY29uc3Qge1xuICAgICAgICBkeW5hbWljLFxuICAgICAgICB2YWx1ZSxcbiAgICAgICAgdHlwZSxcbiAgICAgICAgaXNCbG9jayxcbiAgICAgICAgc3RvcmFnZUNsYXNzVHlwZSxcbiAgICAgICAgaXNfYmxvY2ssXG4gICAgICAgIHN0b3JhZ2VfY2xhc3NfdHlwZSxcbiAgICAgIH0gPSB2IGFzIGFueTtcbiAgICAgIGNvbnN0IGJsb2NrID0gaXNCbG9jayB8fCBpc19ibG9jayB8fCBtZXRhQmxvY2tzLmluY2x1ZGVzKG5hbWUpO1xuICAgICAgY29uc3QgY2xhc3NUeXBlID0gc3RvcmFnZUNsYXNzVHlwZSB8fCBzdG9yYWdlX2NsYXNzX3R5cGU7XG5cbiAgICAgIGlmIChkeW5hbWljKSB7XG4gICAgICAgIHJldHVybiByZW5kZXJEeW5hbWljQmxvY2tzKHtcbiAgICAgICAgICBbbmFtZV06IGR5bmFtaWMudmFsdWUsXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgICAgLy8gU2hvcnQgY2lyY3VpdCB0eXBlIGNoZWNraW5nIGlmIHZhbHVlIGlzIGFuIGV4cHJlc3Npb25cbiAgICAgIGlmICh0eXBlb2YgdmFsdWUgPT09IFwic3RyaW5nXCIgJiYgdmFsdWUuaW5jbHVkZXMoXCIke1wiKSkge1xuICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSAke3JlbmRlclN0cmluZyh2YWx1ZSl9YDtcbiAgICAgIH1cblxuICAgICAgaWYgKGJsb2NrICYmIHR5cGUgIT09IFwibGlzdFwiICYmIHR5cGUgIT09IFwic2V0XCIpIHtcbiAgICAgICAgcmV0dXJuIGAke25hbWV9IHtcbiR7cmVuZGVyQXR0cmlidXRlcyh2YWx1ZSl9XG59YDtcbiAgICAgIH1cbiAgICAgIGlmICh0eXBlID09PSBcImxpc3RcIiB8fCB0eXBlID09PSBcInNldFwiKSB7XG4gICAgICAgIGlmIChibG9jaykge1xuICAgICAgICAgIHJldHVybiByZW5kZXJMaXN0KHYsIG5hbWUpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBgJHtuYW1lfSA9ICR7cmVuZGVyTGlzdCh2KX1gO1xuICAgICAgfVxuICAgICAgaWYgKHR5cGUgPT09IFwibWFwXCIpIHtcbiAgICAgICAgcmV0dXJuIGAke25hbWV9ID0gJHtyZW5kZXJNYXAodmFsdWUpfWA7XG4gICAgICB9XG5cbiAgICAgIGlmICh0eXBlID09PSBcInNpbXBsZVwiKSB7XG4gICAgICAgIGlmIChjbGFzc1R5cGUgPT09IFwic3RyaW5nXCIpIHtcbiAgICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSAke3JlbmRlclN0cmluZyh2YWx1ZSl9YDtcbiAgICAgICAgfVxuICAgICAgICBpZiAoY2xhc3NUeXBlID09PSBcIm51bWJlclwiIHx8IGNsYXNzVHlwZSA9PT0gXCJib29sZWFuXCIpIHtcbiAgICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSAke3ZhbHVlfWA7XG4gICAgICAgIH1cblxuICAgICAgICAvLyBOb3Qgc3VyZSB3aHkgd2UncmUgaGVyZSwgYnV0IHRoZXJlJ3MgZWl0aGVyIGEgYnVnIGluIHRoZSBwcm92aWRlclxuICAgICAgICAvLyBvciB3ZSBoYXZlIHNraXBwZWQgdGhlIGF0dHJpYnV0ZSB0byByZWR1Y2UgdGhlIHNpemUgb2Ygb3VyIHByb3ZpZGVyXG4gICAgICAgIC8vIEluIGVpdGhlciBjYXNlLCB3ZSBzaG91bGQgdHJ5IHRvIG5vdCBvdXRwdXQgW29iamVjdCBPYmplY3RdIGhlcmVcbiAgICAgICAgLy8gYW5kIHRyeSBhIGJlc3QgYXBwcm94aW1hdGlvbi4gVGhvdWdoLCBpdCB3b3VsZCBub3Qgd29yayBmb3JcbiAgICAgICAgLy8gYmxvY2tzXG4gICAgICAgIGlmICh0eXBlb2YgdmFsdWUgPT09IFwib2JqZWN0XCIpIHtcbiAgICAgICAgICByZXR1cm4gYFxuIyBXYXJuaW5nOiBUaGUgZm9sbG93aW5nIGF0dHJpYnV0ZSBpcyBvZiBhbiB1bmV4cGVjdGVkIHR5cGUuIEVpdGhlciB0aGVyZSdzIGEgcHJvYmxlbSB3aXRoIHRoZSBwcm92aWRlclxuIyBvciBDREtURiBoYXMgY2hvc2VuIHRvIHNraXAgZ2VuZXJhdGluZyBhIGRldGFpbGVkIHR5cGUgZm9yIHRoaXMgYmVjYXVzZSBpdCBpbmNyZWFzZXMgdGhlIHNpemUgb2YgdGhlIHByb3ZpZGVyIGxpYnJhcnkgc2lnbmlmaWNhbnRseS4gXG4jIFBsZWFzZSBjaGVjayBpZiB0aGlzIHJlc291cmNlIGlzIGluY2x1ZGVkIGluIG91ciBza2lwIGxpc3Q6IFxuIyBodHRwczovL2dpdGh1Yi5jb20vaGFzaGljb3JwL3RlcnJhZm9ybS1jZGsvYmxvYi8xZmI4NTg4MDk1YTU1ZDQ3NWRjNGJlMjg4ODJlYzRmNDIwMDZlYzhkL3BhY2thZ2VzLyU0MGNka3RmL3Byb3ZpZGVyLWdlbmVyYXRvci9saWIvZ2V0L2dlbmVyYXRvci9za2lwcGVkLWF0dHJpYnV0ZXMudHNcbiMgYW5kIGlmIG5vdCwgcGxlYXNlIGNoZWNrIHdpdGggdGhlIHByb3ZpZGVyIGF1dGhvcnMgdG8gc2VlIGlmIHRoZSBwcm92aWRlciBzY2hlbWEgaXMgYWNjdXJhdGUuXG4jIFxuIyBXZSB1bmRlcnN0YW5kIHRoaXMgaXMgbm90IGlkZWFsLCBzbyB3ZSBzdWdnZXN0IHVzaW5nIEpTT04gc3ludGhlc2lzIGluc3RlYWQgb2YgSENMIHN5bnRoZXNpcyBmb3IgdGhpcyBwYXJ0aWN1bGFyIGNhc2UuXG5cbiR7bmFtZX0gPSAke3JlbmRlckZ1enp5SnNvbkV4cHJlc3Npb24odmFsdWUpfVxuYDtcbiAgICAgICAgfVxuICAgICAgfVxuXG4gICAgICBpZiAodHlwZSA9PT0gXCJhbnlcIikge1xuICAgICAgICByZXR1cm4gYCR7bmFtZX0gPSAke3JlbmRlckZ1enp5SnNvbkV4cHJlc3Npb24odmFsdWUpfWA7XG4gICAgICB9XG5cbiAgICAgIGlmICh0eXBlID09PSBcInJlZmVyZW5jZVwiKSB7XG4gICAgICAgIHJldHVybiBgJHtuYW1lfSA9ICR7dmFsdWV9YDtcbiAgICAgIH1cblxuICAgICAgcmV0dXJuIGAke25hbWV9ID0gJHt2YWx1ZX1gO1xuICAgIH0pXG4gICAgLmZpbHRlcigodikgPT4gdiAhPT0gdW5kZWZpbmVkKVxuICAgIC5qb2luKFwiXFxuXCIpO1xufVxuIl19