"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.anyValue = anyValue;
exports.mapValue = mapValue;
exports.stringValue = stringValue;
exports.numericValue = numericValue;
exports.listOf = listOf;
exports.variadic = variadic;
exports.asString = asString;
exports.asNumber = asNumber;
exports.asList = asList;
exports.asStringMap = asStringMap;
exports.asBoolean = asBoolean;
exports.asAny = asAny;
exports.terraformFunction = terraformFunction;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const token_1 = require("../tokens/token");
const tfExpression_1 = require("../tfExpression");
const encoding_1 = require("../tokens/private/encoding");
const errors_1 = require("../errors");
/**
 * Determines if given str has unescaped double quotes
 * @param str String to test
 * @returns Boolean
 */
function hasUnescapedDoubleQuotes(str) {
    return /(^|[^\\])([\\]{2})*"/.test(str);
}
// Validators
// eslint-disable-next-line jsdoc/require-jsdoc
function anyValue(value) {
    return { value };
}
// eslint-disable-next-line jsdoc/require-jsdoc
function mapValue(value) {
    return { value };
}
// eslint-disable-next-line jsdoc/require-jsdoc
function stringValue(value) {
    if (typeof value !== "string" && !token_1.Tokenization.isResolvable(value)) {
        throw (0, errors_1.valueIsInvalidStringOrToken)(value);
    }
    if (typeof value === "string" && hasUnescapedDoubleQuotes(value)) {
        throw (0, errors_1.valueContainsUnescapedQuotes)(value);
    }
    return { value };
}
// eslint-disable-next-line jsdoc/require-jsdoc
function numericValue(value) {
    if (typeof value !== "number" && !token_1.Tokenization.isResolvable(value)) {
        throw (0, errors_1.valueIsInvalidNumberOrToken)(value);
    }
    return { value };
}
// eslint-disable-next-line jsdoc/require-jsdoc
function listOf(type) {
    return (value) => {
        if (token_1.Tokenization.isResolvable(value)) {
            return { value };
        }
        if (!Array.isArray(value)) {
            //   throw new Error(`${value} is not a valid list`);
            return { value };
        }
        return {
            value: value
                .filter((item) => item !== undefined && item !== null)
                .map((item, i) => {
                if (token_1.Tokenization.isResolvable(item)) {
                    return item;
                }
                if (encoding_1.TokenString.forListToken(item).test()) {
                    return item;
                }
                if ((0, encoding_1.extractTokenDouble)(item, true) !== undefined) {
                    return item;
                }
                if (encoding_1.TokenString.forMapToken(item).test()) {
                    return item;
                }
                if (typeof item === "string") {
                    const tokenList = token_1.Tokenization.reverseString(item);
                    const numberOfTokens = tokenList.tokens.length + tokenList.intrinsic.length;
                    if (numberOfTokens === 1 && tokenList.literals.length === 0) {
                        return item;
                    }
                }
                try {
                    type(item);
                    return typeof item === "string" ? `"${item}"` : item;
                }
                catch (error) {
                    throw (0, errors_1.listElementIsOfWrongType)(value, i, error);
                }
            }),
        };
    };
}
// eslint-disable-next-line jsdoc/require-jsdoc
function variadic(type) {
    return (value) => {
        // we use the list validator and set variadic to true in order to have it expanded in the args passed to the TF function
        return { value: listOf(type)(value).value, variadic: true };
    };
}
// Tokenization
// eslint-disable-next-line jsdoc/require-jsdoc
function asString(value) {
    return token_1.Token.asString(value);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function asNumber(value) {
    return token_1.Token.asNumber(value);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function asList(value) {
    return token_1.Token.asList(value);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function asStringMap(value) {
    return token_1.Token.asStringMap(value);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function asBoolean(value) {
    return value; // Booleans can not be represented as a token
}
// eslint-disable-next-line jsdoc/require-jsdoc
function asAny(value) {
    // Ordinarily casting to any can cause issues, but
    // in this case it makes using functions a bit easier in TS
    // and doesn't really harm other languages.
    // Jsii has issues when returning the value directly,
    // so wrap as a string.
    return asString(value);
}
// eslint-disable-next-line jsdoc/require-jsdoc
function terraformFunction(name, argValidators) {
    return function (...args) {
        if (args.length !== argValidators.length) {
            throw (0, errors_1.functionReceivedWrongNumberOfArgs)(name, argValidators.length, args.length);
        }
        return (0, tfExpression_1.call)(name, 
        // We use flatMap now, since listOf() wraps everything in an extra array but variadic() does not
        // This enables us to have variadic pass multiple args into the call() from a single array
        args.reduce((carry, arg, i) => {
            try {
                const val = argValidators[i](arg);
                if (val.variadic)
                    return [...carry, ...val.value];
                else
                    return [...carry, val.value];
            }
            catch (error) {
                throw (0, errors_1.functionArgumentValidationFailure)(i, name, error);
            }
        }, []));
    };
}
//# sourceMappingURL=data:application/json;base64,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