"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TaggedCloudWorkspaces = exports.NamedCloudWorkspace = exports.CloudWorkspace = exports.CloudBackend = void 0;
exports.getHostNameType = getHostNameType;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const util_1 = require("../util");
const remote_backend_1 = require("./remote-backend");
const terraform_backend_1 = require("../terraform-backend");
const validate_terraform_version_1 = require("../validations/validate-terraform-version");
const errors_1 = require("../errors");
/**
 * checks whether the given hostname belongs to tfc or (else) to tfe
 * If no hostname is given, it will return tfc, as that's the default in backends.
 * @param hostname e.g. app.terraform.io, app.terraform.io:80, tfe.myorg.org
 * @returns "tfc" or "tfe"
 */
function getHostNameType(hostname) {
    if (!hostname)
        return "tfc"; // default is tfc when not passing a hostname to backends
    return hostname.startsWith("app.terraform.io") ? "tfc" : "tfe";
}
/**
 * The Cloud Backend synthesizes a {@link https://developer.hashicorp.com/terraform/cli/cloud/settings#the-cloud-block cloud block}.
 * The cloud block is a nested block within the top-level terraform settings block.
 * It specifies which Terraform Cloud workspaces to use for the current working directory.
 * The cloud block only affects Terraform CLI's behavior.
 * When Terraform Cloud uses a configuration that contains a cloud block - for example, when a workspace is configured to use a VCS provider directly - it ignores the block and behaves according to its own workspace settings.
 */
class CloudBackend extends terraform_backend_1.TerraformBackend {
    constructor(scope, props) {
        super(scope, "backend", "cloud");
        this.props = props;
        this.node.addValidation(new validate_terraform_version_1.ValidateTerraformVersion(">=1.1", `The cloud block is only supported for Terraform >=1.1. Please upgrade your Terraform version.`));
    }
    toHclTerraform() {
        return {
            terraform: {
                cloud: (0, util_1.deepMerge)(this.synthesizeHclAttributes(), this.rawOverrides),
            },
        };
    }
    /**
     * Adds this resource to the terraform JSON output.
     */
    toTerraform() {
        return {
            terraform: {
                cloud: (0, util_1.deepMerge)(this.synthesizeAttributes(), this.rawOverrides),
            },
        };
    }
    toMetadata() {
        const cloud = getHostNameType(this.props.hostname);
        return { ...super.toMetadata(), cloud };
    }
    synthesizeHclAttributes() {
        return {
            ...(0, util_1.keysToSnakeCase)(this.props),
            workspaces: {
                value: this.props.workspaces.toHclTerraform(),
                isBlock: true,
                type: "map",
                storageClassType: "stringmap",
            },
        };
    }
    synthesizeAttributes() {
        return (0, util_1.keysToSnakeCase)({
            ...this.props,
            workspaces: this.props.workspaces.toTerraform(),
        });
    }
    getRemoteStateDataSource(scope, name, _fromStack) {
        // The cloud "backend" does not have a coresponding remote state data source since it's meant to also
        // work with multiple workspaces through tags.
        if ("name" in this.props.workspaces) {
            return new remote_backend_1.DataTerraformRemoteState(scope, name, {
                organization: this.props.organization,
                workspaces: { name: this.props.workspaces.name },
                token: this.props.token,
                hostname: this.props.hostname,
            });
        }
        throw (0, errors_1.cloudBackendWorkspaceIsNotDefinedByName)();
    }
}
exports.CloudBackend = CloudBackend;
_a = JSII_RTTI_SYMBOL_1;
CloudBackend[_a] = { fqn: "cdktf.CloudBackend", version: "0.21.0" };
/**
 * A cloud workspace can either be a single named workspace, or a list of tagged workspaces.
 */
class CloudWorkspace {
}
exports.CloudWorkspace = CloudWorkspace;
_b = JSII_RTTI_SYMBOL_1;
CloudWorkspace[_b] = { fqn: "cdktf.CloudWorkspace", version: "0.21.0" };
/**
 * The name of a single Terraform Cloud workspace.
 * You will only be able to use the workspace specified in the configuration with this working directory, and cannot manage workspaces from the CLI (e.g. terraform workspace select or terraform workspace new).
 */
class NamedCloudWorkspace extends CloudWorkspace {
    constructor(name, project) {
        super();
        this.name = name;
        this.project = project;
    }
    toHclTerraform() {
        return {
            name: {
                value: this.name,
                type: "simple",
                storageClassType: "string",
            },
        };
    }
    toTerraform() {
        return {
            name: this.name,
            project: this.project,
        };
    }
}
exports.NamedCloudWorkspace = NamedCloudWorkspace;
_c = JSII_RTTI_SYMBOL_1;
NamedCloudWorkspace[_c] = { fqn: "cdktf.NamedCloudWorkspace", version: "0.21.0" };
/**
 *  A set of Terraform Cloud workspace tags. You will be able to use this working directory with any workspaces that have all of the specified tags, and can use the terraform workspace commands to switch between them or create new workspaces. New workspaces will automatically have the specified tags. This option conflicts with name.
 */
class TaggedCloudWorkspaces extends CloudWorkspace {
    constructor(tags, project) {
        super();
        this.tags = tags;
        this.project = project;
    }
    toHclTerraform() {
        return {
            tags: {
                value: this.tags,
                type: "list",
                storageClassType: "stringList",
            },
        };
    }
    toTerraform() {
        return {
            tags: this.tags,
            project: this.project,
        };
    }
}
exports.TaggedCloudWorkspaces = TaggedCloudWorkspaces;
_d = JSII_RTTI_SYMBOL_1;
TaggedCloudWorkspaces[_d] = { fqn: "cdktf.TaggedCloudWorkspaces", version: "0.21.0" };
//# sourceMappingURL=data:application/json;base64,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