"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadAssemblyFromFile = exports.loadAssemblyFromPath = exports.loadAssemblyFromBuffer = exports.writeAssembly = exports.replaceAssembly = exports.findAssemblyFile = exports.compressedAssemblyExists = void 0;
const fs = require("fs");
const path = require("path");
const zlib = require("zlib");
const assembly_1 = require("./assembly");
const redirect_1 = require("./redirect");
const validate_assembly_1 = require("./validate-assembly");
/**
 * Returns true if the SPEC_FILE_NAME_COMPRESSED file exists in the directory.
 */
function compressedAssemblyExists(directory) {
    return fs.existsSync(path.join(directory, assembly_1.SPEC_FILE_NAME_COMPRESSED));
}
exports.compressedAssemblyExists = compressedAssemblyExists;
/**
 * Finds the path to the SPEC_FILE_NAME file, which will either
 * be the assembly or hold instructions to find the assembly.
 *
 * @param directory path to a directory with an assembly file
 * @returns path to the SPEC_FILE_NAME file
 */
function findAssemblyFile(directory) {
    const dotJsiiFile = path.join(directory, assembly_1.SPEC_FILE_NAME);
    if (!fs.existsSync(dotJsiiFile)) {
        throw new Error(`Expected to find ${assembly_1.SPEC_FILE_NAME} file in ${directory}, but no such file found`);
    }
    return dotJsiiFile;
}
exports.findAssemblyFile = findAssemblyFile;
/**
 * Replaces the file where the original assembly file *should* be found with a new assembly file.
 * Detects whether or not there is a compressed assembly, and if there is, compresses the new assembly also.
 * Replaces the fingerprint with '**********' rather than recalculating it, since we have modified the assembly.
 */
function replaceAssembly(assembly, directory) {
    writeAssembly(directory, _fingerprint(assembly), {
        compress: compressedAssemblyExists(directory),
    });
}
exports.replaceAssembly = replaceAssembly;
/**
 * Replaces the old fingerprint with '***********'.
 *
 * We could recalculate the fingerprint here so that it looks like the assembly was not modified. However,
 * 1) we are not actually validating the fingerprint in any way, and
 * 2) it feels disingenuous to have a mechanism against tampering and then tamper with it.
 *
 * So, instead of keeping the old (wrong) fingerprint or spending extra time calculating a new fingerprint,
 * we replace with '**********' that demonstrates the fingerprint has changed.
 */
function _fingerprint(assembly) {
    assembly.fingerprint = '*'.repeat(10);
    return assembly;
}
/**
 * Writes the assembly file either as .jsii or .jsii.gz if zipped
 *
 * @param directory the directory path to place the assembly file
 * @param assembly the contents of the assembly
 * @param compress whether or not to zip the assembly (.jsii.gz)
 * @returns whether or not the assembly was zipped
 */
function writeAssembly(directory, assembly, { compress = false } = {}) {
    if (compress) {
        // write .jsii file with instructions on opening the compressed file
        fs.writeFileSync(path.join(directory, assembly_1.SPEC_FILE_NAME), JSON.stringify({
            schema: 'jsii/file-redirect',
            compression: 'gzip',
            filename: assembly_1.SPEC_FILE_NAME_COMPRESSED,
        }), 'utf-8');
        // write actual assembly contents in .jsii.gz
        fs.writeFileSync(path.join(directory, assembly_1.SPEC_FILE_NAME_COMPRESSED), zlib.gzipSync(JSON.stringify(assembly)));
    }
    else {
        fs.writeFileSync(path.join(directory, assembly_1.SPEC_FILE_NAME), JSON.stringify(assembly, null, 2), 'utf-8');
    }
    return compress;
}
exports.writeAssembly = writeAssembly;
const failNoReadfileProvided = (filename) => {
    throw new Error(`Unable to load assembly support file ${JSON.stringify(filename)}: no readFile callback provided!`);
};
/**
 * Parses the assembly buffer and, if instructed to, redirects to the
 * compressed assembly buffer.
 *
 * @param assemblyBuffer buffer containing SPEC_FILE_NAME contents
 * @param readFile a callback to use for reading additional support files
 * @param validate whether or not to validate the assembly
 */
function loadAssemblyFromBuffer(assemblyBuffer, readFile = failNoReadfileProvided, validate = true) {
    let contents = JSON.parse(assemblyBuffer.toString('utf-8'));
    // check if the file holds instructions to the actual assembly file
    while ((0, redirect_1.isAssemblyRedirect)(contents)) {
        contents = followRedirect(contents, readFile);
    }
    return validate ? (0, validate_assembly_1.validateAssembly)(contents) : contents;
}
exports.loadAssemblyFromBuffer = loadAssemblyFromBuffer;
/**
 * Loads the assembly file and, if present, follows instructions
 * found in the file to unzip compressed assemblies.
 *
 * @param directory the directory of the assembly file
 * @param validate whether to validate the contents of the file
 * @returns the assembly file as an Assembly object
 */
function loadAssemblyFromPath(directory, validate = true) {
    const assemblyFile = findAssemblyFile(directory);
    return loadAssemblyFromFile(assemblyFile, validate);
}
exports.loadAssemblyFromPath = loadAssemblyFromPath;
/**
 * Loads the assembly file and, if present, follows instructions
 * found in the file to unzip compressed assemblies.
 *
 * @param pathToFile the path to the SPEC_FILE_NAME file
 * @param validate whether to validate the contents of the file
 * @returns the assembly file as an Assembly object
 */
function loadAssemblyFromFile(pathToFile, validate = true) {
    const data = fs.readFileSync(pathToFile);
    try {
        return loadAssemblyFromBuffer(data, (filename) => fs.readFileSync(path.resolve(pathToFile, '..', filename)), validate);
    }
    catch (e) {
        throw new Error(`Error loading assembly from file ${pathToFile}:\n${e}`);
    }
}
exports.loadAssemblyFromFile = loadAssemblyFromFile;
function followRedirect(assemblyRedirect, readFile) {
    // Validating the schema, this is cheap (the schema is small).
    (0, redirect_1.validateAssemblyRedirect)(assemblyRedirect);
    let data = readFile(assemblyRedirect.filename);
    switch (assemblyRedirect.compression) {
        case 'gzip':
            data = zlib.gunzipSync(data);
            break;
        case undefined:
            break;
        default:
            throw new Error(`Unsupported compression algorithm: ${JSON.stringify(assemblyRedirect.compression)}`);
    }
    const json = data.toString('utf-8');
    return JSON.parse(json);
}
//# sourceMappingURL=assembly-utils.js.map