"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformProviderGenerator = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const codemaker_1 = require("codemaker");
const commons_1 = require("@cdktf/commons");
const provider_schema_1 = require("@cdktf/provider-schema");
const resource_parser_1 = require("./resource-parser");
const emitter_1 = require("./emitter");
const isMatching = (target, terraformSchemaName) => {
    if (target.isModule)
        return false;
    const elements = terraformSchemaName.split("/");
    if (elements.length === 1) {
        return target.source === terraformSchemaName;
    }
    else {
        const [hostname, namespace, provider] = elements;
        if (!hostname || !namespace || !provider) {
            throw new Error(`can't handle ${terraformSchemaName}`);
        }
        // If target.source is set, try to match it
        if (target.source) {
            const targetSource = {
                // defaults
                hostname: "registry.terraform.io",
                namespace: "hashicorp",
                name: "",
            };
            const targetElements = target.source.split("/");
            switch (targetElements.length) {
                case 1: // only name set
                    targetSource.name = targetElements[0].toLowerCase();
                    break;
                case 2: // namespace and name set
                    targetSource.namespace = targetElements[0].toLowerCase();
                    targetSource.name = targetElements[1].toLowerCase();
                    break;
                case 3: // hostname, namespace and name set
                    targetSource.hostname = targetElements[0].toLowerCase();
                    targetSource.namespace = targetElements[1].toLowerCase();
                    targetSource.name = targetElements[2].toLowerCase();
                    break;
                default:
                    throw new Error(`can't handle ${target.source}. Expected string with 1, 2 or 3 elements separated by '/'`);
            }
            return (targetSource.hostname === hostname &&
                targetSource.namespace === namespace &&
                targetSource.name === provider);
        }
        // Else, try to match target.name to the provider name
        return target.name === provider;
    }
};
class TerraformProviderGenerator {
    constructor(code, schema) {
        this.code = code;
        this.schema = schema;
        this.resourceParser = new resource_parser_1.ResourceParser();
        this.versions = {};
        this.code.indentation = 2;
        this.resourceEmitter = new emitter_1.ResourceEmitter(this.code);
        this.structEmitter = new emitter_1.StructEmitter(this.code);
    }
    getProviderByConstraint(providerConstraint) {
        return Object.keys(this.schema.provider_schemas || {}).find((fqpn) => isMatching(providerConstraint, fqpn));
    }
    generate(providerConstraint) {
        var _a;
        const fqpn = this.getProviderByConstraint(providerConstraint);
        if (!fqpn) {
            commons_1.logger.debug(`Could not find provider constraint for ${providerConstraint} in schema: ${JSON.stringify(this.schema, null, 2)}`);
            throw new Error(`Could not find provider with constraint ${JSON.stringify(providerConstraint)}`);
        }
        const providerVersion = (_a = this.schema.provider_versions) === null || _a === void 0 ? void 0 : _a[fqpn];
        this.emitProvider(fqpn, providerVersion, providerConstraint);
        this.versions[fqpn] = providerVersion;
    }
    generateAll() {
        for (const fqpn of Object.keys(this.schema.provider_schemas || {})) {
            this.generate(new commons_1.ConstructsMakerProviderTarget(new commons_1.TerraformProviderConstraint(fqpn), commons_1.LANGUAGES[0]));
        }
    }
    async save(outdir) {
        await this.code.save(outdir);
    }
    buildResourceModels(fqpn, constraint) {
        var _a;
        const provider = (_a = this.schema.provider_schemas) === null || _a === void 0 ? void 0 : _a[fqpn];
        if (!provider) {
            throw new Error(`Can not find provider '${fqpn}' in schema`);
        }
        const resources = Object.entries(provider.resource_schemas || {}).map(([type, resource]) => this.resourceParser.parse(fqpn, type, resource, "resource", constraint));
        const dataSources = Object.entries(provider.data_source_schemas || {}).map(([type, resource]) => this.resourceParser.parse(fqpn, `data_${type}`, resource, "data_source", constraint));
        return [].concat(...resources, ...dataSources);
    }
    getClassNameForResource(terraformType) {
        return this.resourceParser.getClassNameForResource(terraformType);
    }
    getNamespaceNameForResource(terraformType) {
        return this.resourceParser.getNamespaceNameForResource(terraformType);
    }
    emitProvider(fqpn, providerVersion, constraint) {
        var _a;
        const name = (constraint === null || constraint === void 0 ? void 0 : constraint.name)
            ? constraint.name
            : (0, provider_schema_1.parseFQPN)(fqpn).name;
        const provider = (_a = this.schema.provider_schemas) === null || _a === void 0 ? void 0 : _a[fqpn];
        if (!provider) {
            throw new Error(`Can not find provider '${fqpn}' in schema`);
        }
        const files = [];
        this.buildResourceModels(fqpn, constraint).forEach((resourceModel) => {
            if (constraint) {
                resourceModel.providerVersionConstraint = constraint.version;
                resourceModel.terraformProviderSource = constraint.source;
            }
            resourceModel.providerVersion = providerVersion;
            if (resourceModel.structsRequireSharding) {
                files.push(this.emitResourceWithComplexStruct(resourceModel));
            }
            else {
                files.push(this.emitResource(resourceModel));
            }
            this.emitResourceReadme(resourceModel);
        });
        if (provider.provider) {
            const providerResource = this.resourceParser.parse(fqpn, `provider`, provider.provider, "provider", constraint);
            if (constraint) {
                providerResource.providerVersionConstraint = constraint.version;
                providerResource.terraformProviderSource = constraint.source;
                providerResource.terraformProviderName = constraint.name;
            }
            providerResource.providerVersion = providerVersion;
            files.push(this.emitResource(providerResource));
            this.emitResourceReadme(providerResource);
        }
        this.emitIndexFile(name, files);
        this.emitLazyIndexFile(name, files);
    }
    emitResourceReadme(resource) {
        const filePath = `${resource.namespaceFolderPath}/README.md`;
        this.code.openFile(filePath);
        this.code.line(`# \`${resource.terraformType}\``);
        this.code.line();
        const type = resource.isProvider
            ? resource.provider
            : resource.terraformType;
        this.code.line(`Refer to the Terraform Registry for docs: [\`${type}\`](${resource.linkToDocs}).`);
        this.code.closeFile(filePath);
    }
    emitIndexFile(provider, files) {
        const folder = `providers/${provider}`;
        const filePath = `${folder}/index.ts`;
        this.code.openFile(filePath);
        this.code.line("// generated by cdktf get");
        for (const file of files) {
            const dirName = file.replace(`${folder}/`, "").replace("/index.ts", "");
            this.code.line(`export * as ${(0, codemaker_1.toCamelCase)(dirName)} from './${dirName}';`);
        }
        this.code.line();
        this.code.closeFile(filePath);
    }
    emitLazyIndexFile(provider, files) {
        const folder = `providers/${provider}`;
        const filePath = `${folder}/lazy-index.ts`;
        this.code.openFile(filePath);
        this.code.line("// generated by cdktf get");
        for (const file of files) {
            const dirName = file.replace(`${folder}/`, "").replace("/index.ts", "");
            this.code.line(`Object.defineProperty(exports, '${(0, codemaker_1.toCamelCase)(dirName)}', { get: function () { return require('./${dirName}'); } });`);
        }
        this.code.line();
        this.code.closeFile(filePath);
    }
    emitResource(resource) {
        this.code.openFile(resource.filePath);
        this.emitFileHeader(resource);
        this.structEmitter.emit(resource);
        this.resourceEmitter.emit(resource);
        this.code.closeFile(resource.filePath);
        return resource.filePath;
    }
    emitResourceWithComplexStruct(resource) {
        const generatedFiles = [];
        // drop the last segment of the filepath
        const filePath = resource.filePath;
        this.code.openFile(filePath);
        this.code.line(`// generated from terraform resource schema`);
        this.code.line();
        if (resource.structsRequireSharding) {
            if (resource.referencedTypes.length > 0) {
                this.code.line(`import { ${resource.referencedTypes.join(", \n")}} from './${resource.structsFolderName}'`);
            }
            this.code.line(`export * from './${resource.structsFolderName}'`);
            resource.importStatements.forEach((statement) => this.code.line(statement));
            this.structEmitter.emitInterface(resource, resource.configStruct);
            this.resourceEmitter.emit(resource);
            this.code.closeFile(filePath);
            this.structEmitter.emit(resource);
            generatedFiles.push(resource.fileName);
            generatedFiles.push(resource.structsFolderName);
        }
        else {
            resource.importStatements.forEach((statement) => this.code.line(statement));
            this.structEmitter.emit(resource);
            this.resourceEmitter.emit(resource);
            this.code.closeFile(filePath);
            generatedFiles.push(resource.fileName);
        }
        return filePath;
    }
    emitFileHeader(resource) {
        this.code.line(`// ${resource.linkToDocs}`);
        this.code.line(`// generated from terraform resource schema`);
        this.code.line();
        resource.importStatements.forEach((statement) => this.code.line(statement));
        this.code.line();
        this.code.line("// Configuration");
        this.code.line();
    }
}
exports.TerraformProviderGenerator = TerraformProviderGenerator;
//# sourceMappingURL=data:application/json;base64,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