"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModuleGenerator = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const codemaker_1 = require("codemaker");
const models_1 = require("./models");
const sanitized_comments_1 = require("./sanitized-comments");
class ModuleGenerator {
    constructor(code, targets) {
        this.code = code;
        this.targets = targets;
        this.code.indentation = 2;
        for (const target of this.targets) {
            this.emitSubmodule(target);
        }
    }
    emitSubmodule(target) {
        const spec = target.spec;
        if (!spec) {
            throw new Error(`missing spec for ${target.fqn}`);
        }
        this.code.openFile(target.fileName);
        this.code.line(`// generated by cdktf get`);
        this.code.line(`// ${target.source}`);
        this.code.line(`import { TerraformModule, TerraformModuleUserConfig } from 'cdktf';`);
        this.code.line(`import { Construct } from 'constructs';`);
        const baseName = this.code.toPascalCase(target.name.replace(/[-/.]/g, "_"));
        const configType = `${baseName}Config`;
        this.code.openBlock(`export interface ${configType} extends TerraformModuleUserConfig`);
        for (const input of spec.inputs) {
            const optional = input.required && input.default === undefined ? "" : "?";
            const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
            if (input.description) {
                comment.line(input.description);
            }
            if (input.default) {
                comment.line(input.default);
            }
            if (input.type.includes("map(")) {
                comment.line(`The property type contains a map, they have special handling, please see {@link cdk.tf/module-map-inputs the docs}`);
            }
            comment.end();
            this.code.line(`readonly ${models_1.AttributeModel.escapeName((0, codemaker_1.toCamelCase)(input.name))}${optional}: ${parseType(input.type)};`);
        }
        this.code.closeBlock();
        // Add a link to the Terraform Registry if it is sourced from there
        // https://developer.hashicorp.com/terraform/language/modules/sources
        // Registry modules are referred to as <NAMESPACE>/<NAME>/<PROVIDER>
        // Other sources contain dots, e.g.
        // app.terraform.io/example-corp/k8s-cluster/azurerm (private registries)
        // github.com/hashicorp/example (Github)
        // git@github.com:hashicorp/example.git
        // bitbucket.org/hashicorp/terraform-consul-aws (Bitbucket)
        // ... and more
        // ../module and ./module (local paths)
        const isNonRegistryModule = target.source.includes(".");
        let registryPath;
        // Submodules also exist (e.g. terraform-aws-modules/vpc/aws//modules/vpc-endpoints)
        // And linking directly to them in the Registry requires including the version
        // like e.g. https://registry.terraform.io/modules/terraform-aws-modules/vpc/aws/latest/submodules/vpc-endpoints
        if (target.source.includes("//")) {
            registryPath = target.source.replace("//modules", `/${target.version || "latest"}/submodules`);
            // terraform-aws-modules/vpc/aws//modules/vpc-endpoints
            // ->
            // terraform-aws-modules/vpc/aws/latest/submodules/vpc-endpoints
        }
        else {
            // not submodule specified, just append the version
            registryPath = `${target.source}/${target.version || "latest"}`;
        }
        const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        comment.line(`Defines an ${baseName} based on a Terraform module`);
        comment.line(``);
        comment.line(isNonRegistryModule
            ? `Source at ${target.source}`
            : `Docs at Terraform Registry: {@link https://registry.terraform.io/modules/${registryPath} ${target.source}}`);
        comment.end();
        this.code.openBlock(`export class ${baseName} extends TerraformModule`);
        this.code.line(`private readonly inputs: { [name: string]: any } = { }`);
        const allOptional = spec.inputs.find((x) => x.required) ? "" : " = {}";
        this.code.open(`public constructor(scope: Construct, id: string, config: ${configType}${allOptional}) {`);
        this.code.open(`super(scope, id, {`);
        this.code.line("...config,");
        this.code.line(`source: '${target.source}',`);
        if (target.version) {
            this.code.line(`version: '${target.version}',`);
        }
        this.code.close(`});`);
        for (const input of spec.inputs) {
            const inputName = models_1.AttributeModel.escapeName((0, codemaker_1.toCamelCase)(input.name));
            this.code.line(`this.${inputName} = config.${inputName};`);
        }
        this.code.close(`}`); // ctor
        for (const input of spec.inputs) {
            const inputName = models_1.AttributeModel.escapeName((0, codemaker_1.toCamelCase)(input.name));
            const inputType = parseType(input.type) +
                (input.required && input.default === undefined ? "" : " | undefined");
            this.code.openBlock(`public get ${inputName}(): ${inputType}`);
            this.code.line(`return this.inputs['${input.name}'] as ${inputType};`);
            this.code.closeBlock();
            this.code.openBlock(`public set ${inputName}(value: ${inputType})`);
            this.code.line(`this.inputs['${input.name}'] = value;`);
            this.code.closeBlock();
        }
        for (const output of spec.outputs) {
            const outputName = (0, codemaker_1.toCamelCase)(output.name);
            this.code.openBlock(`public get ${outputName}Output()`);
            this.code.line(`return this.getString('${output.name}')`);
            this.code.closeBlock();
        }
        this.code.openBlock(`protected synthesizeAttributes()`);
        this.code.line(`return this.inputs;`);
        this.code.closeBlock();
        this.code.openBlock(`protected synthesizeHclAttributes(): { [name: string]: any }`);
        this.code.line(`return Object.fromEntries(`);
        this.code.line(`  Object.entries(this.inputs)`);
        this.code.line(`    .filter(([, val]) => val !== undefined)`);
        this.code.line(`    .map(([key, val]) => {`);
        this.code.line(`      return [`);
        this.code.line(`        key,`);
        this.code.line(`        {`);
        this.code.line(`          value: val,`);
        this.code.line(`          type: "any",`);
        this.code.line(`        },`);
        this.code.line(`      ];`);
        this.code.line(`    })`);
        this.code.line(`);`);
        this.code.closeBlock();
        this.code.closeBlock(); // class
        this.code.closeFile(target.fileName);
    }
}
exports.ModuleGenerator = ModuleGenerator;
function parseType(type) {
    if (type === "string") {
        return "string";
    }
    if (type === "number") {
        return "number";
    }
    if (type === "bool") {
        return "boolean";
    }
    if (type === "list") {
        return "string[]";
    }
    if (type === "map") {
        return "{ [key: string]: string }";
    }
    if (type === "tuple") {
        return "string[]";
    }
    if (type === "any") {
        return "any";
    }
    const complexType = parseComplexType(type);
    if (complexType) {
        return complexType;
    }
    throw new Error(`unknown type ${type}`);
}
function parseComplexType(type) {
    const complex = /^(object|list|map|set|tuple)\(([\s\S]+)\)/;
    const match = complex.exec(type);
    if (!match) {
        return undefined;
    }
    const [, kind, innerType] = match;
    if (kind === "object") {
        return `any`;
    }
    if (kind === "list" || kind === "set") {
        return `${parseType(innerType)}[]`;
    }
    if (kind === "tuple") {
        return `any[]`;
    }
    if (kind === "map") {
        return `{ [key: string]: ${parseType(innerType)} }`;
    }
    throw new Error(`unexpected kind ${kind}`);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW9kdWxlLWdlbmVyYXRvci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm1vZHVsZS1nZW5lcmF0b3IudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsK0JBQStCO0FBQy9CLG1DQUFtQztBQUNuQyx5Q0FBbUQ7QUFDbkQscUNBQTBDO0FBQzFDLDZEQUF3RDtBQUd4RCxNQUFhLGVBQWU7SUFDMUIsWUFDbUIsSUFBZSxFQUNmLE9BQXNDO1FBRHRDLFNBQUksR0FBSixJQUFJLENBQVc7UUFDZixZQUFPLEdBQVAsT0FBTyxDQUErQjtRQUV2RCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsR0FBRyxDQUFDLENBQUM7UUFFMUIsS0FBSyxNQUFNLE1BQU0sSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDbEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM3QixDQUFDO0lBQ0gsQ0FBQztJQUVPLGFBQWEsQ0FBQyxNQUFtQztRQUN2RCxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDO1FBRXpCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNWLE1BQU0sSUFBSSxLQUFLLENBQUMsb0JBQW9CLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxDQUFDO1FBQ3BELENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFcEMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsMkJBQTJCLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO1FBRXRDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLHFFQUFxRSxDQUN0RSxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMseUNBQXlDLENBQUMsQ0FBQztRQUUxRCxNQUFNLFFBQVEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUM1RSxNQUFNLFVBQVUsR0FBRyxHQUFHLFFBQVEsUUFBUSxDQUFDO1FBRXZDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQixvQkFBb0IsVUFBVSxvQ0FBb0MsQ0FDbkUsQ0FBQztRQUNGLEtBQUssTUFBTSxLQUFLLElBQUksSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO1lBQ2hDLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO1lBRTFFLE1BQU0sT0FBTyxHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzVDLElBQUksS0FBSyxDQUFDLFdBQVcsRUFBRSxDQUFDO2dCQUN0QixPQUFPLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUNsQyxDQUFDO1lBQ0QsSUFBSSxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBQ2xCLE9BQU8sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQzlCLENBQUM7WUFDRCxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUM7Z0JBQ2hDLE9BQU8sQ0FBQyxJQUFJLENBQ1Ysb0hBQW9ILENBQ3JILENBQUM7WUFDSixDQUFDO1lBQ0QsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBRWQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osWUFBWSx1QkFBYyxDQUFDLFVBQVUsQ0FDbkMsSUFBQSx1QkFBVyxFQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FDeEIsR0FBRyxRQUFRLEtBQUssU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUMxQyxDQUFDO1FBQ0osQ0FBQztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsbUVBQW1FO1FBQ25FLHFFQUFxRTtRQUNyRSxvRUFBb0U7UUFDcEUsbUNBQW1DO1FBQ25DLHlFQUF5RTtRQUN6RSx3Q0FBd0M7UUFDeEMsdUNBQXVDO1FBQ3ZDLDJEQUEyRDtRQUMzRCxlQUFlO1FBQ2YsdUNBQXVDO1FBQ3ZDLE1BQU0sbUJBQW1CLEdBQUcsTUFBTSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEQsSUFBSSxZQUFZLENBQUM7UUFDakIsb0ZBQW9GO1FBQ3BGLDhFQUE4RTtRQUM5RSxnSEFBZ0g7UUFDaEgsSUFBSSxNQUFNLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ2pDLFlBQVksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FDbEMsV0FBVyxFQUNYLElBQUksTUFBTSxDQUFDLE9BQU8sSUFBSSxRQUFRLGFBQWEsQ0FDNUMsQ0FBQztZQUNGLHVEQUF1RDtZQUN2RCxLQUFLO1lBQ0wsZ0VBQWdFO1FBQ2xFLENBQUM7YUFBTSxDQUFDO1lBQ04sbURBQW1EO1lBQ25ELFlBQVksR0FBRyxHQUFHLE1BQU0sQ0FBQyxNQUFNLElBQUksTUFBTSxDQUFDLE9BQU8sSUFBSSxRQUFRLEVBQUUsQ0FBQztRQUNsRSxDQUFDO1FBRUQsTUFBTSxPQUFPLEdBQUcsSUFBQSxxQ0FBZ0IsRUFBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDNUMsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLFFBQVEsOEJBQThCLENBQUMsQ0FBQztRQUNuRSxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE9BQU8sQ0FBQyxJQUFJLENBQ1YsbUJBQW1CO1lBQ2pCLENBQUMsQ0FBQyxhQUFhLE1BQU0sQ0FBQyxNQUFNLEVBQUU7WUFDOUIsQ0FBQyxDQUFDLDRFQUE0RSxZQUFZLElBQUksTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUNqSCxDQUFDO1FBQ0YsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2QsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLFFBQVEsMEJBQTBCLENBQUMsQ0FBQztRQUV4RSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO1FBRXpFLE1BQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDO1FBRXZFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLDREQUE0RCxVQUFVLEdBQUcsV0FBVyxLQUFLLENBQzFGLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQzdCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksTUFBTSxDQUFDLE1BQU0sSUFBSSxDQUFDLENBQUM7UUFDOUMsSUFBSSxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDbkIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxNQUFNLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQztRQUNsRCxDQUFDO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFdkIsS0FBSyxNQUFNLEtBQUssSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUM7WUFDaEMsTUFBTSxTQUFTLEdBQUcsdUJBQWMsQ0FBQyxVQUFVLENBQUMsSUFBQSx1QkFBVyxFQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ3JFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsU0FBUyxhQUFhLFNBQVMsR0FBRyxDQUFDLENBQUM7UUFDN0QsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsT0FBTztRQUU3QixLQUFLLE1BQU0sS0FBSyxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQztZQUNoQyxNQUFNLFNBQVMsR0FBRyx1QkFBYyxDQUFDLFVBQVUsQ0FBQyxJQUFBLHVCQUFXLEVBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDckUsTUFBTSxTQUFTLEdBQ2IsU0FBUyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUM7Z0JBQ3JCLENBQUMsS0FBSyxDQUFDLFFBQVEsSUFBSSxLQUFLLENBQUMsT0FBTyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQztZQUN4RSxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLFNBQVMsT0FBTyxTQUFTLEVBQUUsQ0FBQyxDQUFDO1lBQy9ELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHVCQUF1QixLQUFLLENBQUMsSUFBSSxTQUFTLFNBQVMsR0FBRyxDQUFDLENBQUM7WUFDdkUsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUV2QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLFNBQVMsV0FBVyxTQUFTLEdBQUcsQ0FBQyxDQUFDO1lBQ3BFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixLQUFLLENBQUMsSUFBSSxhQUFhLENBQUMsQ0FBQztZQUN4RCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3pCLENBQUM7UUFFRCxLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUNsQyxNQUFNLFVBQVUsR0FBRyxJQUFBLHVCQUFXLEVBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO1lBQzVDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsVUFBVSxVQUFVLENBQUMsQ0FBQztZQUN4RCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQywwQkFBMEIsTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLENBQUM7WUFDMUQsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN6QixDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsa0NBQWtDLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLDhEQUE4RCxDQUMvRCxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLDZDQUE2QyxDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQy9CLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQzVCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDeEMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUN6QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMzQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN6QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNyQixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxRQUFRO1FBQ2hDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUN2QyxDQUFDO0NBQ0Y7QUF2S0QsMENBdUtDO0FBRUQsU0FBUyxTQUFTLENBQUMsSUFBWTtJQUM3QixJQUFJLElBQUksS0FBSyxRQUFRLEVBQUUsQ0FBQztRQUN0QixPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBQ0QsSUFBSSxJQUFJLEtBQUssUUFBUSxFQUFFLENBQUM7UUFDdEIsT0FBTyxRQUFRLENBQUM7SUFDbEIsQ0FBQztJQUNELElBQUksSUFBSSxLQUFLLE1BQU0sRUFBRSxDQUFDO1FBQ3BCLE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFDRCxJQUFJLElBQUksS0FBSyxNQUFNLEVBQUUsQ0FBQztRQUNwQixPQUFPLFVBQVUsQ0FBQztJQUNwQixDQUFDO0lBQ0QsSUFBSSxJQUFJLEtBQUssS0FBSyxFQUFFLENBQUM7UUFDbkIsT0FBTywyQkFBMkIsQ0FBQztJQUNyQyxDQUFDO0lBQ0QsSUFBSSxJQUFJLEtBQUssT0FBTyxFQUFFLENBQUM7UUFDckIsT0FBTyxVQUFVLENBQUM7SUFDcEIsQ0FBQztJQUNELElBQUksSUFBSSxLQUFLLEtBQUssRUFBRSxDQUFDO1FBQ25CLE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVELE1BQU0sV0FBVyxHQUFHLGdCQUFnQixDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNDLElBQUksV0FBVyxFQUFFLENBQUM7UUFDaEIsT0FBTyxXQUFXLENBQUM7SUFDckIsQ0FBQztJQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLElBQUksRUFBRSxDQUFDLENBQUM7QUFDMUMsQ0FBQztBQUVELFNBQVMsZ0JBQWdCLENBQUMsSUFBWTtJQUNwQyxNQUFNLE9BQU8sR0FBRywyQ0FBMkMsQ0FBQztJQUM1RCxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2pDLElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUNYLE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFRCxNQUFNLENBQUMsRUFBRSxJQUFJLEVBQUUsU0FBUyxDQUFDLEdBQUcsS0FBSyxDQUFDO0lBRWxDLElBQUksSUFBSSxLQUFLLFFBQVEsRUFBRSxDQUFDO1FBQ3RCLE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVELElBQUksSUFBSSxLQUFLLE1BQU0sSUFBSSxJQUFJLEtBQUssS0FBSyxFQUFFLENBQUM7UUFDdEMsT0FBTyxHQUFHLFNBQVMsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDO0lBQ3JDLENBQUM7SUFFRCxJQUFJLElBQUksS0FBSyxPQUFPLEVBQUUsQ0FBQztRQUNyQixPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRUQsSUFBSSxJQUFJLEtBQUssS0FBSyxFQUFFLENBQUM7UUFDbkIsT0FBTyxvQkFBb0IsU0FBUyxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUM7SUFDdEQsQ0FBQztJQUVELE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLElBQUksRUFBRSxDQUFDLENBQUM7QUFDN0MsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmNcbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNUEwtMi4wXG5pbXBvcnQgeyBDb2RlTWFrZXIsIHRvQ2FtZWxDYXNlIH0gZnJvbSBcImNvZGVtYWtlclwiO1xuaW1wb3J0IHsgQXR0cmlidXRlTW9kZWwgfSBmcm9tIFwiLi9tb2RlbHNcIjtcbmltcG9ydCB7IHNhbml0aXplZENvbW1lbnQgfSBmcm9tIFwiLi9zYW5pdGl6ZWQtY29tbWVudHNcIjtcbmltcG9ydCB7IENvbnN0cnVjdHNNYWtlck1vZHVsZVRhcmdldCB9IGZyb20gXCJAY2RrdGYvY29tbW9uc1wiO1xuXG5leHBvcnQgY2xhc3MgTW9kdWxlR2VuZXJhdG9yIHtcbiAgY29uc3RydWN0b3IoXG4gICAgcHJpdmF0ZSByZWFkb25seSBjb2RlOiBDb2RlTWFrZXIsXG4gICAgcHJpdmF0ZSByZWFkb25seSB0YXJnZXRzOiBDb25zdHJ1Y3RzTWFrZXJNb2R1bGVUYXJnZXRbXSxcbiAgKSB7XG4gICAgdGhpcy5jb2RlLmluZGVudGF0aW9uID0gMjtcblxuICAgIGZvciAoY29uc3QgdGFyZ2V0IG9mIHRoaXMudGFyZ2V0cykge1xuICAgICAgdGhpcy5lbWl0U3VibW9kdWxlKHRhcmdldCk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBlbWl0U3VibW9kdWxlKHRhcmdldDogQ29uc3RydWN0c01ha2VyTW9kdWxlVGFyZ2V0KSB7XG4gICAgY29uc3Qgc3BlYyA9IHRhcmdldC5zcGVjO1xuXG4gICAgaWYgKCFzcGVjKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYG1pc3Npbmcgc3BlYyBmb3IgJHt0YXJnZXQuZnFufWApO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5vcGVuRmlsZSh0YXJnZXQuZmlsZU5hbWUpO1xuXG4gICAgdGhpcy5jb2RlLmxpbmUoYC8vIGdlbmVyYXRlZCBieSBjZGt0ZiBnZXRgKTtcbiAgICB0aGlzLmNvZGUubGluZShgLy8gJHt0YXJnZXQuc291cmNlfWApO1xuXG4gICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICBgaW1wb3J0IHsgVGVycmFmb3JtTW9kdWxlLCBUZXJyYWZvcm1Nb2R1bGVVc2VyQ29uZmlnIH0gZnJvbSAnY2RrdGYnO2AsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUubGluZShgaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7YCk7XG5cbiAgICBjb25zdCBiYXNlTmFtZSA9IHRoaXMuY29kZS50b1Bhc2NhbENhc2UodGFyZ2V0Lm5hbWUucmVwbGFjZSgvWy0vLl0vZywgXCJfXCIpKTtcbiAgICBjb25zdCBjb25maWdUeXBlID0gYCR7YmFzZU5hbWV9Q29uZmlnYDtcblxuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICBgZXhwb3J0IGludGVyZmFjZSAke2NvbmZpZ1R5cGV9IGV4dGVuZHMgVGVycmFmb3JtTW9kdWxlVXNlckNvbmZpZ2AsXG4gICAgKTtcbiAgICBmb3IgKGNvbnN0IGlucHV0IG9mIHNwZWMuaW5wdXRzKSB7XG4gICAgICBjb25zdCBvcHRpb25hbCA9IGlucHV0LnJlcXVpcmVkICYmIGlucHV0LmRlZmF1bHQgPT09IHVuZGVmaW5lZCA/IFwiXCIgOiBcIj9cIjtcblxuICAgICAgY29uc3QgY29tbWVudCA9IHNhbml0aXplZENvbW1lbnQodGhpcy5jb2RlKTtcbiAgICAgIGlmIChpbnB1dC5kZXNjcmlwdGlvbikge1xuICAgICAgICBjb21tZW50LmxpbmUoaW5wdXQuZGVzY3JpcHRpb24pO1xuICAgICAgfVxuICAgICAgaWYgKGlucHV0LmRlZmF1bHQpIHtcbiAgICAgICAgY29tbWVudC5saW5lKGlucHV0LmRlZmF1bHQpO1xuICAgICAgfVxuICAgICAgaWYgKGlucHV0LnR5cGUuaW5jbHVkZXMoXCJtYXAoXCIpKSB7XG4gICAgICAgIGNvbW1lbnQubGluZShcbiAgICAgICAgICBgVGhlIHByb3BlcnR5IHR5cGUgY29udGFpbnMgYSBtYXAsIHRoZXkgaGF2ZSBzcGVjaWFsIGhhbmRsaW5nLCBwbGVhc2Ugc2VlIHtAbGluayBjZGsudGYvbW9kdWxlLW1hcC1pbnB1dHMgdGhlIGRvY3N9YCxcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICAgIGNvbW1lbnQuZW5kKCk7XG5cbiAgICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgICBgcmVhZG9ubHkgJHtBdHRyaWJ1dGVNb2RlbC5lc2NhcGVOYW1lKFxuICAgICAgICAgIHRvQ2FtZWxDYXNlKGlucHV0Lm5hbWUpLFxuICAgICAgICApfSR7b3B0aW9uYWx9OiAke3BhcnNlVHlwZShpbnB1dC50eXBlKX07YCxcbiAgICAgICk7XG4gICAgfVxuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG5cbiAgICAvLyBBZGQgYSBsaW5rIHRvIHRoZSBUZXJyYWZvcm0gUmVnaXN0cnkgaWYgaXQgaXMgc291cmNlZCBmcm9tIHRoZXJlXG4gICAgLy8gaHR0cHM6Ly9kZXZlbG9wZXIuaGFzaGljb3JwLmNvbS90ZXJyYWZvcm0vbGFuZ3VhZ2UvbW9kdWxlcy9zb3VyY2VzXG4gICAgLy8gUmVnaXN0cnkgbW9kdWxlcyBhcmUgcmVmZXJyZWQgdG8gYXMgPE5BTUVTUEFDRT4vPE5BTUU+LzxQUk9WSURFUj5cbiAgICAvLyBPdGhlciBzb3VyY2VzIGNvbnRhaW4gZG90cywgZS5nLlxuICAgIC8vIGFwcC50ZXJyYWZvcm0uaW8vZXhhbXBsZS1jb3JwL2s4cy1jbHVzdGVyL2F6dXJlcm0gKHByaXZhdGUgcmVnaXN0cmllcylcbiAgICAvLyBnaXRodWIuY29tL2hhc2hpY29ycC9leGFtcGxlIChHaXRodWIpXG4gICAgLy8gZ2l0QGdpdGh1Yi5jb206aGFzaGljb3JwL2V4YW1wbGUuZ2l0XG4gICAgLy8gYml0YnVja2V0Lm9yZy9oYXNoaWNvcnAvdGVycmFmb3JtLWNvbnN1bC1hd3MgKEJpdGJ1Y2tldClcbiAgICAvLyAuLi4gYW5kIG1vcmVcbiAgICAvLyAuLi9tb2R1bGUgYW5kIC4vbW9kdWxlIChsb2NhbCBwYXRocylcbiAgICBjb25zdCBpc05vblJlZ2lzdHJ5TW9kdWxlID0gdGFyZ2V0LnNvdXJjZS5pbmNsdWRlcyhcIi5cIik7XG4gICAgbGV0IHJlZ2lzdHJ5UGF0aDtcbiAgICAvLyBTdWJtb2R1bGVzIGFsc28gZXhpc3QgKGUuZy4gdGVycmFmb3JtLWF3cy1tb2R1bGVzL3ZwYy9hd3MvL21vZHVsZXMvdnBjLWVuZHBvaW50cylcbiAgICAvLyBBbmQgbGlua2luZyBkaXJlY3RseSB0byB0aGVtIGluIHRoZSBSZWdpc3RyeSByZXF1aXJlcyBpbmNsdWRpbmcgdGhlIHZlcnNpb25cbiAgICAvLyBsaWtlIGUuZy4gaHR0cHM6Ly9yZWdpc3RyeS50ZXJyYWZvcm0uaW8vbW9kdWxlcy90ZXJyYWZvcm0tYXdzLW1vZHVsZXMvdnBjL2F3cy9sYXRlc3Qvc3VibW9kdWxlcy92cGMtZW5kcG9pbnRzXG4gICAgaWYgKHRhcmdldC5zb3VyY2UuaW5jbHVkZXMoXCIvL1wiKSkge1xuICAgICAgcmVnaXN0cnlQYXRoID0gdGFyZ2V0LnNvdXJjZS5yZXBsYWNlKFxuICAgICAgICBcIi8vbW9kdWxlc1wiLFxuICAgICAgICBgLyR7dGFyZ2V0LnZlcnNpb24gfHwgXCJsYXRlc3RcIn0vc3VibW9kdWxlc2AsXG4gICAgICApO1xuICAgICAgLy8gdGVycmFmb3JtLWF3cy1tb2R1bGVzL3ZwYy9hd3MvL21vZHVsZXMvdnBjLWVuZHBvaW50c1xuICAgICAgLy8gLT5cbiAgICAgIC8vIHRlcnJhZm9ybS1hd3MtbW9kdWxlcy92cGMvYXdzL2xhdGVzdC9zdWJtb2R1bGVzL3ZwYy1lbmRwb2ludHNcbiAgICB9IGVsc2Uge1xuICAgICAgLy8gbm90IHN1Ym1vZHVsZSBzcGVjaWZpZWQsIGp1c3QgYXBwZW5kIHRoZSB2ZXJzaW9uXG4gICAgICByZWdpc3RyeVBhdGggPSBgJHt0YXJnZXQuc291cmNlfS8ke3RhcmdldC52ZXJzaW9uIHx8IFwibGF0ZXN0XCJ9YDtcbiAgICB9XG5cbiAgICBjb25zdCBjb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGNvbW1lbnQubGluZShgRGVmaW5lcyBhbiAke2Jhc2VOYW1lfSBiYXNlZCBvbiBhIFRlcnJhZm9ybSBtb2R1bGVgKTtcbiAgICBjb21tZW50LmxpbmUoYGApO1xuICAgIGNvbW1lbnQubGluZShcbiAgICAgIGlzTm9uUmVnaXN0cnlNb2R1bGVcbiAgICAgICAgPyBgU291cmNlIGF0ICR7dGFyZ2V0LnNvdXJjZX1gXG4gICAgICAgIDogYERvY3MgYXQgVGVycmFmb3JtIFJlZ2lzdHJ5OiB7QGxpbmsgaHR0cHM6Ly9yZWdpc3RyeS50ZXJyYWZvcm0uaW8vbW9kdWxlcy8ke3JlZ2lzdHJ5UGF0aH0gJHt0YXJnZXQuc291cmNlfX1gLFxuICAgICk7XG4gICAgY29tbWVudC5lbmQoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKGBleHBvcnQgY2xhc3MgJHtiYXNlTmFtZX0gZXh0ZW5kcyBUZXJyYWZvcm1Nb2R1bGVgKTtcblxuICAgIHRoaXMuY29kZS5saW5lKGBwcml2YXRlIHJlYWRvbmx5IGlucHV0czogeyBbbmFtZTogc3RyaW5nXTogYW55IH0gPSB7IH1gKTtcblxuICAgIGNvbnN0IGFsbE9wdGlvbmFsID0gc3BlYy5pbnB1dHMuZmluZCgoeCkgPT4geC5yZXF1aXJlZCkgPyBcIlwiIDogXCIgPSB7fVwiO1xuXG4gICAgdGhpcy5jb2RlLm9wZW4oXG4gICAgICBgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGNvbmZpZzogJHtjb25maWdUeXBlfSR7YWxsT3B0aW9uYWx9KSB7YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5vcGVuKGBzdXBlcihzY29wZSwgaWQsIHtgKTtcbiAgICB0aGlzLmNvZGUubGluZShcIi4uLmNvbmZpZyxcIik7XG4gICAgdGhpcy5jb2RlLmxpbmUoYHNvdXJjZTogJyR7dGFyZ2V0LnNvdXJjZX0nLGApO1xuICAgIGlmICh0YXJnZXQudmVyc2lvbikge1xuICAgICAgdGhpcy5jb2RlLmxpbmUoYHZlcnNpb246ICcke3RhcmdldC52ZXJzaW9ufScsYCk7XG4gICAgfVxuICAgIHRoaXMuY29kZS5jbG9zZShgfSk7YCk7XG5cbiAgICBmb3IgKGNvbnN0IGlucHV0IG9mIHNwZWMuaW5wdXRzKSB7XG4gICAgICBjb25zdCBpbnB1dE5hbWUgPSBBdHRyaWJ1dGVNb2RlbC5lc2NhcGVOYW1lKHRvQ2FtZWxDYXNlKGlucHV0Lm5hbWUpKTtcbiAgICAgIHRoaXMuY29kZS5saW5lKGB0aGlzLiR7aW5wdXROYW1lfSA9IGNvbmZpZy4ke2lucHV0TmFtZX07YCk7XG4gICAgfVxuXG4gICAgdGhpcy5jb2RlLmNsb3NlKGB9YCk7IC8vIGN0b3JcblxuICAgIGZvciAoY29uc3QgaW5wdXQgb2Ygc3BlYy5pbnB1dHMpIHtcbiAgICAgIGNvbnN0IGlucHV0TmFtZSA9IEF0dHJpYnV0ZU1vZGVsLmVzY2FwZU5hbWUodG9DYW1lbENhc2UoaW5wdXQubmFtZSkpO1xuICAgICAgY29uc3QgaW5wdXRUeXBlID1cbiAgICAgICAgcGFyc2VUeXBlKGlucHV0LnR5cGUpICtcbiAgICAgICAgKGlucHV0LnJlcXVpcmVkICYmIGlucHV0LmRlZmF1bHQgPT09IHVuZGVmaW5lZCA/IFwiXCIgOiBcIiB8IHVuZGVmaW5lZFwiKTtcbiAgICAgIHRoaXMuY29kZS5vcGVuQmxvY2soYHB1YmxpYyBnZXQgJHtpbnB1dE5hbWV9KCk6ICR7aW5wdXRUeXBlfWApO1xuICAgICAgdGhpcy5jb2RlLmxpbmUoYHJldHVybiB0aGlzLmlucHV0c1snJHtpbnB1dC5uYW1lfSddIGFzICR7aW5wdXRUeXBlfTtgKTtcbiAgICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG5cbiAgICAgIHRoaXMuY29kZS5vcGVuQmxvY2soYHB1YmxpYyBzZXQgJHtpbnB1dE5hbWV9KHZhbHVlOiAke2lucHV0VHlwZX0pYCk7XG4gICAgICB0aGlzLmNvZGUubGluZShgdGhpcy5pbnB1dHNbJyR7aW5wdXQubmFtZX0nXSA9IHZhbHVlO2ApO1xuICAgICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgICB9XG5cbiAgICBmb3IgKGNvbnN0IG91dHB1dCBvZiBzcGVjLm91dHB1dHMpIHtcbiAgICAgIGNvbnN0IG91dHB1dE5hbWUgPSB0b0NhbWVsQ2FzZShvdXRwdXQubmFtZSk7XG4gICAgICB0aGlzLmNvZGUub3BlbkJsb2NrKGBwdWJsaWMgZ2V0ICR7b3V0cHV0TmFtZX1PdXRwdXQoKWApO1xuICAgICAgdGhpcy5jb2RlLmxpbmUoYHJldHVybiB0aGlzLmdldFN0cmluZygnJHtvdXRwdXQubmFtZX0nKWApO1xuICAgICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgICB9XG5cbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKGBwcm90ZWN0ZWQgc3ludGhlc2l6ZUF0dHJpYnV0ZXMoKWApO1xuICAgIHRoaXMuY29kZS5saW5lKGByZXR1cm4gdGhpcy5pbnB1dHM7YCk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICBgcHJvdGVjdGVkIHN5bnRoZXNpemVIY2xBdHRyaWJ1dGVzKCk6IHsgW25hbWU6IHN0cmluZ106IGFueSB9YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5saW5lKGByZXR1cm4gT2JqZWN0LmZyb21FbnRyaWVzKGApO1xuICAgIHRoaXMuY29kZS5saW5lKGAgIE9iamVjdC5lbnRyaWVzKHRoaXMuaW5wdXRzKWApO1xuICAgIHRoaXMuY29kZS5saW5lKGAgICAgLmZpbHRlcigoWywgdmFsXSkgPT4gdmFsICE9PSB1bmRlZmluZWQpYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYCAgICAubWFwKChba2V5LCB2YWxdKSA9PiB7YCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYCAgICAgIHJldHVybiBbYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYCAgICAgICAga2V5LGApO1xuICAgIHRoaXMuY29kZS5saW5lKGAgICAgICAgIHtgKTtcbiAgICB0aGlzLmNvZGUubGluZShgICAgICAgICAgIHZhbHVlOiB2YWwsYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYCAgICAgICAgICB0eXBlOiBcImFueVwiLGApO1xuICAgIHRoaXMuY29kZS5saW5lKGAgICAgICAgIH0sYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYCAgICAgIF07YCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYCAgICB9KWApO1xuICAgIHRoaXMuY29kZS5saW5lKGApO2ApO1xuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG5cbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpOyAvLyBjbGFzc1xuICAgIHRoaXMuY29kZS5jbG9zZUZpbGUodGFyZ2V0LmZpbGVOYW1lKTtcbiAgfVxufVxuXG5mdW5jdGlvbiBwYXJzZVR5cGUodHlwZTogc3RyaW5nKSB7XG4gIGlmICh0eXBlID09PSBcInN0cmluZ1wiKSB7XG4gICAgcmV0dXJuIFwic3RyaW5nXCI7XG4gIH1cbiAgaWYgKHR5cGUgPT09IFwibnVtYmVyXCIpIHtcbiAgICByZXR1cm4gXCJudW1iZXJcIjtcbiAgfVxuICBpZiAodHlwZSA9PT0gXCJib29sXCIpIHtcbiAgICByZXR1cm4gXCJib29sZWFuXCI7XG4gIH1cbiAgaWYgKHR5cGUgPT09IFwibGlzdFwiKSB7XG4gICAgcmV0dXJuIFwic3RyaW5nW11cIjtcbiAgfVxuICBpZiAodHlwZSA9PT0gXCJtYXBcIikge1xuICAgIHJldHVybiBcInsgW2tleTogc3RyaW5nXTogc3RyaW5nIH1cIjtcbiAgfVxuICBpZiAodHlwZSA9PT0gXCJ0dXBsZVwiKSB7XG4gICAgcmV0dXJuIFwic3RyaW5nW11cIjtcbiAgfVxuICBpZiAodHlwZSA9PT0gXCJhbnlcIikge1xuICAgIHJldHVybiBcImFueVwiO1xuICB9XG5cbiAgY29uc3QgY29tcGxleFR5cGUgPSBwYXJzZUNvbXBsZXhUeXBlKHR5cGUpO1xuICBpZiAoY29tcGxleFR5cGUpIHtcbiAgICByZXR1cm4gY29tcGxleFR5cGU7XG4gIH1cblxuICB0aHJvdyBuZXcgRXJyb3IoYHVua25vd24gdHlwZSAke3R5cGV9YCk7XG59XG5cbmZ1bmN0aW9uIHBhcnNlQ29tcGxleFR5cGUodHlwZTogc3RyaW5nKTogc3RyaW5nIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgY29tcGxleCA9IC9eKG9iamVjdHxsaXN0fG1hcHxzZXR8dHVwbGUpXFwoKFtcXHNcXFNdKylcXCkvO1xuICBjb25zdCBtYXRjaCA9IGNvbXBsZXguZXhlYyh0eXBlKTtcbiAgaWYgKCFtYXRjaCkge1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cblxuICBjb25zdCBbLCBraW5kLCBpbm5lclR5cGVdID0gbWF0Y2g7XG5cbiAgaWYgKGtpbmQgPT09IFwib2JqZWN0XCIpIHtcbiAgICByZXR1cm4gYGFueWA7XG4gIH1cblxuICBpZiAoa2luZCA9PT0gXCJsaXN0XCIgfHwga2luZCA9PT0gXCJzZXRcIikge1xuICAgIHJldHVybiBgJHtwYXJzZVR5cGUoaW5uZXJUeXBlKX1bXWA7XG4gIH1cblxuICBpZiAoa2luZCA9PT0gXCJ0dXBsZVwiKSB7XG4gICAgcmV0dXJuIGBhbnlbXWA7XG4gIH1cblxuICBpZiAoa2luZCA9PT0gXCJtYXBcIikge1xuICAgIHJldHVybiBgeyBba2V5OiBzdHJpbmddOiAke3BhcnNlVHlwZShpbm5lclR5cGUpfSB9YDtcbiAgfVxuXG4gIHRocm93IG5ldyBFcnJvcihgdW5leHBlY3RlZCBraW5kICR7a2luZH1gKTtcbn1cbiJdfQ==