"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceModel = exports.STRUCT_SHARDING_THRESHOLD = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const codemaker_1 = require("codemaker");
const path_1 = __importDefault(require("path"));
const provider_schema_1 = require("@cdktf/provider-schema");
const struct_1 = require("./struct");
// Limit is 1200 to prevent stack size error.
// Could increase now that calculation is more accurate;
// however, probably better to have smaller
exports.STRUCT_SHARDING_THRESHOLD = 400;
class ResourceModel {
    constructor(options) {
        this.recursiveAttributePaths = {};
        this.className = options.className;
        this.filePath = options.filePath;
        this.terraformType = options.terraformType;
        this.baseName = options.baseName;
        this.attributes = options.attributes;
        this.schema = options.schema;
        this.fqpn = options.fqpn;
        this.provider = (0, provider_schema_1.parseFQPN)(options.fqpn).name;
        this.terraformProviderName = this.provider;
        this.fileName = options.fileName;
        this._structs = options.structs;
        this.terraformSchemaType = options.terraformSchemaType;
        this.configStructName = options.configStructName;
        this.dependencies = [
            `import { Construct } from 'constructs';`,
            `import * as cdktf from 'cdktf';`,
        ];
        this.structsRequireSharding =
            this._structs.reduce((partialSum, s) => partialSum + s.exportCount, 0) >
                exports.STRUCT_SHARDING_THRESHOLD;
    }
    get structs() {
        return [this.configStruct, ...this._structs];
    }
    get configStruct() {
        return new struct_1.ConfigStruct(this.configStructName, this.attributes);
    }
    get synthesizableAttributes() {
        return this.configStruct.assignableAttributes;
    }
    get importStatements() {
        return this.dependencies;
    }
    get schemaAsJson() {
        return this.escapeSchema(JSON.stringify(this.schema, undefined, 2));
    }
    get linkToDocs() {
        const { hostname, namespace, name } = (0, provider_schema_1.parseFQPN)(this.fqpn);
        const version = this.providerVersion || "latest";
        const base = `https://${hostname}/providers/${namespace}/${name}/${version}/docs`;
        if (this.isProvider)
            return base;
        if (this.isDataSource)
            return `${base}/data-sources/${this.terraformDocName}`;
        return `${base}/resources/${this.terraformDocName}`;
    }
    get isProvider() {
        return this.terraformSchemaType === "provider";
    }
    get isDataSource() {
        return this.terraformSchemaType === "data_source";
    }
    get parentClassName() {
        return this.isProvider
            ? "TerraformProvider"
            : this.isDataSource
                ? "TerraformDataSource"
                : "TerraformResource";
    }
    get terraformResourceType() {
        return this.isProvider
            ? this.terraformProviderName
            : this.isDataSource
                ? this.terraformType.replace(/^data_/, "")
                : this.terraformType;
    }
    get terraformDocName() {
        return (0, codemaker_1.toSnakeCase)(this.terraformResourceType.replace(new RegExp(`^${this.provider}_`, "i"), ""));
    }
    get referencedTypes() {
        return this.configStruct.referencedTypes;
    }
    get structsFolderName() {
        return `${path_1.default.basename(this.fileName, ".ts")}-structs`;
    }
    get structsFolderPath() {
        return path_1.default.join(this.namespaceFolderPath, this.structsFolderName);
    }
    get namespaceFolderPath() {
        return this.filePath.split("/").slice(0, -1).join("/");
    }
    escapeSchema(schema) {
        return schema.replace(/(\*\/)/gi, `*\\/`);
    }
}
exports.ResourceModel = ResourceModel;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2UtbW9kZWwuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJyZXNvdXJjZS1tb2RlbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSwrQkFBK0I7QUFDL0IsbUNBQW1DO0FBQ25DLHlDQUF3QztBQUN4QyxnREFBd0I7QUFDeEIsNERBQXVFO0FBRXZFLHFDQUFnRDtBQUdoRCw2Q0FBNkM7QUFDN0Msd0RBQXdEO0FBQ3hELDJDQUEyQztBQUM5QixRQUFBLHlCQUF5QixHQUFHLEdBQUcsQ0FBQztBQWlCN0MsTUFBYSxhQUFhO0lBeUJ4QixZQUFZLE9BQTZCO1FBRmxDLDRCQUF1QixHQUE0QyxFQUFFLENBQUM7UUFHM0UsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUMsU0FBUyxDQUFDO1FBQ25DLElBQUksQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDLFFBQVEsQ0FBQztRQUNqQyxJQUFJLENBQUMsYUFBYSxHQUFHLE9BQU8sQ0FBQyxhQUFhLENBQUM7UUFDM0MsSUFBSSxDQUFDLFFBQVEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQ2pDLElBQUksQ0FBQyxVQUFVLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQztRQUNyQyxJQUFJLENBQUMsTUFBTSxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUM7UUFDN0IsSUFBSSxDQUFDLElBQUksR0FBRyxPQUFPLENBQUMsSUFBSSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBQSwyQkFBUyxFQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDN0MsSUFBSSxDQUFDLHFCQUFxQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDM0MsSUFBSSxDQUFDLFFBQVEsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDO1FBQ2pDLElBQUksQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FBQztRQUNoQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsT0FBTyxDQUFDLG1CQUFtQixDQUFDO1FBQ3ZELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsZ0JBQWdCLENBQUM7UUFDakQsSUFBSSxDQUFDLFlBQVksR0FBRztZQUNsQix5Q0FBeUM7WUFDekMsaUNBQWlDO1NBQ2xDLENBQUM7UUFFRixJQUFJLENBQUMsc0JBQXNCO1lBQ3pCLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsVUFBVSxHQUFHLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO2dCQUN0RSxpQ0FBeUIsQ0FBQztJQUM5QixDQUFDO0lBRUQsSUFBVyxPQUFPO1FBQ2hCLE9BQU8sQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQy9DLENBQUM7SUFFRCxJQUFXLFlBQVk7UUFDckIsT0FBTyxJQUFJLHFCQUFZLENBQUMsSUFBSSxDQUFDLGdCQUFnQixFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBRUQsSUFBVyx1QkFBdUI7UUFDaEMsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLG9CQUFvQixDQUFDO0lBQ2hELENBQUM7SUFFRCxJQUFXLGdCQUFnQjtRQUN6QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUM7SUFDM0IsQ0FBQztJQUVELElBQVcsWUFBWTtRQUNyQixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3RFLENBQUM7SUFFRCxJQUFXLFVBQVU7UUFDbkIsTUFBTSxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLEdBQUcsSUFBQSwyQkFBUyxFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUMzRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsZUFBZSxJQUFJLFFBQVEsQ0FBQztRQUNqRCxNQUFNLElBQUksR0FBRyxXQUFXLFFBQVEsY0FBYyxTQUFTLElBQUksSUFBSSxJQUFJLE9BQU8sT0FBTyxDQUFDO1FBQ2xGLElBQUksSUFBSSxDQUFDLFVBQVU7WUFBRSxPQUFPLElBQUksQ0FBQztRQUNqQyxJQUFJLElBQUksQ0FBQyxZQUFZO1lBQ25CLE9BQU8sR0FBRyxJQUFJLGlCQUFpQixJQUFJLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztRQUN6RCxPQUFPLEdBQUcsSUFBSSxjQUFjLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxDQUFDO0lBQ3RELENBQUM7SUFFRCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxJQUFJLENBQUMsbUJBQW1CLEtBQUssVUFBVSxDQUFDO0lBQ2pELENBQUM7SUFFRCxJQUFXLFlBQVk7UUFDckIsT0FBTyxJQUFJLENBQUMsbUJBQW1CLEtBQUssYUFBYSxDQUFDO0lBQ3BELENBQUM7SUFFRCxJQUFXLGVBQWU7UUFDeEIsT0FBTyxJQUFJLENBQUMsVUFBVTtZQUNwQixDQUFDLENBQUMsbUJBQW1CO1lBQ3JCLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBWTtnQkFDakIsQ0FBQyxDQUFDLHFCQUFxQjtnQkFDdkIsQ0FBQyxDQUFDLG1CQUFtQixDQUFDO0lBQzVCLENBQUM7SUFFRCxJQUFXLHFCQUFxQjtRQUM5QixPQUFPLElBQUksQ0FBQyxVQUFVO1lBQ3BCLENBQUMsQ0FBQyxJQUFJLENBQUMscUJBQXFCO1lBQzVCLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBWTtnQkFDakIsQ0FBQyxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUM7Z0JBQzFDLENBQUMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO0lBQzNCLENBQUM7SUFFRCxJQUFXLGdCQUFnQjtRQUN6QixPQUFPLElBQUEsdUJBQVcsRUFDaEIsSUFBSSxDQUFDLHFCQUFxQixDQUFDLE9BQU8sQ0FDaEMsSUFBSSxNQUFNLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQ3JDLEVBQUUsQ0FDSCxDQUNGLENBQUM7SUFDSixDQUFDO0lBRUQsSUFBVyxlQUFlO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLENBQUM7SUFDM0MsQ0FBQztJQUVELElBQVcsaUJBQWlCO1FBQzFCLE9BQU8sR0FBRyxjQUFJLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLFVBQVUsQ0FBQztJQUMxRCxDQUFDO0lBRUQsSUFBVyxpQkFBaUI7UUFDMUIsT0FBTyxjQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztJQUNyRSxDQUFDO0lBRUQsSUFBVyxtQkFBbUI7UUFDNUIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3pELENBQUM7SUFFTyxZQUFZLENBQUMsTUFBYztRQUNqQyxPQUFPLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBQzVDLENBQUM7Q0FDRjtBQW5JRCxzQ0FtSUMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgKGMpIEhhc2hpQ29ycCwgSW5jXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTVBMLTIuMFxuaW1wb3J0IHsgdG9TbmFrZUNhc2UgfSBmcm9tIFwiY29kZW1ha2VyXCI7XG5pbXBvcnQgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IHsgRlFQTiwgcGFyc2VGUVBOLCBQcm92aWRlck5hbWUgfSBmcm9tIFwiQGNka3RmL3Byb3ZpZGVyLXNjaGVtYVwiO1xuaW1wb3J0IHsgQXR0cmlidXRlTW9kZWwgfSBmcm9tIFwiLi9hdHRyaWJ1dGUtbW9kZWxcIjtcbmltcG9ydCB7IFN0cnVjdCwgQ29uZmlnU3RydWN0IH0gZnJvbSBcIi4vc3RydWN0XCI7XG5pbXBvcnQgeyBTY2hlbWEgfSBmcm9tIFwiQGNka3RmL2NvbW1vbnNcIjtcblxuLy8gTGltaXQgaXMgMTIwMCB0byBwcmV2ZW50IHN0YWNrIHNpemUgZXJyb3IuXG4vLyBDb3VsZCBpbmNyZWFzZSBub3cgdGhhdCBjYWxjdWxhdGlvbiBpcyBtb3JlIGFjY3VyYXRlO1xuLy8gaG93ZXZlciwgcHJvYmFibHkgYmV0dGVyIHRvIGhhdmUgc21hbGxlclxuZXhwb3J0IGNvbnN0IFNUUlVDVF9TSEFSRElOR19USFJFU0hPTEQgPSA0MDA7XG5cbmludGVyZmFjZSBSZXNvdXJjZU1vZGVsT3B0aW9ucyB7XG4gIHRlcnJhZm9ybVR5cGU6IHN0cmluZztcbiAgY2xhc3NOYW1lOiBzdHJpbmc7XG4gIGJhc2VOYW1lOiBzdHJpbmc7XG4gIGZpbGVOYW1lOiBzdHJpbmc7XG4gIGZpbGVQYXRoOiBzdHJpbmc7XG4gIGF0dHJpYnV0ZXM6IEF0dHJpYnV0ZU1vZGVsW107XG4gIHN0cnVjdHM6IFN0cnVjdFtdO1xuICBmcXBuOiBGUVBOO1xuICBzY2hlbWE6IFNjaGVtYTtcbiAgdGVycmFmb3JtU2NoZW1hVHlwZTogc3RyaW5nO1xuICBjb25maWdTdHJ1Y3ROYW1lOiBzdHJpbmc7XG59XG5cbnR5cGUgRGVmaW5pdGlvblBhdGggPSBzdHJpbmc7XG5leHBvcnQgY2xhc3MgUmVzb3VyY2VNb2RlbCB7XG4gIHB1YmxpYyBjbGFzc05hbWU6IHN0cmluZztcbiAgcHVibGljIGZpbGVQYXRoOiBzdHJpbmc7XG4gIHB1YmxpYyB0ZXJyYWZvcm1UeXBlOiBzdHJpbmc7XG4gIHB1YmxpYyBiYXNlTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcHJvdmlkZXI6IFByb3ZpZGVyTmFtZTtcbiAgcHVibGljIGZxcG46IEZRUE47XG4gIHB1YmxpYyBwcm92aWRlclZlcnNpb25Db25zdHJhaW50Pzogc3RyaW5nO1xuICBwdWJsaWMgcHJvdmlkZXJWZXJzaW9uPzogc3RyaW5nO1xuICBwdWJsaWMgdGVycmFmb3JtUHJvdmlkZXJTb3VyY2U/OiBzdHJpbmc7XG4gIC8qXG4gICAqIHVzdWFsbHkgc2FtZSBhcyBwcm92aWRlciwgYnV0IGNhbiBiZSBvdmVycmlkZGVuIGlmIHVzZXIgcGlja3MgYSBkaWZmZXJlbnQgbmFtZVxuICAgKiB0byBiZSBhYmxlIHRvIHVzZSB0d28gcHJvdmlkZXJzIHdpdGggdGhlIHNhbWUgbmFtZVxuICAgKi9cbiAgcHVibGljIHRlcnJhZm9ybVByb3ZpZGVyTmFtZTogc3RyaW5nO1xuICBwdWJsaWMgZmlsZU5hbWU6IHN0cmluZztcbiAgcHVibGljIGF0dHJpYnV0ZXM6IEF0dHJpYnV0ZU1vZGVsW107XG4gIHB1YmxpYyBzY2hlbWE6IFNjaGVtYTtcbiAgcHJpdmF0ZSBfc3RydWN0czogU3RydWN0W107XG4gIHByaXZhdGUgZGVwZW5kZW5jaWVzOiBzdHJpbmdbXTtcbiAgcHJpdmF0ZSB0ZXJyYWZvcm1TY2hlbWFUeXBlOiBzdHJpbmc7XG4gIHByaXZhdGUgY29uZmlnU3RydWN0TmFtZTogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgc3RydWN0c1JlcXVpcmVTaGFyZGluZzogYm9vbGVhbjtcbiAgcHVibGljIHJlY3Vyc2l2ZUF0dHJpYnV0ZVBhdGhzOiB7IFt1c2FnZVBhdGg6IHN0cmluZ106IERlZmluaXRpb25QYXRoIH0gPSB7fTtcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBSZXNvdXJjZU1vZGVsT3B0aW9ucykge1xuICAgIHRoaXMuY2xhc3NOYW1lID0gb3B0aW9ucy5jbGFzc05hbWU7XG4gICAgdGhpcy5maWxlUGF0aCA9IG9wdGlvbnMuZmlsZVBhdGg7XG4gICAgdGhpcy50ZXJyYWZvcm1UeXBlID0gb3B0aW9ucy50ZXJyYWZvcm1UeXBlO1xuICAgIHRoaXMuYmFzZU5hbWUgPSBvcHRpb25zLmJhc2VOYW1lO1xuICAgIHRoaXMuYXR0cmlidXRlcyA9IG9wdGlvbnMuYXR0cmlidXRlcztcbiAgICB0aGlzLnNjaGVtYSA9IG9wdGlvbnMuc2NoZW1hO1xuICAgIHRoaXMuZnFwbiA9IG9wdGlvbnMuZnFwbjtcbiAgICB0aGlzLnByb3ZpZGVyID0gcGFyc2VGUVBOKG9wdGlvbnMuZnFwbikubmFtZTtcbiAgICB0aGlzLnRlcnJhZm9ybVByb3ZpZGVyTmFtZSA9IHRoaXMucHJvdmlkZXI7XG4gICAgdGhpcy5maWxlTmFtZSA9IG9wdGlvbnMuZmlsZU5hbWU7XG4gICAgdGhpcy5fc3RydWN0cyA9IG9wdGlvbnMuc3RydWN0cztcbiAgICB0aGlzLnRlcnJhZm9ybVNjaGVtYVR5cGUgPSBvcHRpb25zLnRlcnJhZm9ybVNjaGVtYVR5cGU7XG4gICAgdGhpcy5jb25maWdTdHJ1Y3ROYW1lID0gb3B0aW9ucy5jb25maWdTdHJ1Y3ROYW1lO1xuICAgIHRoaXMuZGVwZW5kZW5jaWVzID0gW1xuICAgICAgYGltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO2AsXG4gICAgICBgaW1wb3J0ICogYXMgY2RrdGYgZnJvbSAnY2RrdGYnO2AsXG4gICAgXTtcblxuICAgIHRoaXMuc3RydWN0c1JlcXVpcmVTaGFyZGluZyA9XG4gICAgICB0aGlzLl9zdHJ1Y3RzLnJlZHVjZSgocGFydGlhbFN1bSwgcykgPT4gcGFydGlhbFN1bSArIHMuZXhwb3J0Q291bnQsIDApID5cbiAgICAgIFNUUlVDVF9TSEFSRElOR19USFJFU0hPTEQ7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHN0cnVjdHMoKTogU3RydWN0W10ge1xuICAgIHJldHVybiBbdGhpcy5jb25maWdTdHJ1Y3QsIC4uLnRoaXMuX3N0cnVjdHNdO1xuICB9XG5cbiAgcHVibGljIGdldCBjb25maWdTdHJ1Y3QoKSB7XG4gICAgcmV0dXJuIG5ldyBDb25maWdTdHJ1Y3QodGhpcy5jb25maWdTdHJ1Y3ROYW1lLCB0aGlzLmF0dHJpYnV0ZXMpO1xuICB9XG5cbiAgcHVibGljIGdldCBzeW50aGVzaXphYmxlQXR0cmlidXRlcygpOiBBdHRyaWJ1dGVNb2RlbFtdIHtcbiAgICByZXR1cm4gdGhpcy5jb25maWdTdHJ1Y3QuYXNzaWduYWJsZUF0dHJpYnV0ZXM7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGltcG9ydFN0YXRlbWVudHMoKTogc3RyaW5nW10ge1xuICAgIHJldHVybiB0aGlzLmRlcGVuZGVuY2llcztcbiAgfVxuXG4gIHB1YmxpYyBnZXQgc2NoZW1hQXNKc29uKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMuZXNjYXBlU2NoZW1hKEpTT04uc3RyaW5naWZ5KHRoaXMuc2NoZW1hLCB1bmRlZmluZWQsIDIpKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgbGlua1RvRG9jcygpOiBzdHJpbmcge1xuICAgIGNvbnN0IHsgaG9zdG5hbWUsIG5hbWVzcGFjZSwgbmFtZSB9ID0gcGFyc2VGUVBOKHRoaXMuZnFwbik7XG4gICAgY29uc3QgdmVyc2lvbiA9IHRoaXMucHJvdmlkZXJWZXJzaW9uIHx8IFwibGF0ZXN0XCI7XG4gICAgY29uc3QgYmFzZSA9IGBodHRwczovLyR7aG9zdG5hbWV9L3Byb3ZpZGVycy8ke25hbWVzcGFjZX0vJHtuYW1lfS8ke3ZlcnNpb259L2RvY3NgO1xuICAgIGlmICh0aGlzLmlzUHJvdmlkZXIpIHJldHVybiBiYXNlO1xuICAgIGlmICh0aGlzLmlzRGF0YVNvdXJjZSlcbiAgICAgIHJldHVybiBgJHtiYXNlfS9kYXRhLXNvdXJjZXMvJHt0aGlzLnRlcnJhZm9ybURvY05hbWV9YDtcbiAgICByZXR1cm4gYCR7YmFzZX0vcmVzb3VyY2VzLyR7dGhpcy50ZXJyYWZvcm1Eb2NOYW1lfWA7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGlzUHJvdmlkZXIoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMudGVycmFmb3JtU2NoZW1hVHlwZSA9PT0gXCJwcm92aWRlclwiO1xuICB9XG5cbiAgcHVibGljIGdldCBpc0RhdGFTb3VyY2UoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMudGVycmFmb3JtU2NoZW1hVHlwZSA9PT0gXCJkYXRhX3NvdXJjZVwiO1xuICB9XG5cbiAgcHVibGljIGdldCBwYXJlbnRDbGFzc05hbWUoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5pc1Byb3ZpZGVyXG4gICAgICA/IFwiVGVycmFmb3JtUHJvdmlkZXJcIlxuICAgICAgOiB0aGlzLmlzRGF0YVNvdXJjZVxuICAgICAgICA/IFwiVGVycmFmb3JtRGF0YVNvdXJjZVwiXG4gICAgICAgIDogXCJUZXJyYWZvcm1SZXNvdXJjZVwiO1xuICB9XG5cbiAgcHVibGljIGdldCB0ZXJyYWZvcm1SZXNvdXJjZVR5cGUoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy5pc1Byb3ZpZGVyXG4gICAgICA/IHRoaXMudGVycmFmb3JtUHJvdmlkZXJOYW1lXG4gICAgICA6IHRoaXMuaXNEYXRhU291cmNlXG4gICAgICAgID8gdGhpcy50ZXJyYWZvcm1UeXBlLnJlcGxhY2UoL15kYXRhXy8sIFwiXCIpXG4gICAgICAgIDogdGhpcy50ZXJyYWZvcm1UeXBlO1xuICB9XG5cbiAgcHVibGljIGdldCB0ZXJyYWZvcm1Eb2NOYW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRvU25ha2VDYXNlKFxuICAgICAgdGhpcy50ZXJyYWZvcm1SZXNvdXJjZVR5cGUucmVwbGFjZShcbiAgICAgICAgbmV3IFJlZ0V4cChgXiR7dGhpcy5wcm92aWRlcn1fYCwgXCJpXCIpLFxuICAgICAgICBcIlwiLFxuICAgICAgKSxcbiAgICApO1xuICB9XG5cbiAgcHVibGljIGdldCByZWZlcmVuY2VkVHlwZXMoKTogc3RyaW5nW10ge1xuICAgIHJldHVybiB0aGlzLmNvbmZpZ1N0cnVjdC5yZWZlcmVuY2VkVHlwZXM7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHN0cnVjdHNGb2xkZXJOYW1lKCk6IHN0cmluZyB7XG4gICAgcmV0dXJuIGAke3BhdGguYmFzZW5hbWUodGhpcy5maWxlTmFtZSwgXCIudHNcIil9LXN0cnVjdHNgO1xuICB9XG5cbiAgcHVibGljIGdldCBzdHJ1Y3RzRm9sZGVyUGF0aCgpOiBzdHJpbmcge1xuICAgIHJldHVybiBwYXRoLmpvaW4odGhpcy5uYW1lc3BhY2VGb2xkZXJQYXRoLCB0aGlzLnN0cnVjdHNGb2xkZXJOYW1lKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgbmFtZXNwYWNlRm9sZGVyUGF0aCgpOiBzdHJpbmcge1xuICAgIHJldHVybiB0aGlzLmZpbGVQYXRoLnNwbGl0KFwiL1wiKS5zbGljZSgwLCAtMSkuam9pbihcIi9cIik7XG4gIH1cblxuICBwcml2YXRlIGVzY2FwZVNjaGVtYShzY2hlbWE6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHNjaGVtYS5yZXBsYWNlKC8oXFwqXFwvKS9naSwgYCpcXFxcL2ApO1xuICB9XG59XG4iXX0=