"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MapAttributeTypeModel = exports.SetAttributeTypeModel = exports.ListAttributeTypeModel = exports.StructAttributeTypeModel = exports.SimpleAttributeTypeModel = exports.SkippedAttributeTypeModel = void 0;
const util_1 = require("../../../util");
class SkippedAttributeTypeModel {
    constructor() { }
    get typeModelType() {
        return "simple";
    }
    get struct() {
        return undefined;
    }
    get isComplex() {
        return false;
    }
    getStoredClassInitializer(_name) {
        // not used
        return "";
    }
    get storedClassType() {
        return "any";
    }
    get inputTypeDefinition() {
        return "any";
    }
    getAttributeAccessFunction(name) {
        return `this.interpolationForAttribute('${name}')`;
    }
    get toTerraformFunction() {
        return `cdktf.anyToTerraform`;
    }
    get toHclTerraformFunction() {
        return `cdktf.anyToHclTerraform`;
    }
    get hasReferenceClass() {
        return false;
    }
    get isTokenizable() {
        return false;
    }
}
exports.SkippedAttributeTypeModel = SkippedAttributeTypeModel;
class SimpleAttributeTypeModel {
    constructor(type) {
        this.type = type;
    }
    get typeModelType() {
        return "simple";
    }
    get struct() {
        return undefined;
    }
    get isComplex() {
        return false;
    }
    getStoredClassInitializer(_name) {
        // not used
        return "";
    }
    get storedClassType() {
        return this.type;
    }
    get inputTypeDefinition() {
        if (this.type === "boolean") {
            return "boolean | cdktf.IResolvable";
        }
        else {
            return this.type;
        }
    }
    getAttributeAccessFunction(name) {
        return `this.get${(0, util_1.uppercaseFirst)(this.type)}Attribute('${name}')`;
    }
    get toTerraformFunction() {
        return `cdktf.${this.type}ToTerraform`;
    }
    get toHclTerraformFunction() {
        return `cdktf.${this.type}ToHclTerraform`;
    }
    get hasReferenceClass() {
        return false;
    }
    get isTokenizable() {
        return true;
    }
}
exports.SimpleAttributeTypeModel = SimpleAttributeTypeModel;
class StructAttributeTypeModel {
    constructor(struct) {
        this.struct = struct;
    }
    get typeModelType() {
        return "struct";
    }
    get isComplex() {
        return true;
    }
    getStoredClassInitializer(name) {
        return `new ${this.struct.outputReferenceName}(this, "${name}")`;
    }
    get storedClassType() {
        return this.struct.name;
    }
    get inputTypeDefinition() {
        return this.struct.name;
    }
    getAttributeAccessFunction(name) {
        // This shouln't actually be called
        return `this.interpolationForAttribute('${name}')`;
    }
    get toTerraformFunction() {
        return `${(0, util_1.downcaseFirst)(this.struct.name)}ToTerraform`;
    }
    get toHclTerraformFunction() {
        return `${(0, util_1.downcaseFirst)(this.struct.name)}ToHclTerraform`;
    }
    get hasReferenceClass() {
        return true;
    }
    get isTokenizable() {
        return false;
    }
}
exports.StructAttributeTypeModel = StructAttributeTypeModel;
class ListAttributeTypeModel {
    constructor(elementType, isSingleItem, isBlock) {
        this.elementType = elementType;
        this.isSingleItem = isSingleItem;
        this.isBlock = isBlock;
        if (this.struct) {
            this.struct.isSingleItem = this.isSingleItem || false;
        }
    }
    get typeModelType() {
        return "list";
    }
    get struct() {
        return this.elementType.struct;
    }
    get isComplex() {
        return this.elementType.isComplex;
    }
    getStoredClassInitializer(name) {
        if (this.isSingleItem) {
            return `new ${this.elementType.storedClassType}OutputReference(this, "${name}")`;
        }
        else {
            if (this.isComplex) {
                return `new ${this.storedClassType}(this, "${name}", false)`;
            }
            else {
                return `new cdktf.${(0, util_1.uppercaseFirst)(this.storedClassType)}(this, "${name}", false)`;
            }
        }
    }
    get storedClassType() {
        return `${this.elementType.storedClassType}List`;
    }
    get inputTypeDefinition() {
        if (this.isSingleItem) {
            return this.elementType.inputTypeDefinition;
        }
        else if (this.elementType.storedClassType === "boolean") {
            return "Array<boolean | cdktf.IResolvable> | cdktf.IResolvable";
        }
        else if (this.isComplex) {
            return `${this.elementType.storedClassType}[] | cdktf.IResolvable`;
        }
        else if (this.elementType.typeModelType !== "simple") {
            return `${this.elementType.inputTypeDefinition}[] | cdktf.IResolvable`;
        }
        else {
            return `${this.elementType.inputTypeDefinition}[]`;
        }
    }
    getAttributeAccessFunction(name) {
        switch (this.elementType.storedClassType) {
            case "string":
                return `this.getListAttribute('${name}')`;
            case "number":
                return `this.getNumberListAttribute('${name}')`;
        }
        return `this.interpolationForAttribute('${name}')`;
    }
    get toTerraformFunction() {
        if (this.isSingleItem) {
            return this.elementType.toTerraformFunction;
        }
        return `cdktf.listMapper(${this.elementType.toTerraformFunction}, ${this.isBlock})`;
    }
    get toHclTerraformFunction() {
        if (this.isSingleItem) {
            return this.elementType.toHclTerraformFunction;
        }
        return `cdktf.listMapperHcl(${this.elementType.toHclTerraformFunction}, ${this.isBlock})`;
    }
    get hasReferenceClass() {
        return this.isSingleItem || this.isComplex;
    }
    get isTokenizable() {
        switch (this.elementType.storedClassType) {
            case "string":
                return true;
            case "number":
                return true;
            default:
                return false;
        }
    }
}
exports.ListAttributeTypeModel = ListAttributeTypeModel;
class SetAttributeTypeModel {
    constructor(elementType, isSingleItem, isBlock) {
        this.elementType = elementType;
        this.isSingleItem = isSingleItem;
        this.isBlock = isBlock;
        if (this.struct) {
            this.struct.isSingleItem = this.isSingleItem || false;
        }
    }
    get typeModelType() {
        return "set";
    }
    get struct() {
        return this.elementType.struct;
    }
    get isComplex() {
        return this.elementType.isComplex;
    }
    getStoredClassInitializer(name) {
        if (this.isSingleItem) {
            return `new ${this.elementType.storedClassType}OutputReference(this, "${name}")`;
        }
        else {
            if (this.isComplex) {
                return `new ${this.storedClassType}(this, "${name}", true)`;
            }
            else {
                return `new cdktf.${(0, util_1.uppercaseFirst)(this.storedClassType)}(this, "${name}", true)`;
            }
        }
    }
    get storedClassType() {
        return `${this.elementType.storedClassType}List`;
    }
    get inputTypeDefinition() {
        if (this.isSingleItem) {
            return this.elementType.inputTypeDefinition;
        }
        else if (this.elementType.storedClassType === "boolean") {
            return "Array<boolean | cdktf.IResolvable> | cdktf.IResolvable";
        }
        else if (this.isComplex) {
            return `${this.elementType.storedClassType}[] | cdktf.IResolvable`;
        }
        else if (this.elementType.typeModelType !== "simple") {
            return `${this.elementType.inputTypeDefinition}[] | cdktf.IResolvable`;
        }
        else {
            return `${this.elementType.inputTypeDefinition}[]`;
        }
    }
    getAttributeAccessFunction(name) {
        switch (this.elementType.storedClassType) {
            case "string":
                return `cdktf.Fn.tolist(this.getListAttribute('${name}'))`;
            case "number":
                return `cdktf.Token.asNumberList(cdktf.Fn.tolist(this.getNumberListAttribute('${name}')))`;
        }
        // Token.asAny is required because tolist returns an Array encoded token which the listMapper
        // would try to map over when this is passed to another resource. With Token.asAny() it is left
        // as is by the listMapper and is later properly resolved to a reference
        // (this only works in TypeScript currently, same as for referencing lists
        // [see "interpolationForAttribute(...)" further below])
        return `cdktf.Token.asAny(cdktf.Fn.tolist(this.interpolationForAttribute('${name}')))`;
    }
    get toTerraformFunction() {
        if (this.isSingleItem) {
            return this.elementType.toTerraformFunction;
        }
        else {
            return `cdktf.listMapper(${this.elementType.toTerraformFunction}, ${this.isBlock})`;
        }
    }
    get toHclTerraformFunction() {
        if (this.isSingleItem) {
            return this.elementType.toHclTerraformFunction;
        }
        return `cdktf.listMapperHcl(${this.elementType.toHclTerraformFunction}, ${this.isBlock})`;
    }
    get hasReferenceClass() {
        return this.isSingleItem || this.isComplex;
    }
    get isTokenizable() {
        switch (this.elementType.storedClassType) {
            case "string":
                return true;
            case "number":
                return true;
            default:
                return false;
        }
    }
}
exports.SetAttributeTypeModel = SetAttributeTypeModel;
class MapAttributeTypeModel {
    constructor(elementType) {
        this.elementType = elementType;
    }
    get typeModelType() {
        return "map";
    }
    get struct() {
        return this.elementType.struct;
    }
    get isComplex() {
        return this.elementType.isComplex;
    }
    getStoredClassInitializer(name) {
        if (this.isComplex) {
            return `new ${this.storedClassType}(this, "${name}")`;
        }
        else {
            return `new cdktf.${(0, util_1.uppercaseFirst)(this.storedClassType)}(this, "${name}")`;
        }
    }
    get storedClassType() {
        return `${this.elementType.storedClassType}Map`;
    }
    get inputTypeDefinition() {
        // map of booleans has token support, but booleans don't
        if (this.elementType.storedClassType === "boolean") {
            return `{ [key: string]: (${this.elementType.storedClassType} | cdktf.IResolvable) }`;
        }
        else if (this.isComplex) {
            return `{ [key: string]: ${this.elementType.storedClassType} } | cdktf.IResolvable`;
        }
        else if (this.elementType.typeModelType !== "simple") {
            return `{ [key: string]: ${this.elementType.inputTypeDefinition} } | cdktf.IResolvable`;
        }
        else {
            return `{ [key: string]: ${this.elementType.storedClassType} }`;
        }
    }
    getAttributeAccessFunction(name) {
        if (!this.isComplex && this.elementType.typeModelType !== "simple") {
            return `this.interpolationForAttribute('${name}')`;
        }
        return `this.get${(0, util_1.uppercaseFirst)(this.storedClassType)}Attribute('${name}')`;
    }
    get toTerraformFunction() {
        return `cdktf.hashMapper(${this.elementType.toTerraformFunction})`;
    }
    get toHclTerraformFunction() {
        return `cdktf.hashMapperHcl(${this.elementType.toHclTerraformFunction})`;
    }
    get hasReferenceClass() {
        return this.isComplex;
    }
    get isTokenizable() {
        switch (this.elementType.storedClassType) {
            case "string":
                return true;
            case "number":
                return true;
            case "boolean":
                return true;
            case "any":
                return true;
            default:
                return false;
        }
    }
}
exports.MapAttributeTypeModel = MapAttributeTypeModel;
//# sourceMappingURL=data:application/json;base64,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