"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AttributeModel = exports.escapeAttributeName = void 0;
function escapeAttributeName(name) {
    // `self` and `build` doesn't work in as property name in Python
    if (name === "self" || name === "build")
        return `${name}Attribute`;
    // jsii can't handle `getFoo` properties, since it's incompatible with Java
    if (name.match(/^get[A-Za-z]+/))
        return name.replace("get", "fetch");
    // `equals` is a prohibited name in jsii
    if (name === "equals")
        return "equalTo";
    // `node` is already used by the Constructs base class
    if (name === "node")
        return "nodeAttribute";
    // `System` shadows built-in types in CSharp (see #1420)
    if (name === "system")
        return "systemAttribute";
    // `tfResourceType` is already used by resources to distinguish between different resource types
    if (name === "tfResourceType")
        return `${name}Attribute`;
    // `importFrom` has potential for common name collision with providers
    if (name === "importFrom")
        return `${name}Attribute`;
    // `move` could have common name collision with providers
    if (name === "moveTo")
        return `${name}Attribute`;
    // `software` attribute can be confused with the JSII Java runtime package (see #3115)
    if (name === "software")
        return `${name}Attribute`;
    return name;
}
exports.escapeAttributeName = escapeAttributeName;
class AttributeModel {
    constructor(options) {
        this.storageName = options.storageName;
        this._name = options.name;
        this.type = options.type;
        this.optional = options.optional;
        this.computed = options.computed;
        this.terraformName = options.terraformName;
        this.terraformFullName = options.terraformFullName;
        this._description = options.description;
        this.provider = options.provider;
        this.required = options.required;
        this.forcePlainGetterType = options.forcePlainGetterType;
    }
    get typeDefinition() {
        const optional = this.optional ? "?" : "";
        return `${this.name}${optional}: ${this.type.inputTypeDefinition}`;
    }
    get isAssignable() {
        return this.required || this.optional;
    }
    get isOptional() {
        return this.optional;
    }
    get isRequired() {
        return this.required;
    }
    get isProvider() {
        return this.provider;
    }
    get getterType() {
        let getterType = { _type: "plain" };
        if (this.forcePlainGetterType) {
            return getterType;
        }
        if (this.isProvider) {
            return getterType;
        }
        if (this.type.hasReferenceClass) {
            getterType = {
                _type: "stored_class",
            };
        }
        else if (this.computed &&
            !this.isAssignable &&
            (!this.type.isTokenizable || this.type.typeModelType === "map")) {
            getterType = {
                _type: "stored_class",
            };
        }
        return getterType;
    }
    get isStored() {
        return this.isAssignable;
    }
    get setterType() {
        if (!this.isStored) {
            return { _type: "none" };
        }
        if (this.getterType._type === "stored_class") {
            return {
                _type: "stored_class",
                type: this.type.inputTypeDefinition,
            };
        }
        return {
            _type: "set",
            type: `${this.type.inputTypeDefinition}${this.isProvider ? " | undefined" : ""}`,
        };
    }
    get name() {
        return AttributeModel.escapeName(this._name);
    }
    static escapeName(name) {
        return escapeAttributeName(name);
    }
    get description() {
        var _a;
        return (_a = this._description) === null || _a === void 0 ? void 0 : _a.replace(/'''/gi, "```");
    }
    getReferencedTypes(isConfigStruct) {
        const attTypeStruct = this.type.struct;
        if (!attTypeStruct) {
            return undefined;
        }
        const types = [];
        if (this.isAssignable) {
            types.push(attTypeStruct.name);
            types.push(attTypeStruct.mapperName);
            types.push(attTypeStruct.hclMapperName);
        }
        if (!attTypeStruct.isSingleItem &&
            (attTypeStruct.nestingMode === "list" ||
                attTypeStruct.nestingMode === "set")) {
            types.push(attTypeStruct.listName);
        }
        else if (attTypeStruct.nestingMode === "map") {
            types.push(attTypeStruct.mapName);
        }
        else if (attTypeStruct.nestingMode === "maplist") {
            types.push(attTypeStruct.mapListName);
        }
        else if (attTypeStruct.nestingMode === "mapset") {
            types.push(attTypeStruct.mapListName);
        }
        else if (attTypeStruct.nestingMode === "listmap") {
            types.push(attTypeStruct.listMapName);
        }
        else if (attTypeStruct.nestingMode === "setmap") {
            types.push(attTypeStruct.listMapName);
        }
        else if (attTypeStruct.nestingMode === "listlist" ||
            attTypeStruct.nestingMode === "listset" ||
            attTypeStruct.nestingMode === "setlist" ||
            attTypeStruct.nestingMode == "setset") {
            types.push(attTypeStruct.listListName);
        }
        else if (!isConfigStruct) {
            types.push(attTypeStruct.outputReferenceName);
        }
        // other types of nested collections aren't supported
        return types;
    }
}
exports.AttributeModel = AttributeModel;
//# sourceMappingURL=data:application/json;base64,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