"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StructEmitter = void 0;
const models_1 = require("../models");
const attributes_emitter_1 = require("./attributes-emitter");
const util_1 = require("../../../util");
const path = __importStar(require("path"));
const resource_model_1 = require("../models/resource-model");
const sanitized_comments_1 = require("../sanitized-comments");
class StructEmitter {
    constructor(code) {
        this.code = code;
        this.attributesEmitter = new attributes_emitter_1.AttributesEmitter(this.code);
    }
    emit(resource) {
        if (resource.structsRequireSharding) {
            this.emitNamespacedStructs(resource);
        }
        else {
            this.emitStructs(resource);
        }
    }
    // Due to https://github.com/hashicorp/terraform-plugin-sdk/commit/2387eb85e32c064b4a62718c9f5c80bf00dc7fb9 all
    // resources from providers using the old SDK have the id field by default
    // We have no way to distinguish them through the provider schema, so a word of warning for our users
    warnAboutIdField(att) {
        if (att.name === "id") {
            this.code.line(`*`);
            this.code.line(`* Please be aware that the id field is automatically added to all resources in Terraform providers using a Terraform provider SDK version below 2.`);
            this.code.line(`* If you experience problems setting this value it might not be settable. Please take a look at the provider documentation to ensure it should be settable.`);
        }
    }
    emitInterface(resource, struct, name = struct.name) {
        if (resource.isProvider) {
            this.code.openBlock(`export interface ${name}`);
        }
        else {
            this.code.openBlock(`export interface ${name}${struct.extends}`);
        }
        for (const att of struct.assignableAttributes) {
            const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
            if (att.description) {
                comment.line(att.description);
                comment.line(``);
                comment.line(`Docs at Terraform Registry: {@link ${resource.linkToDocs}#${att.terraformName} ${resource.className}#${att.terraformName}}`);
                this.warnAboutIdField(att);
                comment.end();
            }
            else {
                comment.line(`Docs at Terraform Registry: {@link ${resource.linkToDocs}#${att.terraformName} ${resource.className}#${att.terraformName}}`);
                this.warnAboutIdField(att);
                comment.end();
            }
            this.code.line(`readonly ${att.typeDefinition};`);
        }
        this.code.closeBlock();
        if (!(struct instanceof models_1.ConfigStruct)) {
            this.emitToTerraformFunction(struct);
            this.emitToHclTerraformFunction(struct);
        }
    }
    emitStructs(resource) {
        resource.structs.forEach((struct) => {
            if (struct instanceof models_1.ConfigStruct) {
                this.emitInterface(resource, struct);
            }
            else {
                // We use the interface here for the configuration / inputs of a resource / nested block
                this.emitInterface(resource, struct);
                // And we use the class for the attributes / outputs of a resource / nested block
                if (!struct.isProvider) {
                    this.emitClass(struct);
                }
            }
        });
    }
    emitNamespacedStructs(resource) {
        // iterate over all structs in batches of 400 to avoid too many exports (> 1200)
        const structImports = {};
        // drop configStruct from resource.structs to avoid double import
        const structsWithoutConfigStruct = resource.structs.slice(1);
        const structSplits = [[]];
        const splitCounts = [0];
        structsWithoutConfigStruct.forEach((struct) => {
            if (splitCounts[splitCounts.length - 1] + struct.exportCount <=
                resource_model_1.STRUCT_SHARDING_THRESHOLD) {
                structSplits[structSplits.length - 1].push(struct);
                splitCounts[splitCounts.length - 1] += struct.exportCount;
            }
            else {
                structSplits.push([struct]);
                splitCounts.push(struct.exportCount);
            }
        });
        // fill the struct imports mapping before code generation so that imports can
        // point to not yet generated struct files
        for (let i = 0; i < structSplits.length; i++) {
            const structFilename = `structs${i * resource_model_1.STRUCT_SHARDING_THRESHOLD}.ts`;
            const structs = structSplits[i];
            // associate current structs batch with the file it will be written to
            // to find it in subsequent files for importing
            structs.forEach((struct) => (structImports[struct.name] = structFilename));
        }
        const structPaths = [];
        for (let i = 0; i < structSplits.length; i++) {
            const structsToImport = {};
            const structs = structSplits[i];
            const structFilename = `structs${i * resource_model_1.STRUCT_SHARDING_THRESHOLD}.ts`;
            structPaths.push(structFilename);
            // find all structs that need to be imported in this file
            structs.forEach((struct) => {
                struct.attributes.forEach((att) => {
                    var _a, _b;
                    const structTypeName = (_a = att.type.struct) === null || _a === void 0 ? void 0 : _a.name;
                    const fileToImport = structImports[structTypeName !== null && structTypeName !== void 0 ? structTypeName : ""];
                    if (fileToImport && fileToImport !== structFilename) {
                        const attTypeStruct = att.type.struct;
                        if (!attTypeStruct)
                            throw new Error(`${structTypeName} is not a struct`);
                        (_b = structsToImport[fileToImport]) !== null && _b !== void 0 ? _b : (structsToImport[fileToImport] = []);
                        const attReferences = att.getReferencedTypes(struct instanceof models_1.ConfigStruct);
                        if (attReferences) {
                            structsToImport[fileToImport].push(...attReferences);
                        }
                    }
                });
            });
            const namespacedFilePath = path.join(resource.structsFolderPath, structFilename);
            this.code.openFile(namespacedFilePath);
            // the structs only makes use of cdktf not constructs
            this.code.line(`import * as cdktf from 'cdktf';`);
            Object.entries(structsToImport).forEach(([fileToImport, structs]) => {
                this.code.line(`import { ${[...new Set(structs)].join(",\n")} } from './${path.basename(fileToImport, ".ts")}'`);
            });
            structs.forEach((struct) => {
                if (struct instanceof models_1.ConfigStruct) {
                    this.emitInterface(resource, struct);
                }
                else {
                    // We use the interface here for the configuration / inputs of a resource / nested block
                    this.emitInterface(resource, struct);
                    // And we use the class for the attributes / outputs of a resource / nested block
                    if (!struct.isProvider) {
                        this.emitClass(struct);
                    }
                }
            });
            this.code.closeFile(namespacedFilePath);
        }
        // emit the index file that exports all the struct files we've just generated
        const indexFilePath = path.join(resource.structsFolderPath, "index.ts");
        this.code.openFile(indexFilePath);
        structPaths.forEach((structPath) => {
            this.code.line(`export * from './${path.basename(structPath, ".ts")}'`);
        });
        this.code.closeFile(indexFilePath);
    }
    emitClass(struct) {
        this.code.openBlock(`export class ${struct.outputReferenceName} extends cdktf.ComplexObject`);
        this.code.line("private isEmptyObject = false;");
        if (!struct.isClass) {
            this.code.line("private resolvableValue?: cdktf.IResolvable;");
        }
        this.code.line();
        const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        comment.line(`@param terraformResource The parent resource`);
        comment.line(`@param terraformAttribute The attribute on the parent resource this class is referencing`);
        if (struct.isSingleItem) {
            comment.end();
            this.code.openBlock(`public constructor(terraformResource: cdktf.IInterpolatingParent, terraformAttribute: string)`);
            this.code.line(`super(terraformResource, terraformAttribute, false, 0);`);
            this.code.closeBlock();
        }
        else if (struct.nestingMode === "single" ||
            struct.nestingMode === "object") {
            comment.end();
            this.code.openBlock(`public constructor(terraformResource: cdktf.IInterpolatingParent, terraformAttribute: string)`);
            this.code.line(`super(terraformResource, terraformAttribute, false);`);
            this.code.closeBlock();
        }
        else if (struct.nestingMode.startsWith("map")) {
            this.code.line(`* @param complexObjectKey the key of this item in the map`);
            comment.end();
            this.code.openBlock(`public constructor(terraformResource: cdktf.IInterpolatingParent, terraformAttribute: string, complexObjectKey: string)`);
            this.code.line(`super(terraformResource, terraformAttribute, false, complexObjectKey);`);
            this.code.closeBlock();
        }
        else {
            comment.line(`@param complexObjectIndex the index of this item in the list`);
            comment.line(`@param complexObjectIsFromSet whether the list is wrapping a set (will add tolist() to be able to access an item via an index)`);
            comment.end();
            this.code.openBlock(`public constructor(terraformResource: cdktf.IInterpolatingParent, terraformAttribute: string, complexObjectIndex: number, complexObjectIsFromSet: boolean)`);
            this.code.line(`super(terraformResource, terraformAttribute, complexObjectIsFromSet, complexObjectIndex);`);
            this.code.closeBlock();
        }
        this.code.line();
        this.emitInternalValueGetter(struct);
        this.code.line();
        this.emitInternalValueSetter(struct);
        for (const att of struct.attributes) {
            this.attributesEmitter.emit(att, this.attributesEmitter.needsResetEscape(att, struct.attributes), this.attributesEmitter.needsInputEscape(att, struct.attributes));
        }
        this.code.closeBlock();
        if (!struct.isSingleItem &&
            (struct.nestingMode === "list" || struct.nestingMode === "set")) {
            this.emitComplexListClass(struct);
        }
        else if (struct.nestingMode === "map") {
            this.emitComplexMapClass(struct);
        }
        else if (struct.nestingMode === "maplist") {
            this.emitComplexMapListClasses(struct);
        }
        else if (struct.nestingMode === "mapset") {
            this.emitComplexMapListClasses(struct);
        }
        else if (struct.nestingMode === "listmap") {
            this.emitComplexListMapClasses(struct, false);
        }
        else if (struct.nestingMode === "setmap") {
            this.emitComplexListMapClasses(struct, true);
        }
        else if (struct.nestingMode === "listlist" ||
            struct.nestingMode === "listset" ||
            struct.nestingMode === "setlist" ||
            struct.nestingMode === "setset") {
            this.emitComplexListListClasses(struct);
        }
        // other types of nested collections aren't supported
    }
    emitComplexListClass(struct) {
        this.code.line();
        this.code.openBlock(`export class ${struct.listName} extends cdktf.ComplexList`);
        if (struct.assignable) {
            this.code.line(`public internalValue? : ${struct.name}[] | cdktf.IResolvable`);
        }
        this.code.line();
        const constructorComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        constructorComment.line(`@param terraformResource The parent resource`);
        constructorComment.line(`@param terraformAttribute The attribute on the parent resource this class is referencing`);
        constructorComment.line(`@param wrapsSet whether the list is wrapping a set (will add tolist() to be able to access an item via an index)`);
        constructorComment.end();
        this.code.openBlock(`constructor(protected terraformResource: cdktf.IInterpolatingParent, protected terraformAttribute: string, protected wrapsSet: boolean)`);
        this.code.line(`super(terraformResource, terraformAttribute, wrapsSet)`);
        this.code.closeBlock();
        this.code.line();
        const getterComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        getterComment.line(`@param index the index of the item to return`);
        getterComment.end();
        this.code.openBlock(`public get(index: number): ${struct.outputReferenceName}`);
        this.code.line(`return new ${struct.outputReferenceName}(this.terraformResource, this.terraformAttribute, index, this.wrapsSet);`);
        this.code.closeBlock();
        this.code.closeBlock();
    }
    emitComplexMapClass(struct) {
        this.code.line();
        this.code.openBlock(`export class ${struct.mapName} extends cdktf.ComplexMap`);
        if (struct.assignable) {
            this.code.line(`public internalValue? : { [key: string]: ${struct.name} } | cdktf.IResolvable`);
        }
        this.code.line();
        const constructorComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        constructorComment.line(`@param terraformResource The parent resource`);
        constructorComment.line(`@param terraformAttribute The attribute on the parent resource this class is referencing`);
        constructorComment.end();
        this.code.openBlock(`constructor(protected terraformResource: cdktf.IInterpolatingParent, protected terraformAttribute: string)`);
        this.code.line(`super(terraformResource, terraformAttribute)`);
        this.code.closeBlock();
        this.code.line();
        const getterComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        getterComment.line(`@param key the key of the item to return`);
        getterComment.end();
        this.code.openBlock(`public get(key: string): ${struct.outputReferenceName}`);
        this.code.line(`return new ${struct.outputReferenceName}(this.terraformResource, this.terraformAttribute, key);`);
        this.code.closeBlock();
        this.code.closeBlock();
    }
    emitComplexMapListClasses(struct) {
        this.emitComplexMapListClass(struct);
        this.emitComplexMapClass(struct);
    }
    emitComplexMapListClass(struct) {
        this.code.line();
        this.code.openBlock(`export class ${struct.mapListName} extends cdktf.MapList`);
        if (struct.assignable) {
            this.code.line(`public internalValue? : ${struct.mapName}[] | cdktf.IResolvable`);
        }
        this.code.line();
        const constructorComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        constructorComment.line(`@param terraformResource The parent resource`);
        constructorComment.line(`@param terraformAttribute The attribute on the parent resource this class is referencing`);
        constructorComment.line(`@param wrapsSet whether the list is wrapping a set (will add tolist() to be able to access an item via an index)`);
        constructorComment.end();
        this.code.openBlock(`constructor(protected terraformResource: cdktf.IInterpolatingParent, protected terraformAttribute: string, protected wrapsSet: boolean)`);
        this.code.line(`super(terraformResource, terraformAttribute, wrapsSet)`);
        this.code.closeBlock();
        this.code.line();
        const getterComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        getterComment.line(`@param index the index of the item to return`);
        getterComment.end();
        this.code.openBlock(`public get(index: number): ${struct.mapName}`);
        this.code.line(`return new ${struct.mapName}(this, \`[\${index}]\`);`);
        this.code.closeBlock();
        this.code.closeBlock();
    }
    emitComplexListMapClasses(struct, isSet) {
        this.emitComplexListMapClass(struct, isSet);
        this.emitComplexListClass(struct);
    }
    emitComplexListMapClass(struct, isSet) {
        this.code.line();
        this.code.openBlock(`export class ${struct.listMapName} extends cdktf.ComplexMap`);
        if (struct.assignable) {
            this.code.line(`public internalValue? : { [key: string]: ${struct.name}[] } | cdktf.IResolvable`);
        }
        this.code.line();
        const constructorComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        constructorComment.line(`@param terraformResource The parent resource`);
        constructorComment.line(`@param terraformAttribute The attribute on the parent resource this class is referencing`);
        constructorComment.end();
        this.code.openBlock(`constructor(protected terraformResource: cdktf.IInterpolatingParent, protected terraformAttribute: string)`);
        this.code.line(`super(terraformResource, terraformAttribute)`);
        this.code.closeBlock();
        this.code.line();
        const getterComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        getterComment.line(`@param key the key of the item to return`);
        getterComment.end();
        this.code.openBlock(`public get(key: string): ${struct.listName}`);
        this.code.line(`return new ${struct.listName}(this, \`[\${key}]\`, ${isSet});`);
        this.code.closeBlock();
        this.code.closeBlock();
    }
    emitComplexListListClasses(struct) {
        this.emitComplexListListClass(struct);
        this.emitComplexListClass(struct);
    }
    emitComplexListListClass(struct) {
        this.code.line();
        this.code.openBlock(`export class ${struct.listListName} extends cdktf.MapList`);
        if (struct.assignable) {
            this.code.line(`public internalValue? : ${struct.name}[][] | cdktf.IResolvable`);
        }
        this.code.line();
        const constructorComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        constructorComment.line(`@param terraformResource The parent resource`);
        constructorComment.line(`@param terraformAttribute The attribute on the parent resource this class is referencing`);
        constructorComment.line(`@param wrapsSet whether the list is wrapping a set (will add tolist() to be able to access an item via an index)`);
        constructorComment.end();
        this.code.openBlock(`constructor(protected terraformResource: cdktf.IInterpolatingParent, protected terraformAttribute: string, protected wrapsSet: boolean)`);
        this.code.line(`super(terraformResource, terraformAttribute, wrapsSet)`);
        this.code.closeBlock();
        this.code.line();
        const getterComment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        getterComment.line(`@param index the index of the item to return`);
        getterComment.end();
        this.code.openBlock(`public get(index: number): ${struct.listName}`);
        this.code.line(`return new ${struct.listName}(this, \`[\${index}]\`, ${struct.nestingMode === "setlist" || struct.nestingMode === "setset"});`);
        this.code.closeBlock();
        this.code.closeBlock();
    }
    emitInternalValueGetter(struct) {
        this.code.openBlock(`public get internalValue(): ${struct.name}${!struct.isClass ? " | cdktf.IResolvable" : ""} | undefined`);
        if (!struct.isClass) {
            this.code.openBlock("if (this.resolvableValue)");
            this.code.line("return this.resolvableValue;");
            this.code.closeBlock();
        }
        this.code.line("let hasAnyValues = this.isEmptyObject;");
        this.code.line("const internalValueResult: any = {};");
        for (const att of struct.attributes) {
            if (att.isStored) {
                if (att.getterType._type === "stored_class") {
                    this.code.openBlock(`if (this.${att.storageName}?.internalValue !== undefined)`);
                }
                else {
                    this.code.openBlock(`if (this.${att.storageName} !== undefined)`);
                }
                this.code.line("hasAnyValues = true;");
                if (att.getterType._type === "stored_class") {
                    this.code.line(`internalValueResult.${att.name} = this.${att.storageName}?.internalValue;`);
                }
                else {
                    this.code.line(`internalValueResult.${att.name} = this.${att.storageName};`);
                }
                this.code.closeBlock();
            }
        }
        this.code.line("return hasAnyValues ? internalValueResult : undefined;");
        this.code.closeBlock();
    }
    emitInternalValueSetter(struct) {
        this.code.openBlock(`public set internalValue(value: ${struct.name}${!struct.isClass ? " | cdktf.IResolvable" : ""} | undefined)`);
        this.code.openBlock("if (value === undefined)");
        this.code.line("this.isEmptyObject = false;");
        if (!struct.isClass) {
            this.code.line("this.resolvableValue = undefined;");
        }
        for (const att of struct.attributes) {
            if (att.isStored) {
                if (att.setterType._type === "stored_class") {
                    this.code.line(`this.${att.storageName}.internalValue = undefined;`);
                }
                else if (att.setterType._type !== "none") {
                    this.code.line(`this.${att.storageName} = undefined;`);
                }
            }
        }
        this.code.closeBlock();
        if (!struct.isClass) {
            this.code.openBlock("else if (cdktf.Tokenization.isResolvable(value))");
            this.code.line("this.isEmptyObject = false;");
            this.code.line("this.resolvableValue = value;");
            this.code.closeBlock();
        }
        this.code.openBlock("else");
        this.code.line("this.isEmptyObject = Object.keys(value).length === 0;");
        if (!struct.isClass) {
            this.code.line("this.resolvableValue = undefined;");
        }
        for (const att of struct.attributes) {
            if (att.isStored) {
                if (att.setterType._type === "stored_class") {
                    this.code.line(`this.${att.storageName}.internalValue = value.${att.name};`);
                }
                else if (att.setterType._type !== "none") {
                    this.code.line(`this.${att.storageName} = value.${att.name};`);
                }
            }
        }
        this.code.closeBlock();
        this.code.closeBlock();
    }
    emitToHclTerraformFunction(struct) {
        this.code.line();
        this.code.openBlock(`export function ${(0, util_1.downcaseFirst)(struct.name)}ToHclTerraform(struct?: ${struct.isSingleItem && !struct.isProvider
            ? `${struct.name}OutputReference | `
            : ""}${struct.name}${!struct.isClass ? " | cdktf.IResolvable" : ""}): any`);
        this.code.line(`if (!cdktf.canInspect(struct) || cdktf.Tokenization.isResolvable(struct)) { return struct; }`);
        this.code.openBlock(`if (cdktf.isComplexElement(struct))`);
        this.code.line(`throw new Error("A complex element was used as configuration, this is not supported: https://cdk.tf/complex-object-as-configuration");`);
        this.code.closeBlock();
        this.code.open(`const attrs = {`);
        for (const att of struct.assignableAttributes) {
            this.attributesEmitter.emitToHclTerraform(att, true);
        }
        this.code.close(`};`);
        if (struct.assignableAttributes.length > 0) {
            this.code.line();
            this.code.line(`// remove undefined attributes`);
            this.code.line(`return Object.fromEntries(Object.entries(attrs).filter(([_, value]) => value !== undefined && value.value !== undefined));`);
        }
        else {
            this.code.line(`return attrs;`);
        }
        this.code.closeBlock();
        this.code.line();
    }
    emitToTerraformFunction(struct) {
        this.code.line();
        this.code.openBlock(`export function ${(0, util_1.downcaseFirst)(struct.name)}ToTerraform(struct?: ${struct.isSingleItem && !struct.isProvider
            ? `${struct.name}OutputReference | `
            : ""}${struct.name}${!struct.isClass ? " | cdktf.IResolvable" : ""}): any`);
        this.code.line(`if (!cdktf.canInspect(struct) || cdktf.Tokenization.isResolvable(struct)) { return struct; }`);
        this.code.openBlock(`if (cdktf.isComplexElement(struct))`);
        this.code.line(`throw new Error("A complex element was used as configuration, this is not supported: https://cdk.tf/complex-object-as-configuration");`);
        this.code.closeBlock();
        this.code.openBlock("return");
        for (const att of struct.assignableAttributes) {
            this.attributesEmitter.emitToTerraform(att, true);
        }
        this.code.closeBlock(";");
        this.code.closeBlock();
        this.code.line();
    }
}
exports.StructEmitter = StructEmitter;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RydWN0LWVtaXR0ZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdHJ1Y3QtZW1pdHRlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztBQUdBLHNDQUFnRTtBQUNoRSw2REFBeUQ7QUFDekQsd0NBQThDO0FBQzlDLDJDQUE2QjtBQUM3Qiw2REFBcUU7QUFFckUsOERBQXlEO0FBQ3pELE1BQWEsYUFBYTtJQUd4QixZQUE2QixJQUFlO1FBQWYsU0FBSSxHQUFKLElBQUksQ0FBVztRQUMxQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxzQ0FBaUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQUVNLElBQUksQ0FBQyxRQUF1QjtRQUNqQyxJQUFJLFFBQVEsQ0FBQyxzQkFBc0IsRUFBRSxDQUFDO1lBQ3BDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN2QyxDQUFDO2FBQU0sQ0FBQztZQUNOLElBQUksQ0FBQyxXQUFXLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDN0IsQ0FBQztJQUNILENBQUM7SUFFRCwrR0FBK0c7SUFDL0csMEVBQTBFO0lBQzFFLHFHQUFxRztJQUM3RixnQkFBZ0IsQ0FBQyxHQUFtQjtRQUMxQyxJQUFJLEdBQUcsQ0FBQyxJQUFJLEtBQUssSUFBSSxFQUFFLENBQUM7WUFDdEIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osb0pBQW9KLENBQ3JKLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWiw2SkFBNkosQ0FDOUosQ0FBQztRQUNKLENBQUM7SUFDSCxDQUFDO0lBRU0sYUFBYSxDQUNsQixRQUF1QixFQUN2QixNQUFjLEVBQ2QsSUFBSSxHQUFHLE1BQU0sQ0FBQyxJQUFJO1FBRWxCLElBQUksUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3hCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLG9CQUFvQixJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2xELENBQUM7YUFBTSxDQUFDO1lBQ04sSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsb0JBQW9CLElBQUksR0FBRyxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUNuRSxDQUFDO1FBRUQsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztZQUM5QyxNQUFNLE9BQU8sR0FBRyxJQUFBLHFDQUFnQixFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUM1QyxJQUFJLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztnQkFDcEIsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQzlCLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7Z0JBQ2pCLE9BQU8sQ0FBQyxJQUFJLENBQ1Ysc0NBQXNDLFFBQVEsQ0FBQyxVQUFVLElBQUksR0FBRyxDQUFDLGFBQWEsSUFBSSxRQUFRLENBQUMsU0FBUyxJQUFJLEdBQUcsQ0FBQyxhQUFhLEdBQUcsQ0FDN0gsQ0FBQztnQkFDRixJQUFJLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQzNCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztZQUNoQixDQUFDO2lCQUFNLENBQUM7Z0JBQ04sT0FBTyxDQUFDLElBQUksQ0FDVixzQ0FBc0MsUUFBUSxDQUFDLFVBQVUsSUFBSSxHQUFHLENBQUMsYUFBYSxJQUFJLFFBQVEsQ0FBQyxTQUFTLElBQUksR0FBRyxDQUFDLGFBQWEsR0FBRyxDQUM3SCxDQUFDO2dCQUNGLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDM0IsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ2hCLENBQUM7WUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsQ0FBQyxjQUFjLEdBQUcsQ0FBQyxDQUFDO1FBQ3BELENBQUM7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxDQUFDLE1BQU0sWUFBWSxxQkFBWSxDQUFDLEVBQUUsQ0FBQztZQUN0QyxJQUFJLENBQUMsdUJBQXVCLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDckMsSUFBSSxDQUFDLDBCQUEwQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFDLENBQUM7SUFDSCxDQUFDO0lBRU8sV0FBVyxDQUFDLFFBQXVCO1FBQ3pDLFFBQVEsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7WUFDbEMsSUFBSSxNQUFNLFlBQVkscUJBQVksRUFBRSxDQUFDO2dCQUNuQyxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQztZQUN2QyxDQUFDO2lCQUFNLENBQUM7Z0JBQ04sd0ZBQXdGO2dCQUN4RixJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQztnQkFDckMsaUZBQWlGO2dCQUNqRixJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO29CQUN2QixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUN6QixDQUFDO1lBQ0gsQ0FBQztRQUNILENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLHFCQUFxQixDQUFDLFFBQXVCO1FBQ25ELGdGQUFnRjtRQUNoRixNQUFNLGFBQWEsR0FBMkIsRUFBRSxDQUFDO1FBRWpELGlFQUFpRTtRQUNqRSxNQUFNLDBCQUEwQixHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRTdELE1BQU0sWUFBWSxHQUFlLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDdEMsTUFBTSxXQUFXLEdBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUNsQywwQkFBMEIsQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUM1QyxJQUNFLFdBQVcsQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxHQUFHLE1BQU0sQ0FBQyxXQUFXO2dCQUN4RCwwQ0FBeUIsRUFDekIsQ0FBQztnQkFDRCxZQUFZLENBQUMsWUFBWSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ25ELFdBQVcsQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxXQUFXLENBQUM7WUFDNUQsQ0FBQztpQkFBTSxDQUFDO2dCQUNOLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO2dCQUM1QixXQUFXLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN2QyxDQUFDO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFFSCw2RUFBNkU7UUFDN0UsMENBQTBDO1FBQzFDLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxZQUFZLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7WUFDN0MsTUFBTSxjQUFjLEdBQUcsVUFBVSxDQUFDLEdBQUcsMENBQXlCLEtBQUssQ0FBQztZQUNwRSxNQUFNLE9BQU8sR0FBRyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFFaEMsc0VBQXNFO1lBQ3RFLCtDQUErQztZQUMvQyxPQUFPLENBQUMsT0FBTyxDQUNiLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsY0FBYyxDQUFDLENBQzFELENBQUM7UUFDSixDQUFDO1FBRUQsTUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO1FBQ3ZCLEtBQUssSUFBSSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsR0FBRyxZQUFZLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUM7WUFDN0MsTUFBTSxlQUFlLEdBQTZCLEVBQUUsQ0FBQztZQUNyRCxNQUFNLE9BQU8sR0FBRyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsTUFBTSxjQUFjLEdBQUcsVUFBVSxDQUFDLEdBQUcsMENBQXlCLEtBQUssQ0FBQztZQUNwRSxXQUFXLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBRWpDLHlEQUF5RDtZQUN6RCxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUU7Z0JBQ3pCLE1BQU0sQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7O29CQUNoQyxNQUFNLGNBQWMsR0FBRyxNQUFBLEdBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSwwQ0FBRSxJQUFJLENBQUM7b0JBQzdDLE1BQU0sWUFBWSxHQUFHLGFBQWEsQ0FBQyxjQUFjLGFBQWQsY0FBYyxjQUFkLGNBQWMsR0FBSSxFQUFFLENBQUMsQ0FBQztvQkFFekQsSUFBSSxZQUFZLElBQUksWUFBWSxLQUFLLGNBQWMsRUFBRSxDQUFDO3dCQUNwRCxNQUFNLGFBQWEsR0FBRyxHQUFHLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQzt3QkFDdEMsSUFBSSxDQUFDLGFBQWE7NEJBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxjQUFjLGtCQUFrQixDQUFDLENBQUM7d0JBRXZELE1BQUEsZUFBZSxDQUFDLFlBQVksQ0FBQyxtQ0FDM0IsQ0FBQyxlQUFlLENBQUMsWUFBWSxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7d0JBRXZDLE1BQU0sYUFBYSxHQUFHLEdBQUcsQ0FBQyxrQkFBa0IsQ0FDMUMsTUFBTSxZQUFZLHFCQUFZLENBQy9CLENBQUM7d0JBQ0YsSUFBSSxhQUFhLEVBQUUsQ0FBQzs0QkFDbEIsZUFBZSxDQUFDLFlBQVksQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLGFBQWEsQ0FBQyxDQUFDO3dCQUN2RCxDQUFDO29CQUNILENBQUM7Z0JBQ0gsQ0FBQyxDQUFDLENBQUM7WUFDTCxDQUFDLENBQUMsQ0FBQztZQUVILE1BQU0sa0JBQWtCLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FDbEMsUUFBUSxDQUFDLGlCQUFpQixFQUMxQixjQUFjLENBQ2YsQ0FBQztZQUVGLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLENBQUM7WUFDdkMscURBQXFEO1lBQ3JELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7WUFDbEQsTUFBTSxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUMsRUFBRSxFQUFFO2dCQUNsRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWixZQUFZLENBQUMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FDcEMsS0FBSyxDQUNOLGNBQWMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUUsS0FBSyxDQUFDLEdBQUcsQ0FDckQsQ0FBQztZQUNKLENBQUMsQ0FBQyxDQUFDO1lBRUgsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFO2dCQUN6QixJQUFJLE1BQU0sWUFBWSxxQkFBWSxFQUFFLENBQUM7b0JBQ25DLElBQUksQ0FBQyxhQUFhLENBQUMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUN2QyxDQUFDO3FCQUFNLENBQUM7b0JBQ04sd0ZBQXdGO29CQUN4RixJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQztvQkFDckMsaUZBQWlGO29CQUNqRixJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO3dCQUN2QixJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO29CQUN6QixDQUFDO2dCQUNILENBQUM7WUFDSCxDQUFDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDMUMsQ0FBQztRQUVELDZFQUE2RTtRQUM3RSxNQUFNLGFBQWEsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUV4RSxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNsQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsVUFBVSxFQUFFLEVBQUU7WUFDakMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMxRSxDQUFDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFTyxTQUFTLENBQUMsTUFBYztRQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsZ0JBQWdCLE1BQU0sQ0FBQyxtQkFBbUIsOEJBQThCLENBQ3pFLENBQUM7UUFFRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUNqRSxDQUFDO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUVqQixNQUFNLE9BQU8sR0FBRyxJQUFBLHFDQUFnQixFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM1QyxPQUFPLENBQUMsSUFBSSxDQUFDLDhDQUE4QyxDQUFDLENBQUM7UUFDN0QsT0FBTyxDQUFDLElBQUksQ0FDViwwRkFBMEYsQ0FDM0YsQ0FBQztRQUNGLElBQUksTUFBTSxDQUFDLFlBQVksRUFBRSxDQUFDO1lBQ3hCLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztZQUNkLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQiwrRkFBK0YsQ0FDaEcsQ0FBQztZQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHlEQUF5RCxDQUFDLENBQUM7WUFDMUUsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN6QixDQUFDO2FBQU0sSUFDTCxNQUFNLENBQUMsV0FBVyxLQUFLLFFBQVE7WUFDL0IsTUFBTSxDQUFDLFdBQVcsS0FBSyxRQUFRLEVBQy9CLENBQUM7WUFDRCxPQUFPLENBQUMsR0FBRyxFQUFFLENBQUM7WUFDZCxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsK0ZBQStGLENBQ2hHLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO1lBQ3ZFLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDekIsQ0FBQzthQUFNLElBQUksTUFBTSxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQztZQUNoRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWiwyREFBMkQsQ0FDNUQsQ0FBQztZQUNGLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztZQUNkLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQix5SEFBeUgsQ0FDMUgsQ0FBQztZQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLHdFQUF3RSxDQUN6RSxDQUFDO1lBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN6QixDQUFDO2FBQU0sQ0FBQztZQUNOLE9BQU8sQ0FBQyxJQUFJLENBQ1YsOERBQThELENBQy9ELENBQUM7WUFDRixPQUFPLENBQUMsSUFBSSxDQUNWLGdJQUFnSSxDQUNqSSxDQUFDO1lBQ0YsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1lBQ2QsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLDRKQUE0SixDQUM3SixDQUFDO1lBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osMkZBQTJGLENBQzVGLENBQUM7WUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3pCLENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNyQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUVyQyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNwQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUN6QixHQUFHLEVBQ0gsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxNQUFNLENBQUMsVUFBVSxDQUFDLEVBQy9ELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUNoRSxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFDRSxDQUFDLE1BQU0sQ0FBQyxZQUFZO1lBQ3BCLENBQUMsTUFBTSxDQUFDLFdBQVcsS0FBSyxNQUFNLElBQUksTUFBTSxDQUFDLFdBQVcsS0FBSyxLQUFLLENBQUMsRUFDL0QsQ0FBQztZQUNELElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNwQyxDQUFDO2FBQU0sSUFBSSxNQUFNLENBQUMsV0FBVyxLQUFLLEtBQUssRUFBRSxDQUFDO1lBQ3hDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNuQyxDQUFDO2FBQU0sSUFBSSxNQUFNLENBQUMsV0FBVyxLQUFLLFNBQVMsRUFBRSxDQUFDO1lBQzVDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxDQUFDO2FBQU0sSUFBSSxNQUFNLENBQUMsV0FBVyxLQUFLLFFBQVEsRUFBRSxDQUFDO1lBQzNDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN6QyxDQUFDO2FBQU0sSUFBSSxNQUFNLENBQUMsV0FBVyxLQUFLLFNBQVMsRUFBRSxDQUFDO1lBQzVDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDaEQsQ0FBQzthQUFNLElBQUksTUFBTSxDQUFDLFdBQVcsS0FBSyxRQUFRLEVBQUUsQ0FBQztZQUMzQyxJQUFJLENBQUMseUJBQXlCLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQy9DLENBQUM7YUFBTSxJQUNMLE1BQU0sQ0FBQyxXQUFXLEtBQUssVUFBVTtZQUNqQyxNQUFNLENBQUMsV0FBVyxLQUFLLFNBQVM7WUFDaEMsTUFBTSxDQUFDLFdBQVcsS0FBSyxTQUFTO1lBQ2hDLE1BQU0sQ0FBQyxXQUFXLEtBQUssUUFBUSxFQUMvQixDQUFDO1lBQ0QsSUFBSSxDQUFDLDBCQUEwQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzFDLENBQUM7UUFDRCxxREFBcUQ7SUFDdkQsQ0FBQztJQUVPLG9CQUFvQixDQUFDLE1BQWM7UUFDekMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsZ0JBQWdCLE1BQU0sQ0FBQyxRQUFRLDRCQUE0QixDQUM1RCxDQUFDO1FBRUYsSUFBSSxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDdEIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osMkJBQTJCLE1BQU0sQ0FBQyxJQUFJLHdCQUF3QixDQUMvRCxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakIsTUFBTSxrQkFBa0IsR0FBRyxJQUFBLHFDQUFnQixFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN4RSxrQkFBa0IsQ0FBQyxJQUFJLENBQ3JCLDBGQUEwRixDQUMzRixDQUFDO1FBQ0Ysa0JBQWtCLENBQUMsSUFBSSxDQUNyQixrSEFBa0gsQ0FDbkgsQ0FBQztRQUNGLGtCQUFrQixDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQix5SUFBeUksQ0FDMUksQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHdEQUF3RCxDQUFDLENBQUM7UUFDekUsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUV2QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLE1BQU0sYUFBYSxHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xELGFBQWEsQ0FBQyxJQUFJLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUNuRSxhQUFhLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLDhCQUE4QixNQUFNLENBQUMsbUJBQW1CLEVBQUUsQ0FDM0QsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLGNBQWMsTUFBTSxDQUFDLG1CQUFtQiwwRUFBMEUsQ0FDbkgsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUN6QixDQUFDO0lBRU8sbUJBQW1CLENBQUMsTUFBYztRQUN4QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQixnQkFBZ0IsTUFBTSxDQUFDLE9BQU8sMkJBQTJCLENBQzFELENBQUM7UUFFRixJQUFJLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWiw0Q0FBNEMsTUFBTSxDQUFDLElBQUksd0JBQXdCLENBQ2hGLENBQUM7UUFDSixDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixNQUFNLGtCQUFrQixHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZELGtCQUFrQixDQUFDLElBQUksQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1FBQ3hFLGtCQUFrQixDQUFDLElBQUksQ0FDckIsMEZBQTBGLENBQzNGLENBQUM7UUFDRixrQkFBa0IsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUN6QixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsNEdBQTRHLENBQzdHLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1FBQy9ELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixNQUFNLGFBQWEsR0FBRyxJQUFBLHFDQUFnQixFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNsRCxhQUFhLENBQUMsSUFBSSxDQUFDLDBDQUEwQyxDQUFDLENBQUM7UUFDL0QsYUFBYSxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3BCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQiw0QkFBNEIsTUFBTSxDQUFDLG1CQUFtQixFQUFFLENBQ3pELENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWixjQUFjLE1BQU0sQ0FBQyxtQkFBbUIseURBQXlELENBQ2xHLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDekIsQ0FBQztJQUVPLHlCQUF5QixDQUFDLE1BQWM7UUFDOUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBRU8sdUJBQXVCLENBQUMsTUFBYztRQUM1QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQixnQkFBZ0IsTUFBTSxDQUFDLFdBQVcsd0JBQXdCLENBQzNELENBQUM7UUFFRixJQUFJLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWiwyQkFBMkIsTUFBTSxDQUFDLE9BQU8sd0JBQXdCLENBQ2xFLENBQUM7UUFDSixDQUFDO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixNQUFNLGtCQUFrQixHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3ZELGtCQUFrQixDQUFDLElBQUksQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1FBQ3hFLGtCQUFrQixDQUFDLElBQUksQ0FDckIsMEZBQTBGLENBQzNGLENBQUM7UUFDRixrQkFBa0IsQ0FBQyxJQUFJLENBQ3JCLGtIQUFrSCxDQUNuSCxDQUFDO1FBQ0Ysa0JBQWtCLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLHlJQUF5SSxDQUMxSSxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsd0RBQXdELENBQUMsQ0FBQztRQUN6RSxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakIsTUFBTSxhQUFhLEdBQUcsSUFBQSxxQ0FBZ0IsRUFBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDbEQsYUFBYSxDQUFDLElBQUksQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1FBQ25FLGFBQWEsQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNwQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyw4QkFBOEIsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDcEUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxNQUFNLENBQUMsT0FBTywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUN6QixDQUFDO0lBRU8seUJBQXlCLENBQUMsTUFBYyxFQUFFLEtBQWM7UUFDOUQsSUFBSSxDQUFDLHVCQUF1QixDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVPLHVCQUF1QixDQUFDLE1BQWMsRUFBRSxLQUFjO1FBQzVELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLGdCQUFnQixNQUFNLENBQUMsV0FBVywyQkFBMkIsQ0FDOUQsQ0FBQztRQUVGLElBQUksTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLDRDQUE0QyxNQUFNLENBQUMsSUFBSSwwQkFBMEIsQ0FDbEYsQ0FBQztRQUNKLENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLE1BQU0sa0JBQWtCLEdBQUcsSUFBQSxxQ0FBZ0IsRUFBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDdkQsa0JBQWtCLENBQUMsSUFBSSxDQUFDLDhDQUE4QyxDQUFDLENBQUM7UUFDeEUsa0JBQWtCLENBQUMsSUFBSSxDQUNyQiwwRkFBMEYsQ0FDM0YsQ0FBQztRQUNGLGtCQUFrQixDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQiw0R0FBNEcsQ0FDN0csQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLDhDQUE4QyxDQUFDLENBQUM7UUFDL0QsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUV2QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLE1BQU0sYUFBYSxHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xELGFBQWEsQ0FBQyxJQUFJLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUMvRCxhQUFhLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsNEJBQTRCLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLGNBQWMsTUFBTSxDQUFDLFFBQVEseUJBQXlCLEtBQUssSUFBSSxDQUNoRSxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUV2QixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ3pCLENBQUM7SUFFTywwQkFBMEIsQ0FBQyxNQUFjO1FBQy9DLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsb0JBQW9CLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVPLHdCQUF3QixDQUFDLE1BQWM7UUFDN0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsZ0JBQWdCLE1BQU0sQ0FBQyxZQUFZLHdCQUF3QixDQUM1RCxDQUFDO1FBRUYsSUFBSSxNQUFNLENBQUMsVUFBVSxFQUFFLENBQUM7WUFDdEIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osMkJBQTJCLE1BQU0sQ0FBQyxJQUFJLDBCQUEwQixDQUNqRSxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakIsTUFBTSxrQkFBa0IsR0FBRyxJQUFBLHFDQUFnQixFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2RCxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUN4RSxrQkFBa0IsQ0FBQyxJQUFJLENBQ3JCLDBGQUEwRixDQUMzRixDQUFDO1FBQ0Ysa0JBQWtCLENBQUMsSUFBSSxDQUNyQixrSEFBa0gsQ0FDbkgsQ0FBQztRQUNGLGtCQUFrQixDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQix5SUFBeUksQ0FDMUksQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHdEQUF3RCxDQUFDLENBQUM7UUFDekUsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUV2QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLE1BQU0sYUFBYSxHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2xELGFBQWEsQ0FBQyxJQUFJLENBQUMsOENBQThDLENBQUMsQ0FBQztRQUNuRSxhQUFhLENBQUMsR0FBRyxFQUFFLENBQUM7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsOEJBQThCLE1BQU0sQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLGNBQWMsTUFBTSxDQUFDLFFBQVEsMkJBQzNCLE1BQU0sQ0FBQyxXQUFXLEtBQUssU0FBUyxJQUFJLE1BQU0sQ0FBQyxXQUFXLEtBQUssUUFDN0QsSUFBSSxDQUNMLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRXZCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDekIsQ0FBQztJQUVPLHVCQUF1QixDQUFDLE1BQWM7UUFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLCtCQUErQixNQUFNLENBQUMsSUFBSSxHQUN4QyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxFQUM3QyxjQUFjLENBQ2YsQ0FBQztRQUVGLElBQUksQ0FBQyxNQUFNLENBQUMsT0FBTyxFQUFFLENBQUM7WUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsMkJBQTJCLENBQUMsQ0FBQztZQUNqRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1lBQy9DLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDekIsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHdDQUF3QyxDQUFDLENBQUM7UUFDekQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUN2RCxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNwQyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDakIsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssS0FBSyxjQUFjLEVBQUUsQ0FBQztvQkFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLFlBQVksR0FBRyxDQUFDLFdBQVcsZ0NBQWdDLENBQzVELENBQUM7Z0JBQ0osQ0FBQztxQkFBTSxDQUFDO29CQUNOLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFlBQVksR0FBRyxDQUFDLFdBQVcsaUJBQWlCLENBQUMsQ0FBQztnQkFDcEUsQ0FBQztnQkFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO2dCQUN2QyxJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxLQUFLLGNBQWMsRUFBRSxDQUFDO29CQUM1QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWix1QkFBdUIsR0FBRyxDQUFDLElBQUksV0FBVyxHQUFHLENBQUMsV0FBVyxrQkFBa0IsQ0FDNUUsQ0FBQztnQkFDSixDQUFDO3FCQUFNLENBQUM7b0JBQ04sSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osdUJBQXVCLEdBQUcsQ0FBQyxJQUFJLFdBQVcsR0FBRyxDQUFDLFdBQVcsR0FBRyxDQUM3RCxDQUFDO2dCQUNKLENBQUM7Z0JBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUN6QixDQUFDO1FBQ0gsQ0FBQztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHdEQUF3RCxDQUFDLENBQUM7UUFDekUsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztJQUN6QixDQUFDO0lBRU8sdUJBQXVCLENBQUMsTUFBYztRQUM1QyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsbUNBQW1DLE1BQU0sQ0FBQyxJQUFJLEdBQzVDLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsc0JBQXNCLENBQUMsQ0FBQyxDQUFDLEVBQzdDLGVBQWUsQ0FDaEIsQ0FBQztRQUVGLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQ3BCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDdEQsQ0FBQztRQUNELEtBQUssTUFBTSxHQUFHLElBQUksTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3BDLElBQUksR0FBRyxDQUFDLFFBQVEsRUFBRSxDQUFDO2dCQUNqQixJQUFJLEdBQUcsQ0FBQyxVQUFVLENBQUMsS0FBSyxLQUFLLGNBQWMsRUFBRSxDQUFDO29CQUM1QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEdBQUcsQ0FBQyxXQUFXLDZCQUE2QixDQUFDLENBQUM7Z0JBQ3ZFLENBQUM7cUJBQU0sSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssS0FBSyxNQUFNLEVBQUUsQ0FBQztvQkFDM0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxHQUFHLENBQUMsV0FBVyxlQUFlLENBQUMsQ0FBQztnQkFDekQsQ0FBQztZQUNILENBQUM7UUFDSCxDQUFDO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN2QixJQUFJLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxDQUFDO1lBQ3BCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7WUFDeEUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsNkJBQTZCLENBQUMsQ0FBQztZQUM5QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1lBQ2hELElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFDekIsQ0FBQztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzVCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLHVEQUF1RCxDQUFDLENBQUM7UUFDeEUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQztZQUNwQixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDO1FBQ3RELENBQUM7UUFDRCxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxVQUFVLEVBQUUsQ0FBQztZQUNwQyxJQUFJLEdBQUcsQ0FBQyxRQUFRLEVBQUUsQ0FBQztnQkFDakIsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssS0FBSyxjQUFjLEVBQUUsQ0FBQztvQkFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osUUFBUSxHQUFHLENBQUMsV0FBVywwQkFBMEIsR0FBRyxDQUFDLElBQUksR0FBRyxDQUM3RCxDQUFDO2dCQUNKLENBQUM7cUJBQU0sSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEtBQUssS0FBSyxNQUFNLEVBQUUsQ0FBQztvQkFDM0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxHQUFHLENBQUMsV0FBVyxZQUFZLEdBQUcsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO2dCQUNqRSxDQUFDO1lBQ0gsQ0FBQztRQUNILENBQUM7UUFDRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDekIsQ0FBQztJQUVPLDBCQUEwQixDQUFDLE1BQWM7UUFDL0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsbUJBQW1CLElBQUEsb0JBQWEsRUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDLDJCQUMzQyxNQUFNLENBQUMsWUFBWSxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVU7WUFDdkMsQ0FBQyxDQUFDLEdBQUcsTUFBTSxDQUFDLElBQUksb0JBQW9CO1lBQ3BDLENBQUMsQ0FBQyxFQUNOLEdBQUcsTUFBTSxDQUFDLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FDdkUsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLDhGQUE4RixDQUMvRixDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMzRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWix3SUFBd0ksQ0FDekksQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUVsQyxLQUFLLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1lBQzlDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDdkQsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXRCLElBQUksTUFBTSxDQUFDLG9CQUFvQixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUMzQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7WUFDakQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osNEhBQTRILENBQzdILENBQUM7UUFDSixDQUFDO2FBQU0sQ0FBQztZQUNOLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ2xDLENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDbkIsQ0FBQztJQUVPLHVCQUF1QixDQUFDLE1BQWM7UUFDNUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FDakIsbUJBQW1CLElBQUEsb0JBQWEsRUFBQyxNQUFNLENBQUMsSUFBSSxDQUFDLHdCQUMzQyxNQUFNLENBQUMsWUFBWSxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVU7WUFDdkMsQ0FBQyxDQUFDLEdBQUcsTUFBTSxDQUFDLElBQUksb0JBQW9CO1lBQ3BDLENBQUMsQ0FBQyxFQUNOLEdBQUcsTUFBTSxDQUFDLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FDdkUsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLDhGQUE4RixDQUMvRixDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMzRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWix3SUFBd0ksQ0FDekksQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7UUFFdkIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDOUIsS0FBSyxNQUFNLEdBQUcsSUFBSSxNQUFNLENBQUMsb0JBQW9CLEVBQUUsQ0FBQztZQUM5QyxJQUFJLENBQUMsaUJBQWlCLENBQUMsZUFBZSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUNwRCxDQUFDO1FBQ0QsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDMUIsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUN2QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ25CLENBQUM7Q0FDRjtBQTNwQkQsc0NBMnBCQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmNcbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNUEwtMi4wXG5pbXBvcnQgeyBDb2RlTWFrZXIgfSBmcm9tIFwiY29kZW1ha2VyXCI7XG5pbXBvcnQgeyBSZXNvdXJjZU1vZGVsLCBTdHJ1Y3QsIENvbmZpZ1N0cnVjdCB9IGZyb20gXCIuLi9tb2RlbHNcIjtcbmltcG9ydCB7IEF0dHJpYnV0ZXNFbWl0dGVyIH0gZnJvbSBcIi4vYXR0cmlidXRlcy1lbWl0dGVyXCI7XG5pbXBvcnQgeyBkb3duY2FzZUZpcnN0IH0gZnJvbSBcIi4uLy4uLy4uL3V0aWxcIjtcbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IFNUUlVDVF9TSEFSRElOR19USFJFU0hPTEQgfSBmcm9tIFwiLi4vbW9kZWxzL3Jlc291cmNlLW1vZGVsXCI7XG5pbXBvcnQgeyBBdHRyaWJ1dGVNb2RlbCB9IGZyb20gXCIuLi9tb2RlbHMvYXR0cmlidXRlLW1vZGVsXCI7XG5pbXBvcnQgeyBzYW5pdGl6ZWRDb21tZW50IH0gZnJvbSBcIi4uL3Nhbml0aXplZC1jb21tZW50c1wiO1xuZXhwb3J0IGNsYXNzIFN0cnVjdEVtaXR0ZXIge1xuICBhdHRyaWJ1dGVzRW1pdHRlcjogQXR0cmlidXRlc0VtaXR0ZXI7XG5cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBjb2RlOiBDb2RlTWFrZXIpIHtcbiAgICB0aGlzLmF0dHJpYnV0ZXNFbWl0dGVyID0gbmV3IEF0dHJpYnV0ZXNFbWl0dGVyKHRoaXMuY29kZSk7XG4gIH1cblxuICBwdWJsaWMgZW1pdChyZXNvdXJjZTogUmVzb3VyY2VNb2RlbCkge1xuICAgIGlmIChyZXNvdXJjZS5zdHJ1Y3RzUmVxdWlyZVNoYXJkaW5nKSB7XG4gICAgICB0aGlzLmVtaXROYW1lc3BhY2VkU3RydWN0cyhyZXNvdXJjZSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuZW1pdFN0cnVjdHMocmVzb3VyY2UpO1xuICAgIH1cbiAgfVxuXG4gIC8vIER1ZSB0byBodHRwczovL2dpdGh1Yi5jb20vaGFzaGljb3JwL3RlcnJhZm9ybS1wbHVnaW4tc2RrL2NvbW1pdC8yMzg3ZWI4NWUzMmMwNjRiNGE2MjcxOGM5ZjVjODBiZjAwZGM3ZmI5IGFsbFxuICAvLyByZXNvdXJjZXMgZnJvbSBwcm92aWRlcnMgdXNpbmcgdGhlIG9sZCBTREsgaGF2ZSB0aGUgaWQgZmllbGQgYnkgZGVmYXVsdFxuICAvLyBXZSBoYXZlIG5vIHdheSB0byBkaXN0aW5ndWlzaCB0aGVtIHRocm91Z2ggdGhlIHByb3ZpZGVyIHNjaGVtYSwgc28gYSB3b3JkIG9mIHdhcm5pbmcgZm9yIG91ciB1c2Vyc1xuICBwcml2YXRlIHdhcm5BYm91dElkRmllbGQoYXR0OiBBdHRyaWJ1dGVNb2RlbCkge1xuICAgIGlmIChhdHQubmFtZSA9PT0gXCJpZFwiKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShgKmApO1xuICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgIGAqIFBsZWFzZSBiZSBhd2FyZSB0aGF0IHRoZSBpZCBmaWVsZCBpcyBhdXRvbWF0aWNhbGx5IGFkZGVkIHRvIGFsbCByZXNvdXJjZXMgaW4gVGVycmFmb3JtIHByb3ZpZGVycyB1c2luZyBhIFRlcnJhZm9ybSBwcm92aWRlciBTREsgdmVyc2lvbiBiZWxvdyAyLmAsXG4gICAgICApO1xuICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgIGAqIElmIHlvdSBleHBlcmllbmNlIHByb2JsZW1zIHNldHRpbmcgdGhpcyB2YWx1ZSBpdCBtaWdodCBub3QgYmUgc2V0dGFibGUuIFBsZWFzZSB0YWtlIGEgbG9vayBhdCB0aGUgcHJvdmlkZXIgZG9jdW1lbnRhdGlvbiB0byBlbnN1cmUgaXQgc2hvdWxkIGJlIHNldHRhYmxlLmAsXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBlbWl0SW50ZXJmYWNlKFxuICAgIHJlc291cmNlOiBSZXNvdXJjZU1vZGVsLFxuICAgIHN0cnVjdDogU3RydWN0LFxuICAgIG5hbWUgPSBzdHJ1Y3QubmFtZSxcbiAgKSB7XG4gICAgaWYgKHJlc291cmNlLmlzUHJvdmlkZXIpIHtcbiAgICAgIHRoaXMuY29kZS5vcGVuQmxvY2soYGV4cG9ydCBpbnRlcmZhY2UgJHtuYW1lfWApO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLmNvZGUub3BlbkJsb2NrKGBleHBvcnQgaW50ZXJmYWNlICR7bmFtZX0ke3N0cnVjdC5leHRlbmRzfWApO1xuICAgIH1cblxuICAgIGZvciAoY29uc3QgYXR0IG9mIHN0cnVjdC5hc3NpZ25hYmxlQXR0cmlidXRlcykge1xuICAgICAgY29uc3QgY29tbWVudCA9IHNhbml0aXplZENvbW1lbnQodGhpcy5jb2RlKTtcbiAgICAgIGlmIChhdHQuZGVzY3JpcHRpb24pIHtcbiAgICAgICAgY29tbWVudC5saW5lKGF0dC5kZXNjcmlwdGlvbik7XG4gICAgICAgIGNvbW1lbnQubGluZShgYCk7XG4gICAgICAgIGNvbW1lbnQubGluZShcbiAgICAgICAgICBgRG9jcyBhdCBUZXJyYWZvcm0gUmVnaXN0cnk6IHtAbGluayAke3Jlc291cmNlLmxpbmtUb0RvY3N9IyR7YXR0LnRlcnJhZm9ybU5hbWV9ICR7cmVzb3VyY2UuY2xhc3NOYW1lfSMke2F0dC50ZXJyYWZvcm1OYW1lfX1gLFxuICAgICAgICApO1xuICAgICAgICB0aGlzLndhcm5BYm91dElkRmllbGQoYXR0KTtcbiAgICAgICAgY29tbWVudC5lbmQoKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGNvbW1lbnQubGluZShcbiAgICAgICAgICBgRG9jcyBhdCBUZXJyYWZvcm0gUmVnaXN0cnk6IHtAbGluayAke3Jlc291cmNlLmxpbmtUb0RvY3N9IyR7YXR0LnRlcnJhZm9ybU5hbWV9ICR7cmVzb3VyY2UuY2xhc3NOYW1lfSMke2F0dC50ZXJyYWZvcm1OYW1lfX1gLFxuICAgICAgICApO1xuICAgICAgICB0aGlzLndhcm5BYm91dElkRmllbGQoYXR0KTtcbiAgICAgICAgY29tbWVudC5lbmQoKTtcbiAgICAgIH1cblxuICAgICAgdGhpcy5jb2RlLmxpbmUoYHJlYWRvbmx5ICR7YXR0LnR5cGVEZWZpbml0aW9ufTtgKTtcbiAgICB9XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIGlmICghKHN0cnVjdCBpbnN0YW5jZW9mIENvbmZpZ1N0cnVjdCkpIHtcbiAgICAgIHRoaXMuZW1pdFRvVGVycmFmb3JtRnVuY3Rpb24oc3RydWN0KTtcbiAgICAgIHRoaXMuZW1pdFRvSGNsVGVycmFmb3JtRnVuY3Rpb24oc3RydWN0KTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGVtaXRTdHJ1Y3RzKHJlc291cmNlOiBSZXNvdXJjZU1vZGVsKSB7XG4gICAgcmVzb3VyY2Uuc3RydWN0cy5mb3JFYWNoKChzdHJ1Y3QpID0+IHtcbiAgICAgIGlmIChzdHJ1Y3QgaW5zdGFuY2VvZiBDb25maWdTdHJ1Y3QpIHtcbiAgICAgICAgdGhpcy5lbWl0SW50ZXJmYWNlKHJlc291cmNlLCBzdHJ1Y3QpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gV2UgdXNlIHRoZSBpbnRlcmZhY2UgaGVyZSBmb3IgdGhlIGNvbmZpZ3VyYXRpb24gLyBpbnB1dHMgb2YgYSByZXNvdXJjZSAvIG5lc3RlZCBibG9ja1xuICAgICAgICB0aGlzLmVtaXRJbnRlcmZhY2UocmVzb3VyY2UsIHN0cnVjdCk7XG4gICAgICAgIC8vIEFuZCB3ZSB1c2UgdGhlIGNsYXNzIGZvciB0aGUgYXR0cmlidXRlcyAvIG91dHB1dHMgb2YgYSByZXNvdXJjZSAvIG5lc3RlZCBibG9ja1xuICAgICAgICBpZiAoIXN0cnVjdC5pc1Byb3ZpZGVyKSB7XG4gICAgICAgICAgdGhpcy5lbWl0Q2xhc3Moc3RydWN0KTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0TmFtZXNwYWNlZFN0cnVjdHMocmVzb3VyY2U6IFJlc291cmNlTW9kZWwpIHtcbiAgICAvLyBpdGVyYXRlIG92ZXIgYWxsIHN0cnVjdHMgaW4gYmF0Y2hlcyBvZiA0MDAgdG8gYXZvaWQgdG9vIG1hbnkgZXhwb3J0cyAoPiAxMjAwKVxuICAgIGNvbnN0IHN0cnVjdEltcG9ydHM6IFJlY29yZDxzdHJpbmcsIHN0cmluZz4gPSB7fTtcblxuICAgIC8vIGRyb3AgY29uZmlnU3RydWN0IGZyb20gcmVzb3VyY2Uuc3RydWN0cyB0byBhdm9pZCBkb3VibGUgaW1wb3J0XG4gICAgY29uc3Qgc3RydWN0c1dpdGhvdXRDb25maWdTdHJ1Y3QgPSByZXNvdXJjZS5zdHJ1Y3RzLnNsaWNlKDEpO1xuXG4gICAgY29uc3Qgc3RydWN0U3BsaXRzOiBTdHJ1Y3RbXVtdID0gW1tdXTtcbiAgICBjb25zdCBzcGxpdENvdW50czogbnVtYmVyW10gPSBbMF07XG4gICAgc3RydWN0c1dpdGhvdXRDb25maWdTdHJ1Y3QuZm9yRWFjaCgoc3RydWN0KSA9PiB7XG4gICAgICBpZiAoXG4gICAgICAgIHNwbGl0Q291bnRzW3NwbGl0Q291bnRzLmxlbmd0aCAtIDFdICsgc3RydWN0LmV4cG9ydENvdW50IDw9XG4gICAgICAgIFNUUlVDVF9TSEFSRElOR19USFJFU0hPTERcbiAgICAgICkge1xuICAgICAgICBzdHJ1Y3RTcGxpdHNbc3RydWN0U3BsaXRzLmxlbmd0aCAtIDFdLnB1c2goc3RydWN0KTtcbiAgICAgICAgc3BsaXRDb3VudHNbc3BsaXRDb3VudHMubGVuZ3RoIC0gMV0gKz0gc3RydWN0LmV4cG9ydENvdW50O1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgc3RydWN0U3BsaXRzLnB1c2goW3N0cnVjdF0pO1xuICAgICAgICBzcGxpdENvdW50cy5wdXNoKHN0cnVjdC5leHBvcnRDb3VudCk7XG4gICAgICB9XG4gICAgfSk7XG5cbiAgICAvLyBmaWxsIHRoZSBzdHJ1Y3QgaW1wb3J0cyBtYXBwaW5nIGJlZm9yZSBjb2RlIGdlbmVyYXRpb24gc28gdGhhdCBpbXBvcnRzIGNhblxuICAgIC8vIHBvaW50IHRvIG5vdCB5ZXQgZ2VuZXJhdGVkIHN0cnVjdCBmaWxlc1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgc3RydWN0U3BsaXRzLmxlbmd0aDsgaSsrKSB7XG4gICAgICBjb25zdCBzdHJ1Y3RGaWxlbmFtZSA9IGBzdHJ1Y3RzJHtpICogU1RSVUNUX1NIQVJESU5HX1RIUkVTSE9MRH0udHNgO1xuICAgICAgY29uc3Qgc3RydWN0cyA9IHN0cnVjdFNwbGl0c1tpXTtcblxuICAgICAgLy8gYXNzb2NpYXRlIGN1cnJlbnQgc3RydWN0cyBiYXRjaCB3aXRoIHRoZSBmaWxlIGl0IHdpbGwgYmUgd3JpdHRlbiB0b1xuICAgICAgLy8gdG8gZmluZCBpdCBpbiBzdWJzZXF1ZW50IGZpbGVzIGZvciBpbXBvcnRpbmdcbiAgICAgIHN0cnVjdHMuZm9yRWFjaChcbiAgICAgICAgKHN0cnVjdCkgPT4gKHN0cnVjdEltcG9ydHNbc3RydWN0Lm5hbWVdID0gc3RydWN0RmlsZW5hbWUpLFxuICAgICAgKTtcbiAgICB9XG5cbiAgICBjb25zdCBzdHJ1Y3RQYXRocyA9IFtdO1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgc3RydWN0U3BsaXRzLmxlbmd0aDsgaSsrKSB7XG4gICAgICBjb25zdCBzdHJ1Y3RzVG9JbXBvcnQ6IFJlY29yZDxzdHJpbmcsIHN0cmluZ1tdPiA9IHt9O1xuICAgICAgY29uc3Qgc3RydWN0cyA9IHN0cnVjdFNwbGl0c1tpXTtcbiAgICAgIGNvbnN0IHN0cnVjdEZpbGVuYW1lID0gYHN0cnVjdHMke2kgKiBTVFJVQ1RfU0hBUkRJTkdfVEhSRVNIT0xEfS50c2A7XG4gICAgICBzdHJ1Y3RQYXRocy5wdXNoKHN0cnVjdEZpbGVuYW1lKTtcblxuICAgICAgLy8gZmluZCBhbGwgc3RydWN0cyB0aGF0IG5lZWQgdG8gYmUgaW1wb3J0ZWQgaW4gdGhpcyBmaWxlXG4gICAgICBzdHJ1Y3RzLmZvckVhY2goKHN0cnVjdCkgPT4ge1xuICAgICAgICBzdHJ1Y3QuYXR0cmlidXRlcy5mb3JFYWNoKChhdHQpID0+IHtcbiAgICAgICAgICBjb25zdCBzdHJ1Y3RUeXBlTmFtZSA9IGF0dC50eXBlLnN0cnVjdD8ubmFtZTtcbiAgICAgICAgICBjb25zdCBmaWxlVG9JbXBvcnQgPSBzdHJ1Y3RJbXBvcnRzW3N0cnVjdFR5cGVOYW1lID8/IFwiXCJdO1xuXG4gICAgICAgICAgaWYgKGZpbGVUb0ltcG9ydCAmJiBmaWxlVG9JbXBvcnQgIT09IHN0cnVjdEZpbGVuYW1lKSB7XG4gICAgICAgICAgICBjb25zdCBhdHRUeXBlU3RydWN0ID0gYXR0LnR5cGUuc3RydWN0O1xuICAgICAgICAgICAgaWYgKCFhdHRUeXBlU3RydWN0KVxuICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYCR7c3RydWN0VHlwZU5hbWV9IGlzIG5vdCBhIHN0cnVjdGApO1xuXG4gICAgICAgICAgICBzdHJ1Y3RzVG9JbXBvcnRbZmlsZVRvSW1wb3J0XSA/P1xuICAgICAgICAgICAgICAoc3RydWN0c1RvSW1wb3J0W2ZpbGVUb0ltcG9ydF0gPSBbXSk7XG5cbiAgICAgICAgICAgIGNvbnN0IGF0dFJlZmVyZW5jZXMgPSBhdHQuZ2V0UmVmZXJlbmNlZFR5cGVzKFxuICAgICAgICAgICAgICBzdHJ1Y3QgaW5zdGFuY2VvZiBDb25maWdTdHJ1Y3QsXG4gICAgICAgICAgICApO1xuICAgICAgICAgICAgaWYgKGF0dFJlZmVyZW5jZXMpIHtcbiAgICAgICAgICAgICAgc3RydWN0c1RvSW1wb3J0W2ZpbGVUb0ltcG9ydF0ucHVzaCguLi5hdHRSZWZlcmVuY2VzKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgIH0pO1xuICAgICAgfSk7XG5cbiAgICAgIGNvbnN0IG5hbWVzcGFjZWRGaWxlUGF0aCA9IHBhdGguam9pbihcbiAgICAgICAgcmVzb3VyY2Uuc3RydWN0c0ZvbGRlclBhdGgsXG4gICAgICAgIHN0cnVjdEZpbGVuYW1lLFxuICAgICAgKTtcblxuICAgICAgdGhpcy5jb2RlLm9wZW5GaWxlKG5hbWVzcGFjZWRGaWxlUGF0aCk7XG4gICAgICAvLyB0aGUgc3RydWN0cyBvbmx5IG1ha2VzIHVzZSBvZiBjZGt0ZiBub3QgY29uc3RydWN0c1xuICAgICAgdGhpcy5jb2RlLmxpbmUoYGltcG9ydCAqIGFzIGNka3RmIGZyb20gJ2Nka3RmJztgKTtcbiAgICAgIE9iamVjdC5lbnRyaWVzKHN0cnVjdHNUb0ltcG9ydCkuZm9yRWFjaCgoW2ZpbGVUb0ltcG9ydCwgc3RydWN0c10pID0+IHtcbiAgICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgICAgYGltcG9ydCB7ICR7Wy4uLm5ldyBTZXQoc3RydWN0cyldLmpvaW4oXG4gICAgICAgICAgICBcIixcXG5cIixcbiAgICAgICAgICApfSB9IGZyb20gJy4vJHtwYXRoLmJhc2VuYW1lKGZpbGVUb0ltcG9ydCwgXCIudHNcIil9J2AsXG4gICAgICAgICk7XG4gICAgICB9KTtcblxuICAgICAgc3RydWN0cy5mb3JFYWNoKChzdHJ1Y3QpID0+IHtcbiAgICAgICAgaWYgKHN0cnVjdCBpbnN0YW5jZW9mIENvbmZpZ1N0cnVjdCkge1xuICAgICAgICAgIHRoaXMuZW1pdEludGVyZmFjZShyZXNvdXJjZSwgc3RydWN0KTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAvLyBXZSB1c2UgdGhlIGludGVyZmFjZSBoZXJlIGZvciB0aGUgY29uZmlndXJhdGlvbiAvIGlucHV0cyBvZiBhIHJlc291cmNlIC8gbmVzdGVkIGJsb2NrXG4gICAgICAgICAgdGhpcy5lbWl0SW50ZXJmYWNlKHJlc291cmNlLCBzdHJ1Y3QpO1xuICAgICAgICAgIC8vIEFuZCB3ZSB1c2UgdGhlIGNsYXNzIGZvciB0aGUgYXR0cmlidXRlcyAvIG91dHB1dHMgb2YgYSByZXNvdXJjZSAvIG5lc3RlZCBibG9ja1xuICAgICAgICAgIGlmICghc3RydWN0LmlzUHJvdmlkZXIpIHtcbiAgICAgICAgICAgIHRoaXMuZW1pdENsYXNzKHN0cnVjdCk7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcbiAgICAgIHRoaXMuY29kZS5jbG9zZUZpbGUobmFtZXNwYWNlZEZpbGVQYXRoKTtcbiAgICB9XG5cbiAgICAvLyBlbWl0IHRoZSBpbmRleCBmaWxlIHRoYXQgZXhwb3J0cyBhbGwgdGhlIHN0cnVjdCBmaWxlcyB3ZSd2ZSBqdXN0IGdlbmVyYXRlZFxuICAgIGNvbnN0IGluZGV4RmlsZVBhdGggPSBwYXRoLmpvaW4ocmVzb3VyY2Uuc3RydWN0c0ZvbGRlclBhdGgsIFwiaW5kZXgudHNcIik7XG5cbiAgICB0aGlzLmNvZGUub3BlbkZpbGUoaW5kZXhGaWxlUGF0aCk7XG4gICAgc3RydWN0UGF0aHMuZm9yRWFjaCgoc3RydWN0UGF0aCkgPT4ge1xuICAgICAgdGhpcy5jb2RlLmxpbmUoYGV4cG9ydCAqIGZyb20gJy4vJHtwYXRoLmJhc2VuYW1lKHN0cnVjdFBhdGgsIFwiLnRzXCIpfSdgKTtcbiAgICB9KTtcbiAgICB0aGlzLmNvZGUuY2xvc2VGaWxlKGluZGV4RmlsZVBhdGgpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0Q2xhc3Moc3RydWN0OiBTdHJ1Y3QpIHtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYGV4cG9ydCBjbGFzcyAke3N0cnVjdC5vdXRwdXRSZWZlcmVuY2VOYW1lfSBleHRlbmRzIGNka3RmLkNvbXBsZXhPYmplY3RgLFxuICAgICk7XG5cbiAgICB0aGlzLmNvZGUubGluZShcInByaXZhdGUgaXNFbXB0eU9iamVjdCA9IGZhbHNlO1wiKTtcbiAgICBpZiAoIXN0cnVjdC5pc0NsYXNzKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcInByaXZhdGUgcmVzb2x2YWJsZVZhbHVlPzogY2RrdGYuSVJlc29sdmFibGU7XCIpO1xuICAgIH1cbiAgICB0aGlzLmNvZGUubGluZSgpO1xuXG4gICAgY29uc3QgY29tbWVudCA9IHNhbml0aXplZENvbW1lbnQodGhpcy5jb2RlKTtcbiAgICBjb21tZW50LmxpbmUoYEBwYXJhbSB0ZXJyYWZvcm1SZXNvdXJjZSBUaGUgcGFyZW50IHJlc291cmNlYCk7XG4gICAgY29tbWVudC5saW5lKFxuICAgICAgYEBwYXJhbSB0ZXJyYWZvcm1BdHRyaWJ1dGUgVGhlIGF0dHJpYnV0ZSBvbiB0aGUgcGFyZW50IHJlc291cmNlIHRoaXMgY2xhc3MgaXMgcmVmZXJlbmNpbmdgLFxuICAgICk7XG4gICAgaWYgKHN0cnVjdC5pc1NpbmdsZUl0ZW0pIHtcbiAgICAgIGNvbW1lbnQuZW5kKCk7XG4gICAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgICBgcHVibGljIGNvbnN0cnVjdG9yKHRlcnJhZm9ybVJlc291cmNlOiBjZGt0Zi5JSW50ZXJwb2xhdGluZ1BhcmVudCwgdGVycmFmb3JtQXR0cmlidXRlOiBzdHJpbmcpYCxcbiAgICAgICk7XG4gICAgICB0aGlzLmNvZGUubGluZShgc3VwZXIodGVycmFmb3JtUmVzb3VyY2UsIHRlcnJhZm9ybUF0dHJpYnV0ZSwgZmFsc2UsIDApO2ApO1xuICAgICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgICB9IGVsc2UgaWYgKFxuICAgICAgc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcInNpbmdsZVwiIHx8XG4gICAgICBzdHJ1Y3QubmVzdGluZ01vZGUgPT09IFwib2JqZWN0XCJcbiAgICApIHtcbiAgICAgIGNvbW1lbnQuZW5kKCk7XG4gICAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgICBgcHVibGljIGNvbnN0cnVjdG9yKHRlcnJhZm9ybVJlc291cmNlOiBjZGt0Zi5JSW50ZXJwb2xhdGluZ1BhcmVudCwgdGVycmFmb3JtQXR0cmlidXRlOiBzdHJpbmcpYCxcbiAgICAgICk7XG4gICAgICB0aGlzLmNvZGUubGluZShgc3VwZXIodGVycmFmb3JtUmVzb3VyY2UsIHRlcnJhZm9ybUF0dHJpYnV0ZSwgZmFsc2UpO2ApO1xuICAgICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgICB9IGVsc2UgaWYgKHN0cnVjdC5uZXN0aW5nTW9kZS5zdGFydHNXaXRoKFwibWFwXCIpKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYCogQHBhcmFtIGNvbXBsZXhPYmplY3RLZXkgdGhlIGtleSBvZiB0aGlzIGl0ZW0gaW4gdGhlIG1hcGAsXG4gICAgICApO1xuICAgICAgY29tbWVudC5lbmQoKTtcbiAgICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICAgIGBwdWJsaWMgY29uc3RydWN0b3IodGVycmFmb3JtUmVzb3VyY2U6IGNka3RmLklJbnRlcnBvbGF0aW5nUGFyZW50LCB0ZXJyYWZvcm1BdHRyaWJ1dGU6IHN0cmluZywgY29tcGxleE9iamVjdEtleTogc3RyaW5nKWAsXG4gICAgICApO1xuICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgIGBzdXBlcih0ZXJyYWZvcm1SZXNvdXJjZSwgdGVycmFmb3JtQXR0cmlidXRlLCBmYWxzZSwgY29tcGxleE9iamVjdEtleSk7YCxcbiAgICAgICk7XG4gICAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICAgIH0gZWxzZSB7XG4gICAgICBjb21tZW50LmxpbmUoXG4gICAgICAgIGBAcGFyYW0gY29tcGxleE9iamVjdEluZGV4IHRoZSBpbmRleCBvZiB0aGlzIGl0ZW0gaW4gdGhlIGxpc3RgLFxuICAgICAgKTtcbiAgICAgIGNvbW1lbnQubGluZShcbiAgICAgICAgYEBwYXJhbSBjb21wbGV4T2JqZWN0SXNGcm9tU2V0IHdoZXRoZXIgdGhlIGxpc3QgaXMgd3JhcHBpbmcgYSBzZXQgKHdpbGwgYWRkIHRvbGlzdCgpIHRvIGJlIGFibGUgdG8gYWNjZXNzIGFuIGl0ZW0gdmlhIGFuIGluZGV4KWAsXG4gICAgICApO1xuICAgICAgY29tbWVudC5lbmQoKTtcbiAgICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICAgIGBwdWJsaWMgY29uc3RydWN0b3IodGVycmFmb3JtUmVzb3VyY2U6IGNka3RmLklJbnRlcnBvbGF0aW5nUGFyZW50LCB0ZXJyYWZvcm1BdHRyaWJ1dGU6IHN0cmluZywgY29tcGxleE9iamVjdEluZGV4OiBudW1iZXIsIGNvbXBsZXhPYmplY3RJc0Zyb21TZXQ6IGJvb2xlYW4pYCxcbiAgICAgICk7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYHN1cGVyKHRlcnJhZm9ybVJlc291cmNlLCB0ZXJyYWZvcm1BdHRyaWJ1dGUsIGNvbXBsZXhPYmplY3RJc0Zyb21TZXQsIGNvbXBsZXhPYmplY3RJbmRleCk7YCxcbiAgICAgICk7XG4gICAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgdGhpcy5lbWl0SW50ZXJuYWxWYWx1ZUdldHRlcihzdHJ1Y3QpO1xuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgdGhpcy5lbWl0SW50ZXJuYWxWYWx1ZVNldHRlcihzdHJ1Y3QpO1xuXG4gICAgZm9yIChjb25zdCBhdHQgb2Ygc3RydWN0LmF0dHJpYnV0ZXMpIHtcbiAgICAgIHRoaXMuYXR0cmlidXRlc0VtaXR0ZXIuZW1pdChcbiAgICAgICAgYXR0LFxuICAgICAgICB0aGlzLmF0dHJpYnV0ZXNFbWl0dGVyLm5lZWRzUmVzZXRFc2NhcGUoYXR0LCBzdHJ1Y3QuYXR0cmlidXRlcyksXG4gICAgICAgIHRoaXMuYXR0cmlidXRlc0VtaXR0ZXIubmVlZHNJbnB1dEVzY2FwZShhdHQsIHN0cnVjdC5hdHRyaWJ1dGVzKSxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIGlmIChcbiAgICAgICFzdHJ1Y3QuaXNTaW5nbGVJdGVtICYmXG4gICAgICAoc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcImxpc3RcIiB8fCBzdHJ1Y3QubmVzdGluZ01vZGUgPT09IFwic2V0XCIpXG4gICAgKSB7XG4gICAgICB0aGlzLmVtaXRDb21wbGV4TGlzdENsYXNzKHN0cnVjdCk7XG4gICAgfSBlbHNlIGlmIChzdHJ1Y3QubmVzdGluZ01vZGUgPT09IFwibWFwXCIpIHtcbiAgICAgIHRoaXMuZW1pdENvbXBsZXhNYXBDbGFzcyhzdHJ1Y3QpO1xuICAgIH0gZWxzZSBpZiAoc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcIm1hcGxpc3RcIikge1xuICAgICAgdGhpcy5lbWl0Q29tcGxleE1hcExpc3RDbGFzc2VzKHN0cnVjdCk7XG4gICAgfSBlbHNlIGlmIChzdHJ1Y3QubmVzdGluZ01vZGUgPT09IFwibWFwc2V0XCIpIHtcbiAgICAgIHRoaXMuZW1pdENvbXBsZXhNYXBMaXN0Q2xhc3NlcyhzdHJ1Y3QpO1xuICAgIH0gZWxzZSBpZiAoc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcImxpc3RtYXBcIikge1xuICAgICAgdGhpcy5lbWl0Q29tcGxleExpc3RNYXBDbGFzc2VzKHN0cnVjdCwgZmFsc2UpO1xuICAgIH0gZWxzZSBpZiAoc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcInNldG1hcFwiKSB7XG4gICAgICB0aGlzLmVtaXRDb21wbGV4TGlzdE1hcENsYXNzZXMoc3RydWN0LCB0cnVlKTtcbiAgICB9IGVsc2UgaWYgKFxuICAgICAgc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcImxpc3RsaXN0XCIgfHxcbiAgICAgIHN0cnVjdC5uZXN0aW5nTW9kZSA9PT0gXCJsaXN0c2V0XCIgfHxcbiAgICAgIHN0cnVjdC5uZXN0aW5nTW9kZSA9PT0gXCJzZXRsaXN0XCIgfHxcbiAgICAgIHN0cnVjdC5uZXN0aW5nTW9kZSA9PT0gXCJzZXRzZXRcIlxuICAgICkge1xuICAgICAgdGhpcy5lbWl0Q29tcGxleExpc3RMaXN0Q2xhc3NlcyhzdHJ1Y3QpO1xuICAgIH1cbiAgICAvLyBvdGhlciB0eXBlcyBvZiBuZXN0ZWQgY29sbGVjdGlvbnMgYXJlbid0IHN1cHBvcnRlZFxuICB9XG5cbiAgcHJpdmF0ZSBlbWl0Q29tcGxleExpc3RDbGFzcyhzdHJ1Y3Q6IFN0cnVjdCkge1xuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBleHBvcnQgY2xhc3MgJHtzdHJ1Y3QubGlzdE5hbWV9IGV4dGVuZHMgY2RrdGYuQ29tcGxleExpc3RgLFxuICAgICk7XG5cbiAgICBpZiAoc3RydWN0LmFzc2lnbmFibGUpIHtcbiAgICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgICBgcHVibGljIGludGVybmFsVmFsdWU/IDogJHtzdHJ1Y3QubmFtZX1bXSB8IGNka3RmLklSZXNvbHZhYmxlYCxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICBjb25zdCBjb25zdHJ1Y3RvckNvbW1lbnQgPSBzYW5pdGl6ZWRDb21tZW50KHRoaXMuY29kZSk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmxpbmUoYEBwYXJhbSB0ZXJyYWZvcm1SZXNvdXJjZSBUaGUgcGFyZW50IHJlc291cmNlYCk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmxpbmUoXG4gICAgICBgQHBhcmFtIHRlcnJhZm9ybUF0dHJpYnV0ZSBUaGUgYXR0cmlidXRlIG9uIHRoZSBwYXJlbnQgcmVzb3VyY2UgdGhpcyBjbGFzcyBpcyByZWZlcmVuY2luZ2AsXG4gICAgKTtcbiAgICBjb25zdHJ1Y3RvckNvbW1lbnQubGluZShcbiAgICAgIGBAcGFyYW0gd3JhcHNTZXQgd2hldGhlciB0aGUgbGlzdCBpcyB3cmFwcGluZyBhIHNldCAod2lsbCBhZGQgdG9saXN0KCkgdG8gYmUgYWJsZSB0byBhY2Nlc3MgYW4gaXRlbSB2aWEgYW4gaW5kZXgpYCxcbiAgICApO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5lbmQoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYGNvbnN0cnVjdG9yKHByb3RlY3RlZCB0ZXJyYWZvcm1SZXNvdXJjZTogY2RrdGYuSUludGVycG9sYXRpbmdQYXJlbnQsIHByb3RlY3RlZCB0ZXJyYWZvcm1BdHRyaWJ1dGU6IHN0cmluZywgcHJvdGVjdGVkIHdyYXBzU2V0OiBib29sZWFuKWAsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUubGluZShgc3VwZXIodGVycmFmb3JtUmVzb3VyY2UsIHRlcnJhZm9ybUF0dHJpYnV0ZSwgd3JhcHNTZXQpYCk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgY29uc3QgZ2V0dGVyQ29tbWVudCA9IHNhbml0aXplZENvbW1lbnQodGhpcy5jb2RlKTtcbiAgICBnZXR0ZXJDb21tZW50LmxpbmUoYEBwYXJhbSBpbmRleCB0aGUgaW5kZXggb2YgdGhlIGl0ZW0gdG8gcmV0dXJuYCk7XG4gICAgZ2V0dGVyQ29tbWVudC5lbmQoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYHB1YmxpYyBnZXQoaW5kZXg6IG51bWJlcik6ICR7c3RydWN0Lm91dHB1dFJlZmVyZW5jZU5hbWV9YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYHJldHVybiBuZXcgJHtzdHJ1Y3Qub3V0cHV0UmVmZXJlbmNlTmFtZX0odGhpcy50ZXJyYWZvcm1SZXNvdXJjZSwgdGhpcy50ZXJyYWZvcm1BdHRyaWJ1dGUsIGluZGV4LCB0aGlzLndyYXBzU2V0KTtgLFxuICAgICk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRDb21wbGV4TWFwQ2xhc3Moc3RydWN0OiBTdHJ1Y3QpIHtcbiAgICB0aGlzLmNvZGUubGluZSgpO1xuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICBgZXhwb3J0IGNsYXNzICR7c3RydWN0Lm1hcE5hbWV9IGV4dGVuZHMgY2RrdGYuQ29tcGxleE1hcGAsXG4gICAgKTtcblxuICAgIGlmIChzdHJ1Y3QuYXNzaWduYWJsZSkge1xuICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgIGBwdWJsaWMgaW50ZXJuYWxWYWx1ZT8gOiB7IFtrZXk6IHN0cmluZ106ICR7c3RydWN0Lm5hbWV9IH0gfCBjZGt0Zi5JUmVzb2x2YWJsZWAsXG4gICAgICApO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgY29uc3QgY29uc3RydWN0b3JDb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5saW5lKGBAcGFyYW0gdGVycmFmb3JtUmVzb3VyY2UgVGhlIHBhcmVudCByZXNvdXJjZWApO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5saW5lKFxuICAgICAgYEBwYXJhbSB0ZXJyYWZvcm1BdHRyaWJ1dGUgVGhlIGF0dHJpYnV0ZSBvbiB0aGUgcGFyZW50IHJlc291cmNlIHRoaXMgY2xhc3MgaXMgcmVmZXJlbmNpbmdgLFxuICAgICk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmVuZCgpO1xuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICBgY29uc3RydWN0b3IocHJvdGVjdGVkIHRlcnJhZm9ybVJlc291cmNlOiBjZGt0Zi5JSW50ZXJwb2xhdGluZ1BhcmVudCwgcHJvdGVjdGVkIHRlcnJhZm9ybUF0dHJpYnV0ZTogc3RyaW5nKWAsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUubGluZShgc3VwZXIodGVycmFmb3JtUmVzb3VyY2UsIHRlcnJhZm9ybUF0dHJpYnV0ZSlgKTtcbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuXG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICBjb25zdCBnZXR0ZXJDb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGdldHRlckNvbW1lbnQubGluZShgQHBhcmFtIGtleSB0aGUga2V5IG9mIHRoZSBpdGVtIHRvIHJldHVybmApO1xuICAgIGdldHRlckNvbW1lbnQuZW5kKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBwdWJsaWMgZ2V0KGtleTogc3RyaW5nKTogJHtzdHJ1Y3Qub3V0cHV0UmVmZXJlbmNlTmFtZX1gLFxuICAgICk7XG4gICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICBgcmV0dXJuIG5ldyAke3N0cnVjdC5vdXRwdXRSZWZlcmVuY2VOYW1lfSh0aGlzLnRlcnJhZm9ybVJlc291cmNlLCB0aGlzLnRlcnJhZm9ybUF0dHJpYnV0ZSwga2V5KTtgLFxuICAgICk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRDb21wbGV4TWFwTGlzdENsYXNzZXMoc3RydWN0OiBTdHJ1Y3QpIHtcbiAgICB0aGlzLmVtaXRDb21wbGV4TWFwTGlzdENsYXNzKHN0cnVjdCk7XG4gICAgdGhpcy5lbWl0Q29tcGxleE1hcENsYXNzKHN0cnVjdCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRDb21wbGV4TWFwTGlzdENsYXNzKHN0cnVjdDogU3RydWN0KSB7XG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYGV4cG9ydCBjbGFzcyAke3N0cnVjdC5tYXBMaXN0TmFtZX0gZXh0ZW5kcyBjZGt0Zi5NYXBMaXN0YCxcbiAgICApO1xuXG4gICAgaWYgKHN0cnVjdC5hc3NpZ25hYmxlKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYHB1YmxpYyBpbnRlcm5hbFZhbHVlPyA6ICR7c3RydWN0Lm1hcE5hbWV9W10gfCBjZGt0Zi5JUmVzb2x2YWJsZWAsXG4gICAgICApO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgY29uc3QgY29uc3RydWN0b3JDb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5saW5lKGBAcGFyYW0gdGVycmFmb3JtUmVzb3VyY2UgVGhlIHBhcmVudCByZXNvdXJjZWApO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5saW5lKFxuICAgICAgYEBwYXJhbSB0ZXJyYWZvcm1BdHRyaWJ1dGUgVGhlIGF0dHJpYnV0ZSBvbiB0aGUgcGFyZW50IHJlc291cmNlIHRoaXMgY2xhc3MgaXMgcmVmZXJlbmNpbmdgLFxuICAgICk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmxpbmUoXG4gICAgICBgQHBhcmFtIHdyYXBzU2V0IHdoZXRoZXIgdGhlIGxpc3QgaXMgd3JhcHBpbmcgYSBzZXQgKHdpbGwgYWRkIHRvbGlzdCgpIHRvIGJlIGFibGUgdG8gYWNjZXNzIGFuIGl0ZW0gdmlhIGFuIGluZGV4KWAsXG4gICAgKTtcbiAgICBjb25zdHJ1Y3RvckNvbW1lbnQuZW5kKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBjb25zdHJ1Y3Rvcihwcm90ZWN0ZWQgdGVycmFmb3JtUmVzb3VyY2U6IGNka3RmLklJbnRlcnBvbGF0aW5nUGFyZW50LCBwcm90ZWN0ZWQgdGVycmFmb3JtQXR0cmlidXRlOiBzdHJpbmcsIHByb3RlY3RlZCB3cmFwc1NldDogYm9vbGVhbilgLFxuICAgICk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYHN1cGVyKHRlcnJhZm9ybVJlc291cmNlLCB0ZXJyYWZvcm1BdHRyaWJ1dGUsIHdyYXBzU2V0KWApO1xuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG5cbiAgICB0aGlzLmNvZGUubGluZSgpO1xuICAgIGNvbnN0IGdldHRlckNvbW1lbnQgPSBzYW5pdGl6ZWRDb21tZW50KHRoaXMuY29kZSk7XG4gICAgZ2V0dGVyQ29tbWVudC5saW5lKGBAcGFyYW0gaW5kZXggdGhlIGluZGV4IG9mIHRoZSBpdGVtIHRvIHJldHVybmApO1xuICAgIGdldHRlckNvbW1lbnQuZW5kKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhgcHVibGljIGdldChpbmRleDogbnVtYmVyKTogJHtzdHJ1Y3QubWFwTmFtZX1gKTtcbiAgICB0aGlzLmNvZGUubGluZShgcmV0dXJuIG5ldyAke3N0cnVjdC5tYXBOYW1lfSh0aGlzLCBcXGBbXFwke2luZGV4fV1cXGApO2ApO1xuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG5cbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0Q29tcGxleExpc3RNYXBDbGFzc2VzKHN0cnVjdDogU3RydWN0LCBpc1NldDogYm9vbGVhbikge1xuICAgIHRoaXMuZW1pdENvbXBsZXhMaXN0TWFwQ2xhc3Moc3RydWN0LCBpc1NldCk7XG4gICAgdGhpcy5lbWl0Q29tcGxleExpc3RDbGFzcyhzdHJ1Y3QpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0Q29tcGxleExpc3RNYXBDbGFzcyhzdHJ1Y3Q6IFN0cnVjdCwgaXNTZXQ6IGJvb2xlYW4pIHtcbiAgICB0aGlzLmNvZGUubGluZSgpO1xuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICBgZXhwb3J0IGNsYXNzICR7c3RydWN0Lmxpc3RNYXBOYW1lfSBleHRlbmRzIGNka3RmLkNvbXBsZXhNYXBgLFxuICAgICk7XG5cbiAgICBpZiAoc3RydWN0LmFzc2lnbmFibGUpIHtcbiAgICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgICBgcHVibGljIGludGVybmFsVmFsdWU/IDogeyBba2V5OiBzdHJpbmddOiAke3N0cnVjdC5uYW1lfVtdIH0gfCBjZGt0Zi5JUmVzb2x2YWJsZWAsXG4gICAgICApO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgY29uc3QgY29uc3RydWN0b3JDb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5saW5lKGBAcGFyYW0gdGVycmFmb3JtUmVzb3VyY2UgVGhlIHBhcmVudCByZXNvdXJjZWApO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5saW5lKFxuICAgICAgYEBwYXJhbSB0ZXJyYWZvcm1BdHRyaWJ1dGUgVGhlIGF0dHJpYnV0ZSBvbiB0aGUgcGFyZW50IHJlc291cmNlIHRoaXMgY2xhc3MgaXMgcmVmZXJlbmNpbmdgLFxuICAgICk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmVuZCgpO1xuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soXG4gICAgICBgY29uc3RydWN0b3IocHJvdGVjdGVkIHRlcnJhZm9ybVJlc291cmNlOiBjZGt0Zi5JSW50ZXJwb2xhdGluZ1BhcmVudCwgcHJvdGVjdGVkIHRlcnJhZm9ybUF0dHJpYnV0ZTogc3RyaW5nKWAsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUubGluZShgc3VwZXIodGVycmFmb3JtUmVzb3VyY2UsIHRlcnJhZm9ybUF0dHJpYnV0ZSlgKTtcbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuXG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICBjb25zdCBnZXR0ZXJDb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGdldHRlckNvbW1lbnQubGluZShgQHBhcmFtIGtleSB0aGUga2V5IG9mIHRoZSBpdGVtIHRvIHJldHVybmApO1xuICAgIGdldHRlckNvbW1lbnQuZW5kKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhgcHVibGljIGdldChrZXk6IHN0cmluZyk6ICR7c3RydWN0Lmxpc3ROYW1lfWApO1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYHJldHVybiBuZXcgJHtzdHJ1Y3QubGlzdE5hbWV9KHRoaXMsIFxcYFtcXCR7a2V5fV1cXGAsICR7aXNTZXR9KTtgLFxuICAgICk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRDb21wbGV4TGlzdExpc3RDbGFzc2VzKHN0cnVjdDogU3RydWN0KSB7XG4gICAgdGhpcy5lbWl0Q29tcGxleExpc3RMaXN0Q2xhc3Moc3RydWN0KTtcbiAgICB0aGlzLmVtaXRDb21wbGV4TGlzdENsYXNzKHN0cnVjdCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRDb21wbGV4TGlzdExpc3RDbGFzcyhzdHJ1Y3Q6IFN0cnVjdCkge1xuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBleHBvcnQgY2xhc3MgJHtzdHJ1Y3QubGlzdExpc3ROYW1lfSBleHRlbmRzIGNka3RmLk1hcExpc3RgLCAvLyBkZXNwaXRlIG5hbWUsIG5lZWQgdGhlIHNhbWUgYmVoYXZpb3JcbiAgICApO1xuXG4gICAgaWYgKHN0cnVjdC5hc3NpZ25hYmxlKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYHB1YmxpYyBpbnRlcm5hbFZhbHVlPyA6ICR7c3RydWN0Lm5hbWV9W11bXSB8IGNka3RmLklSZXNvbHZhYmxlYCxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICBjb25zdCBjb25zdHJ1Y3RvckNvbW1lbnQgPSBzYW5pdGl6ZWRDb21tZW50KHRoaXMuY29kZSk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmxpbmUoYEBwYXJhbSB0ZXJyYWZvcm1SZXNvdXJjZSBUaGUgcGFyZW50IHJlc291cmNlYCk7XG4gICAgY29uc3RydWN0b3JDb21tZW50LmxpbmUoXG4gICAgICBgQHBhcmFtIHRlcnJhZm9ybUF0dHJpYnV0ZSBUaGUgYXR0cmlidXRlIG9uIHRoZSBwYXJlbnQgcmVzb3VyY2UgdGhpcyBjbGFzcyBpcyByZWZlcmVuY2luZ2AsXG4gICAgKTtcbiAgICBjb25zdHJ1Y3RvckNvbW1lbnQubGluZShcbiAgICAgIGBAcGFyYW0gd3JhcHNTZXQgd2hldGhlciB0aGUgbGlzdCBpcyB3cmFwcGluZyBhIHNldCAod2lsbCBhZGQgdG9saXN0KCkgdG8gYmUgYWJsZSB0byBhY2Nlc3MgYW4gaXRlbSB2aWEgYW4gaW5kZXgpYCxcbiAgICApO1xuICAgIGNvbnN0cnVjdG9yQ29tbWVudC5lbmQoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYGNvbnN0cnVjdG9yKHByb3RlY3RlZCB0ZXJyYWZvcm1SZXNvdXJjZTogY2RrdGYuSUludGVycG9sYXRpbmdQYXJlbnQsIHByb3RlY3RlZCB0ZXJyYWZvcm1BdHRyaWJ1dGU6IHN0cmluZywgcHJvdGVjdGVkIHdyYXBzU2V0OiBib29sZWFuKWAsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUubGluZShgc3VwZXIodGVycmFmb3JtUmVzb3VyY2UsIHRlcnJhZm9ybUF0dHJpYnV0ZSwgd3JhcHNTZXQpYCk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgY29uc3QgZ2V0dGVyQ29tbWVudCA9IHNhbml0aXplZENvbW1lbnQodGhpcy5jb2RlKTtcbiAgICBnZXR0ZXJDb21tZW50LmxpbmUoYEBwYXJhbSBpbmRleCB0aGUgaW5kZXggb2YgdGhlIGl0ZW0gdG8gcmV0dXJuYCk7XG4gICAgZ2V0dGVyQ29tbWVudC5lbmQoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKGBwdWJsaWMgZ2V0KGluZGV4OiBudW1iZXIpOiAke3N0cnVjdC5saXN0TmFtZX1gKTtcbiAgICB0aGlzLmNvZGUubGluZShcbiAgICAgIGByZXR1cm4gbmV3ICR7c3RydWN0Lmxpc3ROYW1lfSh0aGlzLCBcXGBbXFwke2luZGV4fV1cXGAsICR7XG4gICAgICAgIHN0cnVjdC5uZXN0aW5nTW9kZSA9PT0gXCJzZXRsaXN0XCIgfHwgc3RydWN0Lm5lc3RpbmdNb2RlID09PSBcInNldHNldFwiXG4gICAgICB9KTtgLFxuICAgICk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcblxuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRJbnRlcm5hbFZhbHVlR2V0dGVyKHN0cnVjdDogU3RydWN0KSB7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBwdWJsaWMgZ2V0IGludGVybmFsVmFsdWUoKTogJHtzdHJ1Y3QubmFtZX0ke1xuICAgICAgICAhc3RydWN0LmlzQ2xhc3MgPyBcIiB8IGNka3RmLklSZXNvbHZhYmxlXCIgOiBcIlwiXG4gICAgICB9IHwgdW5kZWZpbmVkYCxcbiAgICApO1xuXG4gICAgaWYgKCFzdHJ1Y3QuaXNDbGFzcykge1xuICAgICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcImlmICh0aGlzLnJlc29sdmFibGVWYWx1ZSlcIik7XG4gICAgICB0aGlzLmNvZGUubGluZShcInJldHVybiB0aGlzLnJlc29sdmFibGVWYWx1ZTtcIik7XG4gICAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5saW5lKFwibGV0IGhhc0FueVZhbHVlcyA9IHRoaXMuaXNFbXB0eU9iamVjdDtcIik7XG4gICAgdGhpcy5jb2RlLmxpbmUoXCJjb25zdCBpbnRlcm5hbFZhbHVlUmVzdWx0OiBhbnkgPSB7fTtcIik7XG4gICAgZm9yIChjb25zdCBhdHQgb2Ygc3RydWN0LmF0dHJpYnV0ZXMpIHtcbiAgICAgIGlmIChhdHQuaXNTdG9yZWQpIHtcbiAgICAgICAgaWYgKGF0dC5nZXR0ZXJUeXBlLl90eXBlID09PSBcInN0b3JlZF9jbGFzc1wiKSB7XG4gICAgICAgICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgICAgICAgIGBpZiAodGhpcy4ke2F0dC5zdG9yYWdlTmFtZX0/LmludGVybmFsVmFsdWUgIT09IHVuZGVmaW5lZClgLFxuICAgICAgICAgICk7XG4gICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgdGhpcy5jb2RlLm9wZW5CbG9jayhgaWYgKHRoaXMuJHthdHQuc3RvcmFnZU5hbWV9ICE9PSB1bmRlZmluZWQpYCk7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5jb2RlLmxpbmUoXCJoYXNBbnlWYWx1ZXMgPSB0cnVlO1wiKTtcbiAgICAgICAgaWYgKGF0dC5nZXR0ZXJUeXBlLl90eXBlID09PSBcInN0b3JlZF9jbGFzc1wiKSB7XG4gICAgICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgICAgICBgaW50ZXJuYWxWYWx1ZVJlc3VsdC4ke2F0dC5uYW1lfSA9IHRoaXMuJHthdHQuc3RvcmFnZU5hbWV9Py5pbnRlcm5hbFZhbHVlO2AsXG4gICAgICAgICAgKTtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgICAgIGBpbnRlcm5hbFZhbHVlUmVzdWx0LiR7YXR0Lm5hbWV9ID0gdGhpcy4ke2F0dC5zdG9yYWdlTmFtZX07YCxcbiAgICAgICAgICApO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gICAgICB9XG4gICAgfVxuICAgIHRoaXMuY29kZS5saW5lKFwicmV0dXJuIGhhc0FueVZhbHVlcyA/IGludGVybmFsVmFsdWVSZXN1bHQgOiB1bmRlZmluZWQ7XCIpO1xuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRJbnRlcm5hbFZhbHVlU2V0dGVyKHN0cnVjdDogU3RydWN0KSB7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBwdWJsaWMgc2V0IGludGVybmFsVmFsdWUodmFsdWU6ICR7c3RydWN0Lm5hbWV9JHtcbiAgICAgICAgIXN0cnVjdC5pc0NsYXNzID8gXCIgfCBjZGt0Zi5JUmVzb2x2YWJsZVwiIDogXCJcIlxuICAgICAgfSB8IHVuZGVmaW5lZClgLFxuICAgICk7XG5cbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFwiaWYgKHZhbHVlID09PSB1bmRlZmluZWQpXCIpO1xuICAgIHRoaXMuY29kZS5saW5lKFwidGhpcy5pc0VtcHR5T2JqZWN0ID0gZmFsc2U7XCIpO1xuICAgIGlmICghc3RydWN0LmlzQ2xhc3MpIHtcbiAgICAgIHRoaXMuY29kZS5saW5lKFwidGhpcy5yZXNvbHZhYmxlVmFsdWUgPSB1bmRlZmluZWQ7XCIpO1xuICAgIH1cbiAgICBmb3IgKGNvbnN0IGF0dCBvZiBzdHJ1Y3QuYXR0cmlidXRlcykge1xuICAgICAgaWYgKGF0dC5pc1N0b3JlZCkge1xuICAgICAgICBpZiAoYXR0LnNldHRlclR5cGUuX3R5cGUgPT09IFwic3RvcmVkX2NsYXNzXCIpIHtcbiAgICAgICAgICB0aGlzLmNvZGUubGluZShgdGhpcy4ke2F0dC5zdG9yYWdlTmFtZX0uaW50ZXJuYWxWYWx1ZSA9IHVuZGVmaW5lZDtgKTtcbiAgICAgICAgfSBlbHNlIGlmIChhdHQuc2V0dGVyVHlwZS5fdHlwZSAhPT0gXCJub25lXCIpIHtcbiAgICAgICAgICB0aGlzLmNvZGUubGluZShgdGhpcy4ke2F0dC5zdG9yYWdlTmFtZX0gPSB1bmRlZmluZWQ7YCk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgICBpZiAoIXN0cnVjdC5pc0NsYXNzKSB7XG4gICAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFwiZWxzZSBpZiAoY2RrdGYuVG9rZW5pemF0aW9uLmlzUmVzb2x2YWJsZSh2YWx1ZSkpXCIpO1xuICAgICAgdGhpcy5jb2RlLmxpbmUoXCJ0aGlzLmlzRW1wdHlPYmplY3QgPSBmYWxzZTtcIik7XG4gICAgICB0aGlzLmNvZGUubGluZShcInRoaXMucmVzb2x2YWJsZVZhbHVlID0gdmFsdWU7XCIpO1xuICAgICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgICB9XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcImVsc2VcIik7XG4gICAgdGhpcy5jb2RlLmxpbmUoXCJ0aGlzLmlzRW1wdHlPYmplY3QgPSBPYmplY3Qua2V5cyh2YWx1ZSkubGVuZ3RoID09PSAwO1wiKTtcbiAgICBpZiAoIXN0cnVjdC5pc0NsYXNzKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcInRoaXMucmVzb2x2YWJsZVZhbHVlID0gdW5kZWZpbmVkO1wiKTtcbiAgICB9XG4gICAgZm9yIChjb25zdCBhdHQgb2Ygc3RydWN0LmF0dHJpYnV0ZXMpIHtcbiAgICAgIGlmIChhdHQuaXNTdG9yZWQpIHtcbiAgICAgICAgaWYgKGF0dC5zZXR0ZXJUeXBlLl90eXBlID09PSBcInN0b3JlZF9jbGFzc1wiKSB7XG4gICAgICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgICAgICBgdGhpcy4ke2F0dC5zdG9yYWdlTmFtZX0uaW50ZXJuYWxWYWx1ZSA9IHZhbHVlLiR7YXR0Lm5hbWV9O2AsXG4gICAgICAgICAgKTtcbiAgICAgICAgfSBlbHNlIGlmIChhdHQuc2V0dGVyVHlwZS5fdHlwZSAhPT0gXCJub25lXCIpIHtcbiAgICAgICAgICB0aGlzLmNvZGUubGluZShgdGhpcy4ke2F0dC5zdG9yYWdlTmFtZX0gPSB2YWx1ZS4ke2F0dC5uYW1lfTtgKTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRUb0hjbFRlcnJhZm9ybUZ1bmN0aW9uKHN0cnVjdDogU3RydWN0KSB7XG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYGV4cG9ydCBmdW5jdGlvbiAke2Rvd25jYXNlRmlyc3Qoc3RydWN0Lm5hbWUpfVRvSGNsVGVycmFmb3JtKHN0cnVjdD86ICR7XG4gICAgICAgIHN0cnVjdC5pc1NpbmdsZUl0ZW0gJiYgIXN0cnVjdC5pc1Byb3ZpZGVyXG4gICAgICAgICAgPyBgJHtzdHJ1Y3QubmFtZX1PdXRwdXRSZWZlcmVuY2UgfCBgXG4gICAgICAgICAgOiBcIlwiXG4gICAgICB9JHtzdHJ1Y3QubmFtZX0keyFzdHJ1Y3QuaXNDbGFzcyA/IFwiIHwgY2RrdGYuSVJlc29sdmFibGVcIiA6IFwiXCJ9KTogYW55YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYGlmICghY2RrdGYuY2FuSW5zcGVjdChzdHJ1Y3QpIHx8IGNka3RmLlRva2VuaXphdGlvbi5pc1Jlc29sdmFibGUoc3RydWN0KSkgeyByZXR1cm4gc3RydWN0OyB9YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soYGlmIChjZGt0Zi5pc0NvbXBsZXhFbGVtZW50KHN0cnVjdCkpYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICBgdGhyb3cgbmV3IEVycm9yKFwiQSBjb21wbGV4IGVsZW1lbnQgd2FzIHVzZWQgYXMgY29uZmlndXJhdGlvbiwgdGhpcyBpcyBub3Qgc3VwcG9ydGVkOiBodHRwczovL2Nkay50Zi9jb21wbGV4LW9iamVjdC1hcy1jb25maWd1cmF0aW9uXCIpO2AsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuXG4gICAgdGhpcy5jb2RlLm9wZW4oYGNvbnN0IGF0dHJzID0ge2ApO1xuXG4gICAgZm9yIChjb25zdCBhdHQgb2Ygc3RydWN0LmFzc2lnbmFibGVBdHRyaWJ1dGVzKSB7XG4gICAgICB0aGlzLmF0dHJpYnV0ZXNFbWl0dGVyLmVtaXRUb0hjbFRlcnJhZm9ybShhdHQsIHRydWUpO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5jbG9zZShgfTtgKTtcblxuICAgIGlmIChzdHJ1Y3QuYXNzaWduYWJsZUF0dHJpYnV0ZXMubGVuZ3RoID4gMCkge1xuICAgICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICAgIHRoaXMuY29kZS5saW5lKGAvLyByZW1vdmUgdW5kZWZpbmVkIGF0dHJpYnV0ZXNgKTtcbiAgICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgICBgcmV0dXJuIE9iamVjdC5mcm9tRW50cmllcyhPYmplY3QuZW50cmllcyhhdHRycykuZmlsdGVyKChbXywgdmFsdWVdKSA9PiB2YWx1ZSAhPT0gdW5kZWZpbmVkICYmIHZhbHVlLnZhbHVlICE9PSB1bmRlZmluZWQpKTtgLFxuICAgICAgKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5jb2RlLmxpbmUoYHJldHVybiBhdHRycztgKTtcbiAgICB9XG5cbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRUb1RlcnJhZm9ybUZ1bmN0aW9uKHN0cnVjdDogU3RydWN0KSB7XG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYGV4cG9ydCBmdW5jdGlvbiAke2Rvd25jYXNlRmlyc3Qoc3RydWN0Lm5hbWUpfVRvVGVycmFmb3JtKHN0cnVjdD86ICR7XG4gICAgICAgIHN0cnVjdC5pc1NpbmdsZUl0ZW0gJiYgIXN0cnVjdC5pc1Byb3ZpZGVyXG4gICAgICAgICAgPyBgJHtzdHJ1Y3QubmFtZX1PdXRwdXRSZWZlcmVuY2UgfCBgXG4gICAgICAgICAgOiBcIlwiXG4gICAgICB9JHtzdHJ1Y3QubmFtZX0keyFzdHJ1Y3QuaXNDbGFzcyA/IFwiIHwgY2RrdGYuSVJlc29sdmFibGVcIiA6IFwiXCJ9KTogYW55YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYGlmICghY2RrdGYuY2FuSW5zcGVjdChzdHJ1Y3QpIHx8IGNka3RmLlRva2VuaXphdGlvbi5pc1Jlc29sdmFibGUoc3RydWN0KSkgeyByZXR1cm4gc3RydWN0OyB9YCxcbiAgICApO1xuICAgIHRoaXMuY29kZS5vcGVuQmxvY2soYGlmIChjZGt0Zi5pc0NvbXBsZXhFbGVtZW50KHN0cnVjdCkpYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICBgdGhyb3cgbmV3IEVycm9yKFwiQSBjb21wbGV4IGVsZW1lbnQgd2FzIHVzZWQgYXMgY29uZmlndXJhdGlvbiwgdGhpcyBpcyBub3Qgc3VwcG9ydGVkOiBodHRwczovL2Nkay50Zi9jb21wbGV4LW9iamVjdC1hcy1jb25maWd1cmF0aW9uXCIpO2AsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuXG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcInJldHVyblwiKTtcbiAgICBmb3IgKGNvbnN0IGF0dCBvZiBzdHJ1Y3QuYXNzaWduYWJsZUF0dHJpYnV0ZXMpIHtcbiAgICAgIHRoaXMuYXR0cmlidXRlc0VtaXR0ZXIuZW1pdFRvVGVycmFmb3JtKGF0dCwgdHJ1ZSk7XG4gICAgfVxuICAgIHRoaXMuY29kZS5jbG9zZUJsb2NrKFwiO1wiKTtcbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gIH1cbn1cbiJdfQ==