"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResourceEmitter = void 0;
const attributes_emitter_1 = require("./attributes-emitter");
const sanitized_comments_1 = require("../sanitized-comments");
class ResourceEmitter {
    constructor(code) {
        this.code = code;
        this.attributesEmitter = new attributes_emitter_1.AttributesEmitter(this.code);
    }
    emit(resource) {
        this.code.line();
        const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        comment.line(`Represents a {@link ${resource.linkToDocs} ${resource.terraformResourceType}}`);
        comment.end();
        this.code.openBlock(`export class ${resource.className} extends cdktf.${resource.parentClassName}`);
        this.emitHeader("STATIC PROPERTIES");
        this.emitStaticProperties(resource);
        this.emitHeader("STATIC Methods");
        this.emitStaticMethods(resource);
        this.emitHeader("INITIALIZER");
        this.emitInitializer(resource);
        this.emitHeader("ATTRIBUTES");
        this.emitResourceAttributes(resource);
        // synthesis
        this.emitHeader("SYNTHESIS");
        this.emitResourceSynthesis(resource);
        this.emitHclResourceSynthesis(resource);
        this.code.closeBlock(); // construct
    }
    emitHeader(title) {
        this.code.line();
        this.code.line("// " + "=".repeat(title.length));
        this.code.line(`// ${title}`);
        this.code.line("// " + "=".repeat(title.length));
    }
    emitStaticProperties(resource) {
        this.code.line(`public static readonly tfResourceType = "${resource.terraformResourceType}";`);
    }
    emitStaticMethods(resource) {
        const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        comment.line(`Generates CDKTF code for importing a ${resource.className} resource upon running "cdktf plan <stack-name>"`);
        comment.line(`@param scope The scope in which to define this construct`);
        comment.line(`@param importToId The construct id used in the generated config for the ${resource.className} to import`);
        comment.line(`@param importFromId The id of the existing ${resource.className} that should be imported. Refer to the {@link ${resource.linkToDocs}#import import section} in the documentation of this resource for the id to use`);
        comment.line(`@param provider? Optional instance of the provider where the ${resource.className} to import is found`);
        comment.end();
        this.code.line(`public static generateConfigForImport(scope: Construct, importToId: string, importFromId: string, provider?: cdktf.TerraformProvider) {
        return new cdktf.ImportableResource(scope, importToId, { terraformResourceType: "${resource.terraformResourceType}", importId: importFromId, provider });
      }`);
    }
    emitHclResourceSynthesis(resource) {
        this.code.line();
        this.code.openBlock(`protected synthesizeHclAttributes(): { [name: string]: any }`);
        this.code.open(`const attrs = {`);
        for (const att of resource.synthesizableAttributes) {
            this.attributesEmitter.emitToHclTerraform(att, false);
        }
        this.code.close(`};`);
        if (resource.synthesizableAttributes.length > 0) {
            this.code.line();
            this.code.line(`// remove undefined attributes`);
            this.code.line(`return Object.fromEntries(Object.entries(attrs).filter(([_, value]) => value !== undefined && value.value !== undefined ))`);
        }
        else {
            this.code.line(`return attrs;`);
        }
        this.code.closeBlock();
    }
    emitResourceSynthesis(resource) {
        this.code.line();
        this.code.openBlock(`protected synthesizeAttributes(): { [name: string]: any }`);
        this.code.open(`return {`);
        for (const att of resource.synthesizableAttributes) {
            this.attributesEmitter.emitToTerraform(att, false);
        }
        this.code.close(`};`);
        this.code.closeBlock();
    }
    emitResourceAttributes(resource) {
        for (const att of resource.attributes) {
            this.attributesEmitter.emit(att, this.attributesEmitter.needsResetEscape(att, resource.attributes), this.attributesEmitter.needsInputEscape(att, resource.attributes));
        }
    }
    emitInitializer(resource) {
        this.code.line();
        const comment = (0, sanitized_comments_1.sanitizedComment)(this.code);
        comment.line(`Create a new {@link ${resource.linkToDocs} ${resource.terraformResourceType}} ${resource.isDataSource ? "Data Source" : "Resource"}`);
        comment.line(``);
        comment.line(`@param scope The scope in which to define this construct`);
        comment.line(`@param id The scoped construct ID. Must be unique amongst siblings in the same scope`);
        comment.line(`@param options ${resource.configStruct.attributeType}`);
        comment.end();
        this.code.openBlock(`public constructor(scope: Construct, id: string, config: ${resource.configStruct.attributeType})`);
        resource.isProvider
            ? this.emitProviderSuper(resource)
            : this.emitResourceSuper(resource);
        // initialize config properties
        for (const att of resource.configStruct.assignableAttributes) {
            if (att.setterType._type === "stored_class") {
                this.code.line(`this.${att.storageName}.internalValue = config.${att.name};`);
            }
            else {
                this.code.line(`this.${att.storageName} = config.${att.name};`);
            }
        }
        this.code.closeBlock();
    }
    emitResourceSuper(resource) {
        this.code.open(`super(scope, id, {`);
        this.code.line(`terraformResourceType: '${resource.terraformResourceType}',`);
        this.emitTerraformGeneratorMetadata(resource);
        this.code.line(`provider: config.provider,`);
        this.code.line(`dependsOn: config.dependsOn,`);
        this.code.line(`count: config.count,`);
        this.code.line(`lifecycle: config.lifecycle,`);
        this.code.line(`provisioners: config.provisioners,`);
        this.code.line(`connection: config.connection,`);
        this.code.line(`forEach: config.forEach`);
        this.code.close(`});`);
    }
    emitProviderSuper(resource) {
        this.code.open(`super(scope, id, {`);
        this.code.line(`terraformResourceType: '${resource.terraformResourceType}',`);
        this.emitTerraformGeneratorMetadata(resource);
        this.code.line(`terraformProviderSource: '${resource.terraformProviderSource}'`);
        this.code.close(`});`);
    }
    emitTerraformGeneratorMetadata(resource) {
        this.code.open(`terraformGeneratorMetadata: {`);
        this.code.line(`providerName: '${resource.provider}'${resource.providerVersion || resource.providerVersionConstraint
            ? ","
            : ""}`);
        if (resource.providerVersion) {
            this.code.line(`providerVersion: '${resource.providerVersion}'${resource.providerVersionConstraint ? "," : ""}`);
        }
        if (resource.providerVersionConstraint) {
            this.code.line(`providerVersionConstraint: '${resource.providerVersionConstraint}'`);
        }
        this.code.close(`},`);
    }
}
exports.ResourceEmitter = ResourceEmitter;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVzb3VyY2UtZW1pdHRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInJlc291cmNlLWVtaXR0ZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBSUEsNkRBQXlEO0FBQ3pELDhEQUF5RDtBQUV6RCxNQUFhLGVBQWU7SUFHMUIsWUFBNkIsSUFBZTtRQUFmLFNBQUksR0FBSixJQUFJLENBQVc7UUFDMUMsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksc0NBQWlCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFTSxJQUFJLENBQUMsUUFBdUI7UUFDakMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixNQUFNLE9BQU8sR0FBRyxJQUFBLHFDQUFnQixFQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM1QyxPQUFPLENBQUMsSUFBSSxDQUNWLHVCQUF1QixRQUFRLENBQUMsVUFBVSxJQUFJLFFBQVEsQ0FBQyxxQkFBcUIsR0FBRyxDQUNoRixDQUFDO1FBQ0YsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2QsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLGdCQUFnQixRQUFRLENBQUMsU0FBUyxrQkFBa0IsUUFBUSxDQUFDLGVBQWUsRUFBRSxDQUMvRSxDQUFDO1FBRUYsSUFBSSxDQUFDLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUVwQyxJQUFJLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDbEMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRWpDLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDL0IsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUUvQixJQUFJLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQzlCLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUV0QyxZQUFZO1FBQ1osSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUM3QixJQUFJLENBQUMscUJBQXFCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxZQUFZO0lBQ3RDLENBQUM7SUFFTyxVQUFVLENBQUMsS0FBYTtRQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sS0FBSyxFQUFFLENBQUMsQ0FBQztRQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBRU8sb0JBQW9CLENBQUMsUUFBdUI7UUFDbEQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osNENBQTRDLFFBQVEsQ0FBQyxxQkFBcUIsSUFBSSxDQUMvRSxDQUFDO0lBQ0osQ0FBQztJQUVPLGlCQUFpQixDQUFDLFFBQXVCO1FBQy9DLE1BQU0sT0FBTyxHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVDLE9BQU8sQ0FBQyxJQUFJLENBQ1Ysd0NBQXdDLFFBQVEsQ0FBQyxTQUFTLGtEQUFrRCxDQUM3RyxDQUFDO1FBQ0YsT0FBTyxDQUFDLElBQUksQ0FBQywwREFBMEQsQ0FBQyxDQUFDO1FBQ3pFLE9BQU8sQ0FBQyxJQUFJLENBQ1YsMkVBQTJFLFFBQVEsQ0FBQyxTQUFTLFlBQVksQ0FDMUcsQ0FBQztRQUNGLE9BQU8sQ0FBQyxJQUFJLENBQ1YsOENBQThDLFFBQVEsQ0FBQyxTQUFTLGlEQUFpRCxRQUFRLENBQUMsVUFBVSxpRkFBaUYsQ0FDdE4sQ0FBQztRQUNGLE9BQU8sQ0FBQyxJQUFJLENBQ1YsZ0VBQWdFLFFBQVEsQ0FBQyxTQUFTLHFCQUFxQixDQUN4RyxDQUFDO1FBQ0YsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2QsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1o7MkZBQ3FGLFFBQVEsQ0FBQyxxQkFBcUI7UUFDakgsQ0FDSCxDQUFDO0lBQ0osQ0FBQztJQUVPLHdCQUF3QixDQUFDLFFBQXVCO1FBQ3RELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDakIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQ2pCLDhEQUE4RCxDQUMvRCxDQUFDO1FBQ0YsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUVsQyxLQUFLLE1BQU0sR0FBRyxJQUFJLFFBQVEsQ0FBQyx1QkFBdUIsRUFBRSxDQUFDO1lBQ25ELElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEQsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXRCLElBQUksUUFBUSxDQUFDLHVCQUF1QixDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUNoRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1lBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7WUFDakQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osNEhBQTRILENBQzdILENBQUM7UUFDSixDQUFDO2FBQU0sQ0FBQztZQUNOLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQ2xDLENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ3pCLENBQUM7SUFFTyxxQkFBcUIsQ0FBQyxRQUF1QjtRQUNuRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQiwyREFBMkQsQ0FDNUQsQ0FBQztRQUNGLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRTNCLEtBQUssTUFBTSxHQUFHLElBQUksUUFBUSxDQUFDLHVCQUF1QixFQUFFLENBQUM7WUFDbkQsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGVBQWUsQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDckQsQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUM7SUFDekIsQ0FBQztJQUVPLHNCQUFzQixDQUFDLFFBQXVCO1FBQ3BELEtBQUssTUFBTSxHQUFHLElBQUksUUFBUSxDQUFDLFVBQVUsRUFBRSxDQUFDO1lBQ3RDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLENBQ3pCLEdBQUcsRUFDSCxJQUFJLENBQUMsaUJBQWlCLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxVQUFVLENBQUMsRUFDakUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsVUFBVSxDQUFDLENBQ2xFLENBQUM7UUFDSixDQUFDO0lBQ0gsQ0FBQztJQUVPLGVBQWUsQ0FBQyxRQUF1QjtRQUM3QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2pCLE1BQU0sT0FBTyxHQUFHLElBQUEscUNBQWdCLEVBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzVDLE9BQU8sQ0FBQyxJQUFJLENBQ1YsdUJBQXVCLFFBQVEsQ0FBQyxVQUFVLElBQ3hDLFFBQVEsQ0FBQyxxQkFDWCxLQUFLLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsVUFBVSxFQUFFLENBQzFELENBQUM7UUFDRixPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLE9BQU8sQ0FBQyxJQUFJLENBQUMsMERBQTBELENBQUMsQ0FBQztRQUN6RSxPQUFPLENBQUMsSUFBSSxDQUNWLHNGQUFzRixDQUN2RixDQUFDO1FBQ0YsT0FBTyxDQUFDLElBQUksQ0FBQyxrQkFBa0IsUUFBUSxDQUFDLFlBQVksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxDQUFDO1FBQ3RFLE9BQU8sQ0FBQyxHQUFHLEVBQUUsQ0FBQztRQUNkLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUNqQiw0REFBNEQsUUFBUSxDQUFDLFlBQVksQ0FBQyxhQUFhLEdBQUcsQ0FDbkcsQ0FBQztRQUVGLFFBQVEsQ0FBQyxVQUFVO1lBQ2pCLENBQUMsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxDQUFDO1lBQ2xDLENBQUMsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFckMsK0JBQStCO1FBQy9CLEtBQUssTUFBTSxHQUFHLElBQUksUUFBUSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsRUFBRSxDQUFDO1lBQzdELElBQUksR0FBRyxDQUFDLFVBQVUsQ0FBQyxLQUFLLEtBQUssY0FBYyxFQUFFLENBQUM7Z0JBQzVDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLFFBQVEsR0FBRyxDQUFDLFdBQVcsMkJBQTJCLEdBQUcsQ0FBQyxJQUFJLEdBQUcsQ0FDOUQsQ0FBQztZQUNKLENBQUM7aUJBQU0sQ0FBQztnQkFDTixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEdBQUcsQ0FBQyxXQUFXLGFBQWEsR0FBRyxDQUFDLElBQUksR0FBRyxDQUFDLENBQUM7WUFDbEUsQ0FBQztRQUNILENBQUM7UUFFRCxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQ3pCLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxRQUF1QjtRQUMvQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLDJCQUEyQixRQUFRLENBQUMscUJBQXFCLElBQUksQ0FDOUQsQ0FBQztRQUNGLElBQUksQ0FBQyw4QkFBOEIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLDhCQUE4QixDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQy9DLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDckQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQzFDLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3pCLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxRQUF1QjtRQUMvQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUNaLDJCQUEyQixRQUFRLENBQUMscUJBQXFCLElBQUksQ0FDOUQsQ0FBQztRQUNGLElBQUksQ0FBQyw4QkFBOEIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWiw2QkFBNkIsUUFBUSxDQUFDLHVCQUF1QixHQUFHLENBQ2pFLENBQUM7UUFDRixJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUN6QixDQUFDO0lBRU8sOEJBQThCLENBQUMsUUFBdUI7UUFDNUQsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUNoRCxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWixrQkFBa0IsUUFBUSxDQUFDLFFBQVEsSUFDakMsUUFBUSxDQUFDLGVBQWUsSUFBSSxRQUFRLENBQUMseUJBQXlCO1lBQzVELENBQUMsQ0FBQyxHQUFHO1lBQ0wsQ0FBQyxDQUFDLEVBQ04sRUFBRSxDQUNILENBQUM7UUFFRixJQUFJLFFBQVEsQ0FBQyxlQUFlLEVBQUUsQ0FBQztZQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FDWixxQkFBcUIsUUFBUSxDQUFDLGVBQWUsSUFDM0MsUUFBUSxDQUFDLHlCQUF5QixDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQzdDLEVBQUUsQ0FDSCxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksUUFBUSxDQUFDLHlCQUF5QixFQUFFLENBQUM7WUFDdkMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQ1osK0JBQStCLFFBQVEsQ0FBQyx5QkFBeUIsR0FBRyxDQUNyRSxDQUFDO1FBQ0osQ0FBQztRQUVELElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3hCLENBQUM7Q0FDRjtBQXhORCwwQ0F3TkMiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgKGMpIEhhc2hpQ29ycCwgSW5jXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTVBMLTIuMFxuaW1wb3J0IHsgQ29kZU1ha2VyIH0gZnJvbSBcImNvZGVtYWtlclwiO1xuaW1wb3J0IHsgUmVzb3VyY2VNb2RlbCB9IGZyb20gXCIuLi9tb2RlbHNcIjtcbmltcG9ydCB7IEF0dHJpYnV0ZXNFbWl0dGVyIH0gZnJvbSBcIi4vYXR0cmlidXRlcy1lbWl0dGVyXCI7XG5pbXBvcnQgeyBzYW5pdGl6ZWRDb21tZW50IH0gZnJvbSBcIi4uL3Nhbml0aXplZC1jb21tZW50c1wiO1xuXG5leHBvcnQgY2xhc3MgUmVzb3VyY2VFbWl0dGVyIHtcbiAgYXR0cmlidXRlc0VtaXR0ZXI6IEF0dHJpYnV0ZXNFbWl0dGVyO1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgY29kZTogQ29kZU1ha2VyKSB7XG4gICAgdGhpcy5hdHRyaWJ1dGVzRW1pdHRlciA9IG5ldyBBdHRyaWJ1dGVzRW1pdHRlcih0aGlzLmNvZGUpO1xuICB9XG5cbiAgcHVibGljIGVtaXQocmVzb3VyY2U6IFJlc291cmNlTW9kZWwpIHtcbiAgICB0aGlzLmNvZGUubGluZSgpO1xuICAgIGNvbnN0IGNvbW1lbnQgPSBzYW5pdGl6ZWRDb21tZW50KHRoaXMuY29kZSk7XG4gICAgY29tbWVudC5saW5lKFxuICAgICAgYFJlcHJlc2VudHMgYSB7QGxpbmsgJHtyZXNvdXJjZS5saW5rVG9Eb2NzfSAke3Jlc291cmNlLnRlcnJhZm9ybVJlc291cmNlVHlwZX19YCxcbiAgICApO1xuICAgIGNvbW1lbnQuZW5kKCk7XG4gICAgdGhpcy5jb2RlLm9wZW5CbG9jayhcbiAgICAgIGBleHBvcnQgY2xhc3MgJHtyZXNvdXJjZS5jbGFzc05hbWV9IGV4dGVuZHMgY2RrdGYuJHtyZXNvdXJjZS5wYXJlbnRDbGFzc05hbWV9YCxcbiAgICApO1xuXG4gICAgdGhpcy5lbWl0SGVhZGVyKFwiU1RBVElDIFBST1BFUlRJRVNcIik7XG4gICAgdGhpcy5lbWl0U3RhdGljUHJvcGVydGllcyhyZXNvdXJjZSk7XG5cbiAgICB0aGlzLmVtaXRIZWFkZXIoXCJTVEFUSUMgTWV0aG9kc1wiKTtcbiAgICB0aGlzLmVtaXRTdGF0aWNNZXRob2RzKHJlc291cmNlKTtcblxuICAgIHRoaXMuZW1pdEhlYWRlcihcIklOSVRJQUxJWkVSXCIpO1xuICAgIHRoaXMuZW1pdEluaXRpYWxpemVyKHJlc291cmNlKTtcblxuICAgIHRoaXMuZW1pdEhlYWRlcihcIkFUVFJJQlVURVNcIik7XG4gICAgdGhpcy5lbWl0UmVzb3VyY2VBdHRyaWJ1dGVzKHJlc291cmNlKTtcblxuICAgIC8vIHN5bnRoZXNpc1xuICAgIHRoaXMuZW1pdEhlYWRlcihcIlNZTlRIRVNJU1wiKTtcbiAgICB0aGlzLmVtaXRSZXNvdXJjZVN5bnRoZXNpcyhyZXNvdXJjZSk7XG4gICAgdGhpcy5lbWl0SGNsUmVzb3VyY2VTeW50aGVzaXMocmVzb3VyY2UpO1xuXG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTsgLy8gY29uc3RydWN0XG4gIH1cblxuICBwcml2YXRlIGVtaXRIZWFkZXIodGl0bGU6IHN0cmluZykge1xuICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoXCIvLyBcIiArIFwiPVwiLnJlcGVhdCh0aXRsZS5sZW5ndGgpKTtcbiAgICB0aGlzLmNvZGUubGluZShgLy8gJHt0aXRsZX1gKTtcbiAgICB0aGlzLmNvZGUubGluZShcIi8vIFwiICsgXCI9XCIucmVwZWF0KHRpdGxlLmxlbmd0aCkpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0U3RhdGljUHJvcGVydGllcyhyZXNvdXJjZTogUmVzb3VyY2VNb2RlbCkge1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgdGZSZXNvdXJjZVR5cGUgPSBcIiR7cmVzb3VyY2UudGVycmFmb3JtUmVzb3VyY2VUeXBlfVwiO2AsXG4gICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgZW1pdFN0YXRpY01ldGhvZHMocmVzb3VyY2U6IFJlc291cmNlTW9kZWwpIHtcbiAgICBjb25zdCBjb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGNvbW1lbnQubGluZShcbiAgICAgIGBHZW5lcmF0ZXMgQ0RLVEYgY29kZSBmb3IgaW1wb3J0aW5nIGEgJHtyZXNvdXJjZS5jbGFzc05hbWV9IHJlc291cmNlIHVwb24gcnVubmluZyBcImNka3RmIHBsYW4gPHN0YWNrLW5hbWU+XCJgLFxuICAgICk7XG4gICAgY29tbWVudC5saW5lKGBAcGFyYW0gc2NvcGUgVGhlIHNjb3BlIGluIHdoaWNoIHRvIGRlZmluZSB0aGlzIGNvbnN0cnVjdGApO1xuICAgIGNvbW1lbnQubGluZShcbiAgICAgIGBAcGFyYW0gaW1wb3J0VG9JZCBUaGUgY29uc3RydWN0IGlkIHVzZWQgaW4gdGhlIGdlbmVyYXRlZCBjb25maWcgZm9yIHRoZSAke3Jlc291cmNlLmNsYXNzTmFtZX0gdG8gaW1wb3J0YCxcbiAgICApO1xuICAgIGNvbW1lbnQubGluZShcbiAgICAgIGBAcGFyYW0gaW1wb3J0RnJvbUlkIFRoZSBpZCBvZiB0aGUgZXhpc3RpbmcgJHtyZXNvdXJjZS5jbGFzc05hbWV9IHRoYXQgc2hvdWxkIGJlIGltcG9ydGVkLiBSZWZlciB0byB0aGUge0BsaW5rICR7cmVzb3VyY2UubGlua1RvRG9jc30jaW1wb3J0IGltcG9ydCBzZWN0aW9ufSBpbiB0aGUgZG9jdW1lbnRhdGlvbiBvZiB0aGlzIHJlc291cmNlIGZvciB0aGUgaWQgdG8gdXNlYCxcbiAgICApO1xuICAgIGNvbW1lbnQubGluZShcbiAgICAgIGBAcGFyYW0gcHJvdmlkZXI/IE9wdGlvbmFsIGluc3RhbmNlIG9mIHRoZSBwcm92aWRlciB3aGVyZSB0aGUgJHtyZXNvdXJjZS5jbGFzc05hbWV9IHRvIGltcG9ydCBpcyBmb3VuZGAsXG4gICAgKTtcbiAgICBjb21tZW50LmVuZCgpO1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYHB1YmxpYyBzdGF0aWMgZ2VuZXJhdGVDb25maWdGb3JJbXBvcnQoc2NvcGU6IENvbnN0cnVjdCwgaW1wb3J0VG9JZDogc3RyaW5nLCBpbXBvcnRGcm9tSWQ6IHN0cmluZywgcHJvdmlkZXI/OiBjZGt0Zi5UZXJyYWZvcm1Qcm92aWRlcikge1xuICAgICAgICByZXR1cm4gbmV3IGNka3RmLkltcG9ydGFibGVSZXNvdXJjZShzY29wZSwgaW1wb3J0VG9JZCwgeyB0ZXJyYWZvcm1SZXNvdXJjZVR5cGU6IFwiJHtyZXNvdXJjZS50ZXJyYWZvcm1SZXNvdXJjZVR5cGV9XCIsIGltcG9ydElkOiBpbXBvcnRGcm9tSWQsIHByb3ZpZGVyIH0pO1xuICAgICAgfWAsXG4gICAgKTtcbiAgfVxuXG4gIHByaXZhdGUgZW1pdEhjbFJlc291cmNlU3ludGhlc2lzKHJlc291cmNlOiBSZXNvdXJjZU1vZGVsKSB7XG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYHByb3RlY3RlZCBzeW50aGVzaXplSGNsQXR0cmlidXRlcygpOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfWAsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUub3BlbihgY29uc3QgYXR0cnMgPSB7YCk7XG5cbiAgICBmb3IgKGNvbnN0IGF0dCBvZiByZXNvdXJjZS5zeW50aGVzaXphYmxlQXR0cmlidXRlcykge1xuICAgICAgdGhpcy5hdHRyaWJ1dGVzRW1pdHRlci5lbWl0VG9IY2xUZXJyYWZvcm0oYXR0LCBmYWxzZSk7XG4gICAgfVxuXG4gICAgdGhpcy5jb2RlLmNsb3NlKGB9O2ApO1xuXG4gICAgaWYgKHJlc291cmNlLnN5bnRoZXNpemFibGVBdHRyaWJ1dGVzLmxlbmd0aCA+IDApIHtcbiAgICAgIHRoaXMuY29kZS5saW5lKCk7XG4gICAgICB0aGlzLmNvZGUubGluZShgLy8gcmVtb3ZlIHVuZGVmaW5lZCBhdHRyaWJ1dGVzYCk7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYHJldHVybiBPYmplY3QuZnJvbUVudHJpZXMoT2JqZWN0LmVudHJpZXMoYXR0cnMpLmZpbHRlcigoW18sIHZhbHVlXSkgPT4gdmFsdWUgIT09IHVuZGVmaW5lZCAmJiB2YWx1ZS52YWx1ZSAhPT0gdW5kZWZpbmVkICkpYCxcbiAgICAgICk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuY29kZS5saW5lKGByZXR1cm4gYXR0cnM7YCk7XG4gICAgfVxuXG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgfVxuXG4gIHByaXZhdGUgZW1pdFJlc291cmNlU3ludGhlc2lzKHJlc291cmNlOiBSZXNvdXJjZU1vZGVsKSB7XG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYHByb3RlY3RlZCBzeW50aGVzaXplQXR0cmlidXRlcygpOiB7IFtuYW1lOiBzdHJpbmddOiBhbnkgfWAsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUub3BlbihgcmV0dXJuIHtgKTtcblxuICAgIGZvciAoY29uc3QgYXR0IG9mIHJlc291cmNlLnN5bnRoZXNpemFibGVBdHRyaWJ1dGVzKSB7XG4gICAgICB0aGlzLmF0dHJpYnV0ZXNFbWl0dGVyLmVtaXRUb1RlcnJhZm9ybShhdHQsIGZhbHNlKTtcbiAgICB9XG5cbiAgICB0aGlzLmNvZGUuY2xvc2UoYH07YCk7XG4gICAgdGhpcy5jb2RlLmNsb3NlQmxvY2soKTtcbiAgfVxuXG4gIHByaXZhdGUgZW1pdFJlc291cmNlQXR0cmlidXRlcyhyZXNvdXJjZTogUmVzb3VyY2VNb2RlbCkge1xuICAgIGZvciAoY29uc3QgYXR0IG9mIHJlc291cmNlLmF0dHJpYnV0ZXMpIHtcbiAgICAgIHRoaXMuYXR0cmlidXRlc0VtaXR0ZXIuZW1pdChcbiAgICAgICAgYXR0LFxuICAgICAgICB0aGlzLmF0dHJpYnV0ZXNFbWl0dGVyLm5lZWRzUmVzZXRFc2NhcGUoYXR0LCByZXNvdXJjZS5hdHRyaWJ1dGVzKSxcbiAgICAgICAgdGhpcy5hdHRyaWJ1dGVzRW1pdHRlci5uZWVkc0lucHV0RXNjYXBlKGF0dCwgcmVzb3VyY2UuYXR0cmlidXRlcyksXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgZW1pdEluaXRpYWxpemVyKHJlc291cmNlOiBSZXNvdXJjZU1vZGVsKSB7XG4gICAgdGhpcy5jb2RlLmxpbmUoKTtcbiAgICBjb25zdCBjb21tZW50ID0gc2FuaXRpemVkQ29tbWVudCh0aGlzLmNvZGUpO1xuICAgIGNvbW1lbnQubGluZShcbiAgICAgIGBDcmVhdGUgYSBuZXcge0BsaW5rICR7cmVzb3VyY2UubGlua1RvRG9jc30gJHtcbiAgICAgICAgcmVzb3VyY2UudGVycmFmb3JtUmVzb3VyY2VUeXBlXG4gICAgICB9fSAke3Jlc291cmNlLmlzRGF0YVNvdXJjZSA/IFwiRGF0YSBTb3VyY2VcIiA6IFwiUmVzb3VyY2VcIn1gLFxuICAgICk7XG4gICAgY29tbWVudC5saW5lKGBgKTtcbiAgICBjb21tZW50LmxpbmUoYEBwYXJhbSBzY29wZSBUaGUgc2NvcGUgaW4gd2hpY2ggdG8gZGVmaW5lIHRoaXMgY29uc3RydWN0YCk7XG4gICAgY29tbWVudC5saW5lKFxuICAgICAgYEBwYXJhbSBpZCBUaGUgc2NvcGVkIGNvbnN0cnVjdCBJRC4gTXVzdCBiZSB1bmlxdWUgYW1vbmdzdCBzaWJsaW5ncyBpbiB0aGUgc2FtZSBzY29wZWAsXG4gICAgKTtcbiAgICBjb21tZW50LmxpbmUoYEBwYXJhbSBvcHRpb25zICR7cmVzb3VyY2UuY29uZmlnU3RydWN0LmF0dHJpYnV0ZVR5cGV9YCk7XG4gICAgY29tbWVudC5lbmQoKTtcbiAgICB0aGlzLmNvZGUub3BlbkJsb2NrKFxuICAgICAgYHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBjb25maWc6ICR7cmVzb3VyY2UuY29uZmlnU3RydWN0LmF0dHJpYnV0ZVR5cGV9KWAsXG4gICAgKTtcblxuICAgIHJlc291cmNlLmlzUHJvdmlkZXJcbiAgICAgID8gdGhpcy5lbWl0UHJvdmlkZXJTdXBlcihyZXNvdXJjZSlcbiAgICAgIDogdGhpcy5lbWl0UmVzb3VyY2VTdXBlcihyZXNvdXJjZSk7XG5cbiAgICAvLyBpbml0aWFsaXplIGNvbmZpZyBwcm9wZXJ0aWVzXG4gICAgZm9yIChjb25zdCBhdHQgb2YgcmVzb3VyY2UuY29uZmlnU3RydWN0LmFzc2lnbmFibGVBdHRyaWJ1dGVzKSB7XG4gICAgICBpZiAoYXR0LnNldHRlclR5cGUuX3R5cGUgPT09IFwic3RvcmVkX2NsYXNzXCIpIHtcbiAgICAgICAgdGhpcy5jb2RlLmxpbmUoXG4gICAgICAgICAgYHRoaXMuJHthdHQuc3RvcmFnZU5hbWV9LmludGVybmFsVmFsdWUgPSBjb25maWcuJHthdHQubmFtZX07YCxcbiAgICAgICAgKTtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIHRoaXMuY29kZS5saW5lKGB0aGlzLiR7YXR0LnN0b3JhZ2VOYW1lfSA9IGNvbmZpZy4ke2F0dC5uYW1lfTtgKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aGlzLmNvZGUuY2xvc2VCbG9jaygpO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0UmVzb3VyY2VTdXBlcihyZXNvdXJjZTogUmVzb3VyY2VNb2RlbCkge1xuICAgIHRoaXMuY29kZS5vcGVuKGBzdXBlcihzY29wZSwgaWQsIHtgKTtcbiAgICB0aGlzLmNvZGUubGluZShcbiAgICAgIGB0ZXJyYWZvcm1SZXNvdXJjZVR5cGU6ICcke3Jlc291cmNlLnRlcnJhZm9ybVJlc291cmNlVHlwZX0nLGAsXG4gICAgKTtcbiAgICB0aGlzLmVtaXRUZXJyYWZvcm1HZW5lcmF0b3JNZXRhZGF0YShyZXNvdXJjZSk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYHByb3ZpZGVyOiBjb25maWcucHJvdmlkZXIsYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYGRlcGVuZHNPbjogY29uZmlnLmRlcGVuZHNPbixgKTtcbiAgICB0aGlzLmNvZGUubGluZShgY291bnQ6IGNvbmZpZy5jb3VudCxgKTtcbiAgICB0aGlzLmNvZGUubGluZShgbGlmZWN5Y2xlOiBjb25maWcubGlmZWN5Y2xlLGApO1xuICAgIHRoaXMuY29kZS5saW5lKGBwcm92aXNpb25lcnM6IGNvbmZpZy5wcm92aXNpb25lcnMsYCk7XG4gICAgdGhpcy5jb2RlLmxpbmUoYGNvbm5lY3Rpb246IGNvbmZpZy5jb25uZWN0aW9uLGApO1xuICAgIHRoaXMuY29kZS5saW5lKGBmb3JFYWNoOiBjb25maWcuZm9yRWFjaGApO1xuICAgIHRoaXMuY29kZS5jbG9zZShgfSk7YCk7XG4gIH1cblxuICBwcml2YXRlIGVtaXRQcm92aWRlclN1cGVyKHJlc291cmNlOiBSZXNvdXJjZU1vZGVsKSB7XG4gICAgdGhpcy5jb2RlLm9wZW4oYHN1cGVyKHNjb3BlLCBpZCwge2ApO1xuICAgIHRoaXMuY29kZS5saW5lKFxuICAgICAgYHRlcnJhZm9ybVJlc291cmNlVHlwZTogJyR7cmVzb3VyY2UudGVycmFmb3JtUmVzb3VyY2VUeXBlfScsYCxcbiAgICApO1xuICAgIHRoaXMuZW1pdFRlcnJhZm9ybUdlbmVyYXRvck1ldGFkYXRhKHJlc291cmNlKTtcbiAgICB0aGlzLmNvZGUubGluZShcbiAgICAgIGB0ZXJyYWZvcm1Qcm92aWRlclNvdXJjZTogJyR7cmVzb3VyY2UudGVycmFmb3JtUHJvdmlkZXJTb3VyY2V9J2AsXG4gICAgKTtcbiAgICB0aGlzLmNvZGUuY2xvc2UoYH0pO2ApO1xuICB9XG5cbiAgcHJpdmF0ZSBlbWl0VGVycmFmb3JtR2VuZXJhdG9yTWV0YWRhdGEocmVzb3VyY2U6IFJlc291cmNlTW9kZWwpIHtcbiAgICB0aGlzLmNvZGUub3BlbihgdGVycmFmb3JtR2VuZXJhdG9yTWV0YWRhdGE6IHtgKTtcbiAgICB0aGlzLmNvZGUubGluZShcbiAgICAgIGBwcm92aWRlck5hbWU6ICcke3Jlc291cmNlLnByb3ZpZGVyfScke1xuICAgICAgICByZXNvdXJjZS5wcm92aWRlclZlcnNpb24gfHwgcmVzb3VyY2UucHJvdmlkZXJWZXJzaW9uQ29uc3RyYWludFxuICAgICAgICAgID8gXCIsXCJcbiAgICAgICAgICA6IFwiXCJcbiAgICAgIH1gLFxuICAgICk7XG5cbiAgICBpZiAocmVzb3VyY2UucHJvdmlkZXJWZXJzaW9uKSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYHByb3ZpZGVyVmVyc2lvbjogJyR7cmVzb3VyY2UucHJvdmlkZXJWZXJzaW9ufScke1xuICAgICAgICAgIHJlc291cmNlLnByb3ZpZGVyVmVyc2lvbkNvbnN0cmFpbnQgPyBcIixcIiA6IFwiXCJcbiAgICAgICAgfWAsXG4gICAgICApO1xuICAgIH1cblxuICAgIGlmIChyZXNvdXJjZS5wcm92aWRlclZlcnNpb25Db25zdHJhaW50KSB7XG4gICAgICB0aGlzLmNvZGUubGluZShcbiAgICAgICAgYHByb3ZpZGVyVmVyc2lvbkNvbnN0cmFpbnQ6ICcke3Jlc291cmNlLnByb3ZpZGVyVmVyc2lvbkNvbnN0cmFpbnR9J2AsXG4gICAgICApO1xuICAgIH1cblxuICAgIHRoaXMuY29kZS5jbG9zZShgfSxgKTtcbiAgfVxufVxuIl19