"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AttributesEmitter = void 0;
const models_1 = require("../models");
const custom_defaults_1 = require("../custom-defaults");
function titleCase(value) {
    return value[0].toUpperCase() + value.slice(1);
}
class AttributesEmitter {
    constructor(code) {
        this.code = code;
    }
    emit(att, escapeReset, escapeInput) {
        this.code.line();
        this.code.line(`// ${att.terraformName} - computed: ${att.computed}, optional: ${att.isOptional}, required: ${att.isRequired}`);
        const isStored = att.isStored;
        const hasResetMethod = isStored && !att.isRequired && att.setterType._type !== "none";
        const hasInputMethod = isStored && att.setterType._type !== "none";
        const getterType = att.getterType;
        if (getterType._type === "stored_class") {
            this.code.line(`private ${att.storageName} = ${this.storedClassInit(att)};`);
        }
        else if (isStored) {
            this.code.line(`private ${att.storageName}?: ${att.type.inputTypeDefinition}; `);
        }
        switch (getterType._type) {
            case "plain":
                this.code.openBlock(`public get ${att.name}()`);
                this.code.line(`return ${this.determineGetAttCall(att)};`);
                this.code.closeBlock();
                break;
            case "args":
                this.code.openBlock(`public ${att.name}(${getterType.args})${getterType.returnType ? ": " + getterType.returnType : ""}`);
                this.code.line(`return ${getterType.returnStatement};`);
                this.code.closeBlock();
                break;
            case "stored_class":
                this.code.openBlock(`public get ${att.name}()`);
                this.code.line(`return this.${att.storageName};`);
                this.code.closeBlock();
                break;
        }
        const setterType = att.setterType;
        switch (setterType._type) {
            case "set":
                this.code.openBlock(`public set ${att.name}(value: ${setterType.type})`);
                this.code.line(`this.${att.storageName} = value;`);
                this.code.closeBlock();
                break;
            case "put":
                this.code.openBlock(`public put${titleCase(att.name)}(value: ${setterType.type})`);
                this.code.line(`this.${att.storageName} = value;`);
                this.code.closeBlock();
                break;
            case "stored_class":
                this.code.openBlock(`public put${titleCase(att.name)}(value: ${setterType.type})`);
                this.code.line(`this.${att.storageName}.internalValue = value;`);
                this.code.closeBlock();
                break;
        }
        if (hasResetMethod) {
            this.code.openBlock(`public ${this.getResetName(att.name, escapeReset)}()`);
            if (setterType._type === "stored_class") {
                this.code.line(`this.${att.storageName}.internalValue = undefined;`);
            }
            else {
                this.code.line(`this.${att.storageName} = undefined;`);
            }
            this.code.closeBlock();
        }
        if (hasInputMethod) {
            this.code.line(`// Temporarily expose input value. Use with caution.`);
            this.code.openBlock(`public get ${this.getInputName(att, escapeInput)}()`);
            if (setterType._type === "stored_class") {
                this.code.line(`return this.${att.storageName}.internalValue;`);
            }
            else {
                this.code.line(`return this.${att.storageName};`);
            }
            this.code.closeBlock();
        }
    }
    // returns an invocation of a stored class, e.g. 'new DeplotmentMetadataOutputReference(this, "metadata")'
    storedClassInit(att) {
        return att.type.getStoredClassInitializer(att.terraformName);
    }
    determineGetAttCall(att) {
        if (att.isProvider) {
            return `this.${att.storageName}`;
        }
        return att.type.getAttributeAccessFunction(att.terraformName);
    }
    needsInputEscape(att, attributes) {
        return (attributes.find((a) => a.terraformName.match(`^${att.terraformName}_input$`)) instanceof models_1.AttributeModel);
    }
    getInputName(att, escape) {
        if (escape) {
            return `${att.name}TfInput`;
        }
        else {
            return `${att.name}Input`;
        }
    }
    needsResetEscape(att, attributes) {
        return (attributes.find((a) => a.terraformName.match(`^reset_${att.terraformName}$`)) instanceof models_1.AttributeModel);
    }
    getResetName(name, escape) {
        if (!name)
            return name;
        if (escape) {
            return `resetTf${titleCase(name)}`;
        }
        else {
            return `reset${titleCase(name)}`;
        }
    }
    emitToHclTerraform(att, isStruct) {
        var _a;
        const type = att.type;
        const context = isStruct ? "struct!" : "this";
        const name = isStruct ? att.name : att.storageName;
        const customDefault = custom_defaults_1.CUSTOM_DEFAULTS[att.terraformFullName];
        const varReference = `${context}.${name}${!isStruct &&
            type.isComplex &&
            !att.isProvider &&
            (((_a = type.struct) === null || _a === void 0 ? void 0 : _a.isClass) || att.getterType._type === "stored_class")
            ? ".internalValue"
            : ""}`;
        const defaultCheck = customDefault !== undefined
            ? `${varReference} === undefined ? ${customDefault} : `
            : "";
        const value = `${defaultCheck}${type.toHclTerraformFunction}(${varReference})`;
        const isBlock = att.type.isComplex;
        const tt = att.type.typeModelType;
        this.code.open(`${att.terraformName}: {`);
        this.code.line(`value: ${value},`);
        this.code.line(`isBlock: ${isBlock},`);
        this.code.line(`type: "${tt}",`);
        this.code.line(`storageClassType: "${att.type.storedClassType}",`);
        this.code.close("},");
    }
    emitToTerraform(att, isStruct) {
        var _a;
        const type = att.type;
        const context = isStruct ? "struct!" : "this";
        const name = isStruct ? att.name : att.storageName;
        const customDefault = custom_defaults_1.CUSTOM_DEFAULTS[att.terraformFullName];
        const varReference = `${context}.${name}${!isStruct &&
            type.isComplex &&
            !att.isProvider &&
            (((_a = type.struct) === null || _a === void 0 ? void 0 : _a.isClass) || att.getterType._type === "stored_class")
            ? ".internalValue"
            : ""}`;
        const defaultCheck = customDefault !== undefined
            ? `${varReference} === undefined ? ${customDefault} : `
            : "";
        this.code.line(`${att.terraformName}: ${defaultCheck}${type.toTerraformFunction}(${varReference}),`);
    }
}
exports.AttributesEmitter = AttributesEmitter;
//# sourceMappingURL=data:application/json;base64,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