"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDesiredType = exports.isMapAttribute = exports.getTypeAtPath = void 0;
const provider_1 = require("./provider");
function getResourceAtPath(schema, path) {
    var _a;
    const parts = path.split(".");
    if (parts.length < 2) {
        // Too short to be a valid path
        return null;
    }
    const isDataSource = parts[0] === "data";
    if (isDataSource) {
        parts.shift();
    }
    const providerName = parts.shift();
    const resourceName = parts.shift();
    const fullProviderName = (0, provider_1.getFullProviderName)(schema, providerName);
    // Hack: In the case of 'external', the name of the data source 'external' doesn't have a prefix
    // so we repeat the name as both provider prefix and the data source name
    const fullResourceName = providerName === resourceName
        ? providerName
        : `${providerName}_${resourceName}`;
    if (!fullProviderName) {
        // No provider found with that name
        return null;
    }
    const provider = (_a = schema.provider_schemas) === null || _a === void 0 ? void 0 : _a[fullProviderName];
    if (!provider) {
        // Could not find provider
        return null;
    }
    if (resourceName.endsWith("Provider")) {
        // This is a provider
        return { resource: provider.provider, parts };
    }
    const resources = isDataSource
        ? provider.data_source_schemas
        : provider.resource_schemas;
    const resource = resources[fullResourceName];
    if (!resource) {
        // Could not find resource
        return null;
    }
    return { resource, parts };
}
function findType(item, parts) {
    if (!parts.length) {
        return item;
    }
    const [currentPart, ...otherParts] = parts;
    // Promitive attributes can be returned directly
    if (typeof item === "string") {
        return item;
    }
    // Complex attributes
    if (Array.isArray(item)) {
        if (item[0] === "set" || item[0] === "list") {
            if (currentPart === "[]") {
                return findType(item[1], otherParts);
            }
            // Trying to access a property on a list
            return null;
        }
        if (item[0] === "object") {
            if (currentPart === "[]") {
                // Trying to access a property on a list
                return null;
            }
            return findType(item[1][currentPart], otherParts);
        }
        if (item[0] === "map") {
            // We don't care what the key is
            return findType(item[1], otherParts);
        }
        // Unknown type
        return null;
    }
    // If we are not in an attribute, we can ignore the [] (since max_item=1 parts don't have [], so we just ignore it)
    if (currentPart === "[]") {
        return findType(item, otherParts);
    }
    // Block
    if (item.block) {
        // Find block type in block
        if (item.block.block_types) {
            const blockType = item.block.block_types[currentPart];
            if (blockType) {
                return findType(blockType, otherParts);
            }
        }
        // Find attribute in block
        if (item.block.attributes) {
            const attribute = item.block.attributes[currentPart];
            if (attribute && attribute.type) {
                return findType(attribute.type, otherParts);
            }
        }
    }
    // Could not find the type
    return null;
}
function getTypeAtPath(schema, path) {
    const resourceSchema = getResourceAtPath(schema, path);
    if (!resourceSchema) {
        return null;
    }
    const { resource, parts } = resourceSchema;
    return findType(resource, parts);
}
exports.getTypeAtPath = getTypeAtPath;
const isMapAttribute = (attribute) => (Array.isArray(attribute) ? attribute[0] === "map" : false);
exports.isMapAttribute = isMapAttribute;
function getDesiredType(scope, path) {
    const attributeOrBlockType = getTypeAtPath(scope.providerSchema, path);
    // Attribute type is not defined
    if (!attributeOrBlockType) {
        return "dynamic";
    }
    // Primitive attribute type
    if (typeof attributeOrBlockType === "string") {
        return attributeOrBlockType;
    }
    // Complex attribute type
    if (Array.isArray(attributeOrBlockType)) {
        return attributeOrBlockType;
    }
    // Schema
    if ("version" in attributeOrBlockType) {
        return "dynamic";
    }
    // Block type
    return "dynamic";
}
exports.getDesiredType = getDesiredType;
//# sourceMappingURL=data:application/json;base64,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