"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const partialCode_1 = require("../partialCode");
const getScope = () => ({
    providerSchema: {
        format_version: "0.1",
        provider_schemas: {
            aws: {
                data_source_schemas: {},
                provider: {
                    block: {
                        attributes: {},
                        block_types: {},
                    },
                    version: 0,
                },
                resource_schemas: {
                    aws_image: {
                        version: 0,
                        block: {
                            block_types: {
                                c: {
                                    nesting_mode: "single",
                                    block: {
                                        attributes: {
                                            d: {
                                                type: "string",
                                                description_kind: "plain",
                                                optional: true,
                                                required: true,
                                            },
                                        },
                                        block_types: {},
                                    },
                                },
                                d: {
                                    nesting_mode: "list",
                                    block: {
                                        attributes: {
                                            e: {
                                                type: "string",
                                                description_kind: "plain",
                                                optional: true,
                                                required: true,
                                            },
                                        },
                                        block_types: {},
                                    },
                                    min_items: 1,
                                },
                                timeouts: {
                                    nesting_mode: "single",
                                    block: {
                                        attributes: {
                                            read: {
                                                type: "string",
                                                description_kind: "plain",
                                                optional: true,
                                            },
                                        },
                                        block_types: {},
                                        description_kind: "plain",
                                    },
                                },
                            },
                            attributes: {
                                a: {
                                    type: "string",
                                    required: true,
                                },
                                b: {
                                    type: "string",
                                    required: true,
                                },
                            },
                        },
                    },
                },
            },
        },
    },
    constructs: new Set(),
    hasTokenBasedTypeCoercion: false,
    nodeIds: [],
    providerGenerator: {},
    variables: {},
    importables: [],
    // This should be mutated
    topLevelConfig: {},
});
describe("fillWithConfigAccessors", () => {
    it("adds a missing required attributes", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            c: {
                d: "d",
            },
            timeouts: {
                read: "20m",
            },
            d: [{ e: "e" }],
        }, "aws.image");
        expect(scope.topLevelConfig.b).toEqual("aws.image.b");
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": {
          "computed": false,
          "object": {
            "name": "config",
            "type": "Identifier",
          },
          "optional": null,
          "property": {
            "name": "b",
            "type": "Identifier",
          },
          "type": "MemberExpression",
        },
        "c": {
          "d": "d",
        },
        "d": [
          {
            "e": "e",
          },
        ],
        "timeouts": {
          "read": "20m",
        },
      }
    `);
    });
    it("ignores missing unrequired set block types", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            c: {
                d: "d",
            },
            d: [{ e: "e" }],
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({});
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "d": "d",
        },
        "d": [
          {
            "e": "e",
          },
        ],
      }
    `);
    });
    it("fills within a single block type", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            c: {},
            d: [{ e: "e" }],
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({
            d: "aws.image.c.d",
        });
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "d": {
            "computed": false,
            "object": {
              "name": "config",
              "type": "Identifier",
            },
            "optional": null,
            "property": {
              "name": "d",
              "type": "Identifier",
            },
            "type": "MemberExpression",
          },
        },
        "d": [
          {
            "e": "e",
          },
        ],
      }
    `);
    });
    it("fills an entire single block type", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            d: [{ e: "e" }],
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({
            c: "aws.image.c",
        });
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "computed": false,
          "object": {
            "name": "config",
            "type": "Identifier",
          },
          "optional": null,
          "property": {
            "name": "c",
            "type": "Identifier",
          },
          "type": "MemberExpression",
        },
        "d": [
          {
            "e": "e",
          },
        ],
      }
    `);
    });
    it("fills within a list block type", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            c: { d: "d" },
            d: [{}],
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({
            e: "aws.image.d.[].e",
        });
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "d": "d",
        },
        "d": [
          {
            "e": {
              "computed": false,
              "object": {
                "name": "config",
                "type": "Identifier",
              },
              "optional": null,
              "property": {
                "name": "e",
                "type": "Identifier",
              },
              "type": "MemberExpression",
            },
          },
        ],
      }
    `);
    });
    it("fills an entire list block type", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            c: { d: "d" },
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({
            d: "aws.image.d",
        });
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "d": "d",
        },
        "d": {
          "computed": false,
          "object": {
            "name": "config",
            "type": "Identifier",
          },
          "optional": null,
          "property": {
            "name": "d",
            "type": "Identifier",
          },
          "type": "MemberExpression",
        },
      }
    `);
    });
    it("can handle multiple fills with the same name", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            c: {},
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({
            d: "aws.image.c.d",
            d1: "aws.image.d",
        });
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "d": {
            "computed": false,
            "object": {
              "name": "config",
              "type": "Identifier",
            },
            "optional": null,
            "property": {
              "name": "d",
              "type": "Identifier",
            },
            "type": "MemberExpression",
          },
        },
        "d": {
          "computed": false,
          "object": {
            "name": "config",
            "type": "Identifier",
          },
          "optional": null,
          "property": {
            "name": "d1",
            "type": "Identifier",
          },
          "type": "MemberExpression",
        },
      }
    `);
    });
    it("fills lists with min_items > 0", () => {
        const scope = getScope();
        const result = (0, partialCode_1.fillWithConfigAccessors)(scope, {
            a: "a",
            b: "b",
            c: { d: "d" },
            d: [],
        }, "aws.image");
        expect(scope.topLevelConfig).toEqual({
            d: "aws.image.d",
        });
        expect(result).toMatchInlineSnapshot(`
      {
        "a": "a",
        "b": "b",
        "c": {
          "d": "d",
        },
        "d": {
          "computed": false,
          "object": {
            "name": "config",
            "type": "Identifier",
          },
          "optional": null,
          "property": {
            "name": "d",
            "type": "Identifier",
          },
          "type": "MemberExpression",
        },
      }
    `);
    });
});
//# sourceMappingURL=data:application/json;base64,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