"use strict";
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
Object.defineProperty(exports, "__esModule", { value: true });
const jsii_rosetta_workarounds_1 = require("../jsii-rosetta-workarounds");
describe("jsii-rosetta-workarounds", () => {
    describe("replacePythonImports", () => {
        it("normal imports", () => {
            const code = `
import constructs as constructs
# Provider bindings are generated by running cdktf get.
# See https://cdk.tf/provider-generation for more details.
import ...gen.providers.scaleway as scaleway
class MyConvertedCode(constructs.Construct):
    def __init__(self, scope, name):
        super().__init__(scope, name)
        scaleway.provider.ScalewayProvider(self, "scaleway",
            region="fr-par",
            zone="fr-par-1"
        )`;
            expect((0, jsii_rosetta_workarounds_1.replacePythonImports)(code)).toEqual(expect.stringContaining("import imports.scaleway as scaleway"));
        });
        it("granular imports", () => {
            const code = `from ...gen.providers.aws.lib.rdsCluster import RdsCluster`;
            expect((0, jsii_rosetta_workarounds_1.replacePythonImports)(code)).toEqual("from imports.aws.rdsCluster import RdsCluster");
            const withoutLibCode = `from ...gen.providers.azurerm.resource_group import ResourceGroup`;
            expect((0, jsii_rosetta_workarounds_1.replacePythonImports)(withoutLibCode)).toEqual("from imports.azurerm.resource_group import ResourceGroup");
        });
        it("resource imports", () => {
            const code = `import ...gen.providers.aws.lib.rdsCluster as RdsCluster`;
            expect((0, jsii_rosetta_workarounds_1.replacePythonImports)(code)).toEqual("import imports.aws.rdsCluster as RdsCluster");
        });
        it("fixes module imports", () => {
            const code = `import ...gen.modules.hello_module as HelloModule`;
            expect((0, jsii_rosetta_workarounds_1.replacePythonImports)(code)).toEqual("import imports.hello_module as HelloModule");
        });
    });
    describe("replaceCsharpImports", () => {
        it("normal imports", () => {
            const code = `using Gen.Providers.Aws.Lib.RdsCluster;`;
            expect((0, jsii_rosetta_workarounds_1.replaceCsharpImports)(code)).toEqual("using aws.RdsCluster;");
        });
        it("fixes module imports", () => {
            const code = `using Gen.Modules.HelloModule`;
            expect((0, jsii_rosetta_workarounds_1.replaceCsharpImports)(code)).toEqual("using HelloModule");
            const anotherCode = `using Gen.Modules.Hello.Module`;
            expect((0, jsii_rosetta_workarounds_1.replaceCsharpImports)(anotherCode)).toEqual("using Hello.Module");
        });
    });
    describe("replaceJavaImports", () => {
        it("normal imports", () => {
            const code = `import gen.providers.scaleway.objectResource.*;`;
            expect((0, jsii_rosetta_workarounds_1.replaceJavaImports)(code)).toEqual("import imports.scaleway.objectResource.*;");
        });
        it("precise imports", () => {
            const code = `import gen.providers.aws.lib.s3Bucket.S3Bucket;`;
            expect((0, jsii_rosetta_workarounds_1.replaceJavaImports)(code)).toEqual("import imports.aws.s3Bucket.S3Bucket;");
            const withoutLibImport = `import gen.providers.aws.s3Bucket.S3Bucket;`;
            expect((0, jsii_rosetta_workarounds_1.replaceJavaImports)(withoutLibImport)).toEqual("import imports.aws.s3Bucket.S3Bucket;");
        });
        it("fixes module imports", () => {
            const code = `import gen.modules.hello.module.*;`;
            expect((0, jsii_rosetta_workarounds_1.replaceJavaImports)(code)).toEqual("import imports.hello.module.*;");
            const anotherCode = `import gen.modules.helloModule.*;`;
            expect((0, jsii_rosetta_workarounds_1.replaceJavaImports)(anotherCode)).toEqual("import imports.helloModule.*;");
        });
    });
    describe("replaceGoImports", () => {
        it("normal imports", () => {
            const code = `import "github.com/aws-samples/dummy/gen/providers/scaleway/objectBucket"`;
            expect((0, jsii_rosetta_workarounds_1.replaceGoImports)(code)).toEqual(`import "cdk.tf/go/stack/generated/scaleway/objectBucket"`);
        });
        it("precise imports", () => {
            const code = `import "github.com/aws-samples/dummy/gen/providers/aws/lib/dbInstance"`;
            expect((0, jsii_rosetta_workarounds_1.replaceGoImports)(code)).toEqual(`import "cdk.tf/go/stack/generated/aws/dbInstance"`);
        });
        it("fixes module imports", () => {
            const code = `import helloModule "github.com/aws-samples/dummy/gen/modules/hello_module"`;
            expect((0, jsii_rosetta_workarounds_1.replaceGoImports)(code)).toEqual(
            // "github.com/hashicorp/terraform-cdk/examples/go/documentation/generated/terraform-aws-modules/aws/vpc"
            `import helloModule "cdk.tf/go/stack/generated/hello_module"`);
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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