"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const coerceType_1 = require("../coerceType");
const template_1 = __importDefault(require("@babel/template"));
const generator_1 = __importDefault(require("@babel/generator"));
function expressionify(code) {
    return template_1.default.ast(code).expression;
}
function generateCode(ast) {
    return (0, generator_1.default)(ast).code;
}
const scope = {
    importables: [],
    variables: {
        a_user_defined_variable: {
            resource: "var",
            variableName: "aUserDefinedVariable",
        },
        a_user_defined_local: {
            resource: "var",
            variableName: "aUserDefinedTerraformLocal",
        },
    },
};
describe("coerceType", () => {
    it.each([
        { code: `"hello-world"`, type: "string", targetType: undefined },
        { code: `"hello-world"`, type: "string", targetType: "string" },
    ])("should change nothing for %p ", ({ code, type, targetType }) => {
        expect(generateCode((0, coerceType_1.coerceType)(scope, expressionify(code), type, targetType))).toEqual(code);
    });
    it.each([
        {
            code: `"300"`,
            type: "string",
            targetType: "number",
            expectedCode: `Token.asNumber("300")`,
        },
        {
            code: `300`,
            type: "number",
            targetType: "string",
            expectedCode: `Token.asString(300)`,
        },
        {
            code: `"['foo', 'bar']"`,
            type: "string",
            targetType: ["list", "string"],
            expectedCode: `Token.asList("['foo', 'bar']")`,
        },
        {
            code: `"[1,2,3]"`,
            type: "string",
            targetType: ["list", "number"],
            expectedCode: `Token.asNumberList("[1,2,3]")`,
        },
        {
            code: `"['foo', 'bar']"`,
            type: "string",
            targetType: ["set", "string"],
            expectedCode: `Token.asList("['foo', 'bar']")`,
        },
        {
            code: `"[1,2,3]"`,
            type: "string",
            targetType: ["set", "number"],
            expectedCode: `Token.asNumberList("[1,2,3]")`,
        },
        // We don't have a token function for boolean lists, so we need to take an asAny
        {
            code: `"[true,false,true]"`,
            type: "string",
            targetType: ["list", "bool"],
            expectedCode: `Token.asAny("[true,false,true]")`,
        },
        {
            code: `"{ foo: 'bar' }"`,
            type: "string",
            targetType: ["map", "string"],
            expectedCode: `Token.asStringMap("{ foo: 'bar' }")`,
        },
        {
            code: `"{ foo: 1 }"`,
            type: "string",
            targetType: ["map", "number"],
            expectedCode: `Token.asNumberMap("{ foo: 1 }")`,
        },
        {
            code: `"{ foo: true }"`,
            type: "string",
            targetType: ["map", "bool"],
            expectedCode: `Token.asBooleanMap("{ foo: true }")`,
        },
        {
            code: `"{ foo: {bar: true} }"`,
            type: "string",
            targetType: ["map", ["map", "boolean"]],
            expectedCode: `Token.asAnyMap("{ foo: {bar: true} }")`,
        },
    ])("should use Token functions %p", ({ code, type, targetType, expectedCode }) => {
        expect(generateCode((0, coerceType_1.coerceType)(scope, expressionify(code), type, targetType))).toEqual(expectedCode);
    });
    it.each([
        {
            code: `aUserDefinedVariable.value`,
            type: "dynamic",
            targetType: "string",
            expectedCode: `aUserDefinedVariable.stringValue`,
        },
        {
            code: `aUserDefinedVariable.value`,
            type: "dynamic",
            targetType: "number",
            expectedCode: `aUserDefinedVariable.numberValue`,
        },
        {
            code: `aUserDefinedVariable.value`,
            type: "dynamic",
            targetType: ["list", "string"],
            expectedCode: `aUserDefinedVariable.listValue`,
        },
        {
            code: `aUserDefinedVariable.value`,
            type: "dynamic",
            targetType: "bool",
            expectedCode: `aUserDefinedVariable.booleanValue`,
        },
    ])("should use TerraformVariables accessors %p", ({ code, type, targetType, expectedCode }) => {
        expect(generateCode((0, coerceType_1.coerceType)(scope, expressionify(code), type, targetType))).toEqual(expectedCode);
    });
    it.each([
        {
            code: `aUserDefinedTerraformLocal.value`,
            type: "dynamic",
            targetType: "string",
            expectedCode: `aUserDefinedTerraformLocal.stringValue`,
        },
        {
            code: `aUserDefinedTerraformLocal.value`,
            type: "dynamic",
            targetType: "number",
            expectedCode: `aUserDefinedTerraformLocal.numberValue`,
        },
        {
            code: `aUserDefinedTerraformLocal.value`,
            type: "dynamic",
            targetType: ["list", "string"],
            expectedCode: `aUserDefinedTerraformLocal.listValue`,
        },
        {
            code: `aUserDefinedTerraformLocal.value`,
            type: "dynamic",
            targetType: "bool",
            expectedCode: `aUserDefinedTerraformLocal.booleanValue`,
        },
    ])("should use TerraformLocal accessors %p", ({ code, type, targetType, expectedCode }) => {
        expect(generateCode((0, coerceType_1.coerceType)(scope, expressionify(code), type, targetType))).toEqual(expectedCode);
    });
});
//# sourceMappingURL=data:application/json;base64,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