"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getGradlePackageVersionFromBuild = exports.getGradlePackageVersion = exports.getDependencyInformationFromLine = exports.getGradleDependencies = exports.isGradleProject = void 0;
const path_1 = __importDefault(require("path"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const logging_1 = require("./logging");
const util_1 = require("./util");
function isGradleProject(workingDirectory) {
    const buildGradlePath = path_1.default.join(workingDirectory, "build.gradle");
    try {
        fs_extra_1.default.accessSync(buildGradlePath, fs_extra_1.default.constants.R_OK | fs_extra_1.default.constants.W_OK);
        return true;
    }
    catch (_a) {
        logging_1.logger.debug(`No build.gradle found at ${buildGradlePath}`);
        return false;
    }
}
exports.isGradleProject = isGradleProject;
async function getGradleDependencies() {
    let output;
    try {
        output = await (0, util_1.exec)("gradle", ["dependencies", "--console=plain"], {
            env: process.env,
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'gradle dependencies': ${e}`);
        return undefined;
    }
    const lines = output.split(/\r\n|\r|\n/);
    // find the implementation section
    const implementationSection = lines.findIndex((line) => line.includes("implementation - "));
    if (implementationSection === -1) {
        logging_1.logger.debug(`Unable to find implementation section in output of 'gradle dependencies': ${output}`);
        return undefined;
    }
    const emptyLineRegex = /^\s*$/;
    const implementationSectionLines = lines.slice(implementationSection + 1);
    const sectionEnd = implementationSectionLines.findIndex((line) => emptyLineRegex.test(line));
    const implementationLines = implementationSectionLines.slice(0, sectionEnd);
    // find the api section
    const apiSection = lines.findIndex((line) => line.includes("api - "));
    if (apiSection === -1) {
        logging_1.logger.debug(`Unable to find api section in output of 'gradle dependencies': ${output}`);
        return undefined;
    }
    const apiSectionLines = lines.slice(apiSection + 1);
    const apiSectionEnd = apiSectionLines.findIndex((line) => emptyLineRegex.test(line));
    const apiLines = apiSectionLines.slice(0, apiSectionEnd);
    const prefixRegex = /^\s*[+\\-]+\s*/;
    const suffixRegex = /\s+\([nc*]\)\s*$/;
    return [...implementationLines, ...apiLines]
        .filter((line) => line !== "No dependencies")
        .map((line) => line.replace(prefixRegex, "").replace(suffixRegex, ""));
}
exports.getGradleDependencies = getGradleDependencies;
function getDependencyInformationFromLine(line) {
    const packageNameRegex = /^\s*([^:]+):([^:]+)(?::([^\s]+))?/;
    const matches = line.match(packageNameRegex);
    if (!matches) {
        logging_1.logger.debug("Unexpected format for gradle build. Please file an issue at https://cdk.tf/bug");
        return undefined;
    }
    if (matches[3] === undefined) {
        return {
            group: "",
            name: matches[1],
            version: matches[2],
        };
    }
    return {
        group: matches[1],
        name: matches[2],
        version: matches[3],
    };
}
exports.getDependencyInformationFromLine = getDependencyInformationFromLine;
/*
 * Example output:
  implementation - Implementation dependencies for the 'main' feature. (n)
  +--- com.hashicorp:cdktf:0.18.0 (n)
  +--- software.constructs:constructs:10.0.25 (n)
  +--- junit:junit:4.13.2 (n)
  \--- org.junit.jupiter:junit-jupiter:5.8.0 (n)
*/
async function getGradlePackageVersion(packageName) {
    const translationMap = {
        jsii: "jsii-runtime",
    };
    const gradlePackageName = translationMap[packageName] || packageName;
    logging_1.logger.debug("Running 'gradle dependencies' to find package version", gradlePackageName);
    const lines = await getGradleDependencies();
    if (!lines || lines.length === 0) {
        return undefined;
    }
    // loop through the subsequent lines to find the one starting with package name
    for (const line of lines) {
        if (line.includes(`:${gradlePackageName}:`)) {
            const dep = getDependencyInformationFromLine(line);
            if (dep) {
                return dep.version;
            }
            return undefined;
        }
    }
    // Didn't find the right file, read the build.gradle file as a backup
    const buildVersion = await getGradlePackageVersionFromBuild(packageName);
    if (buildVersion) {
        return buildVersion.version;
    }
    return undefined;
}
exports.getGradlePackageVersion = getGradlePackageVersion;
async function getGradlePackageVersionFromBuild(packageName) {
    const buildGradlePath = path_1.default.join(process.cwd(), "build.gradle");
    const buildGradleContents = await fs_extra_1.default.readFile(buildGradlePath, "utf-8");
    const buildLines = buildGradleContents.split(/\r\n|\r|\n/);
    const dependenciesRegex = /^\s*dependencies\s*\{/i;
    const dependenciesStart = buildLines.findIndex((line) => dependenciesRegex.test(line));
    if (dependenciesStart === -1) {
        logging_1.logger.debug(`Unable to find dependencies section in output build.gradle`);
        return undefined;
    }
    const foundIndex = buildLines.findIndex((line) => line.includes(packageName));
    if (foundIndex === -1) {
        logging_1.logger.debug(`Unable to find package ${packageName} in output build.gradle`);
        return undefined;
    }
    const line = buildLines[foundIndex];
    const colonSeparatedPackageNameRegex = new RegExp(`([^:]+):${packageName}(?::([^\\s]+))?`, "i");
    const colonMatch = colonSeparatedPackageNameRegex.exec(line);
    if (colonMatch) {
        return {
            group: colonMatch[1],
            name: packageName,
            version: colonMatch[2] || "",
        };
    }
    const fileSeparatedPackageNameRegex = new RegExp(`java/(.*)/${packageName}/([^/]+)/.*\\.jar`, "i");
    const fileMatch = fileSeparatedPackageNameRegex.exec(line);
    if (fileMatch) {
        return {
            group: fileMatch[1].replace(/\//g, "."),
            name: packageName,
            version: fileMatch[2],
        };
    }
    return undefined;
}
exports.getGradlePackageVersionFromBuild = getGradlePackageVersionFromBuild;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhZGxlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZ3JhZGxlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7Ozs7OztBQUVILGdEQUF3QjtBQUN4Qix3REFBMEI7QUFDMUIsdUNBQW1DO0FBQ25DLGlDQUE4QjtBQUU5QixTQUFnQixlQUFlLENBQUMsZ0JBQXdCO0lBQ3RELE1BQU0sZUFBZSxHQUFHLGNBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsY0FBYyxDQUFDLENBQUM7SUFFcEUsSUFBSSxDQUFDO1FBQ0gsa0JBQUUsQ0FBQyxVQUFVLENBQUMsZUFBZSxFQUFFLGtCQUFFLENBQUMsU0FBUyxDQUFDLElBQUksR0FBRyxrQkFBRSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN0RSxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFBQyxXQUFNLENBQUM7UUFDUCxnQkFBTSxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsZUFBZSxFQUFFLENBQUMsQ0FBQztRQUM1RCxPQUFPLEtBQUssQ0FBQztJQUNmLENBQUM7QUFDSCxDQUFDO0FBVkQsMENBVUM7QUFFTSxLQUFLLFVBQVUscUJBQXFCO0lBQ3pDLElBQUksTUFBTSxDQUFDO0lBQ1gsSUFBSSxDQUFDO1FBQ0gsTUFBTSxHQUFHLE1BQU0sSUFBQSxXQUFJLEVBQUMsUUFBUSxFQUFFLENBQUMsY0FBYyxFQUFFLGlCQUFpQixDQUFDLEVBQUU7WUFDakUsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFHO1NBQ2pCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1FBQ1gsZ0JBQU0sQ0FBQyxLQUFLLENBQUMsd0NBQXdDLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDMUQsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVELE1BQU0sS0FBSyxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7SUFFekMsa0NBQWtDO0lBQ2xDLE1BQU0scUJBQXFCLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQ3JELElBQUksQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsQ0FDbkMsQ0FBQztJQUNGLElBQUkscUJBQXFCLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNqQyxnQkFBTSxDQUFDLEtBQUssQ0FDViw2RUFBNkUsTUFBTSxFQUFFLENBQ3RGLENBQUM7UUFDRixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQsTUFBTSxjQUFjLEdBQUcsT0FBTyxDQUFDO0lBQy9CLE1BQU0sMEJBQTBCLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUMxRSxNQUFNLFVBQVUsR0FBRywwQkFBMEIsQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUMvRCxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUMxQixDQUFDO0lBQ0YsTUFBTSxtQkFBbUIsR0FBRywwQkFBMEIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBRTVFLHVCQUF1QjtJQUN2QixNQUFNLFVBQVUsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7SUFDdEUsSUFBSSxVQUFVLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUN0QixnQkFBTSxDQUFDLEtBQUssQ0FDVixrRUFBa0UsTUFBTSxFQUFFLENBQzNFLENBQUM7UUFDRixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxVQUFVLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDcEQsTUFBTSxhQUFhLEdBQUcsZUFBZSxDQUFDLFNBQVMsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQ3ZELGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQzFCLENBQUM7SUFFRixNQUFNLFFBQVEsR0FBRyxlQUFlLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxhQUFhLENBQUMsQ0FBQztJQUV6RCxNQUFNLFdBQVcsR0FBRyxnQkFBZ0IsQ0FBQztJQUNyQyxNQUFNLFdBQVcsR0FBRyxrQkFBa0IsQ0FBQztJQUV2QyxPQUFPLENBQUMsR0FBRyxtQkFBbUIsRUFBRSxHQUFHLFFBQVEsQ0FBQztTQUN6QyxNQUFNLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksS0FBSyxpQkFBaUIsQ0FBQztTQUM1QyxHQUFHLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQztBQUMzRSxDQUFDO0FBckRELHNEQXFEQztBQU9ELFNBQWdCLGdDQUFnQyxDQUM5QyxJQUFZO0lBRVosTUFBTSxnQkFBZ0IsR0FBRyxtQ0FBbUMsQ0FBQztJQUM3RCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDN0MsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ2IsZ0JBQU0sQ0FBQyxLQUFLLENBQ1YsZ0ZBQWdGLENBQ2pGLENBQUM7UUFDRixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQsSUFBSSxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssU0FBUyxFQUFFLENBQUM7UUFDN0IsT0FBTztZQUNMLEtBQUssRUFBRSxFQUFFO1lBQ1QsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDaEIsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDcEIsQ0FBQztJQUNKLENBQUM7SUFFRCxPQUFPO1FBQ0wsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDakIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDaEIsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7S0FDcEIsQ0FBQztBQUNKLENBQUM7QUF6QkQsNEVBeUJDO0FBRUQ7Ozs7Ozs7RUFPRTtBQUNLLEtBQUssVUFBVSx1QkFBdUIsQ0FBQyxXQUFtQjtJQUMvRCxNQUFNLGNBQWMsR0FBMkI7UUFDN0MsSUFBSSxFQUFFLGNBQWM7S0FDckIsQ0FBQztJQUNGLE1BQU0saUJBQWlCLEdBQUcsY0FBYyxDQUFDLFdBQVcsQ0FBQyxJQUFJLFdBQVcsQ0FBQztJQUNyRSxnQkFBTSxDQUFDLEtBQUssQ0FDVix1REFBdUQsRUFDdkQsaUJBQWlCLENBQ2xCLENBQUM7SUFFRixNQUFNLEtBQUssR0FBRyxNQUFNLHFCQUFxQixFQUFFLENBQUM7SUFDNUMsSUFBSSxDQUFDLEtBQUssSUFBSSxLQUFLLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRSxDQUFDO1FBQ2pDLE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7SUFFRCwrRUFBK0U7SUFDL0UsS0FBSyxNQUFNLElBQUksSUFBSSxLQUFLLEVBQUUsQ0FBQztRQUN6QixJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxpQkFBaUIsR0FBRyxDQUFDLEVBQUUsQ0FBQztZQUM1QyxNQUFNLEdBQUcsR0FBRyxnQ0FBZ0MsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUNuRCxJQUFJLEdBQUcsRUFBRSxDQUFDO2dCQUNSLE9BQU8sR0FBRyxDQUFDLE9BQU8sQ0FBQztZQUNyQixDQUFDO1lBQ0QsT0FBTyxTQUFTLENBQUM7UUFDbkIsQ0FBQztJQUNILENBQUM7SUFFRCxxRUFBcUU7SUFDckUsTUFBTSxZQUFZLEdBQUcsTUFBTSxnQ0FBZ0MsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUN6RSxJQUFJLFlBQVksRUFBRSxDQUFDO1FBQ2pCLE9BQU8sWUFBWSxDQUFDLE9BQU8sQ0FBQztJQUM5QixDQUFDO0lBRUQsT0FBTyxTQUFTLENBQUM7QUFDbkIsQ0FBQztBQWpDRCwwREFpQ0M7QUFFTSxLQUFLLFVBQVUsZ0NBQWdDLENBQUMsV0FBbUI7SUFDeEUsTUFBTSxlQUFlLEdBQUcsY0FBSSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsR0FBRyxFQUFFLEVBQUUsY0FBYyxDQUFDLENBQUM7SUFDakUsTUFBTSxtQkFBbUIsR0FBRyxNQUFNLGtCQUFFLENBQUMsUUFBUSxDQUFDLGVBQWUsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN4RSxNQUFNLFVBQVUsR0FBRyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7SUFFM0QsTUFBTSxpQkFBaUIsR0FBRyx3QkFBd0IsQ0FBQztJQUNuRCxNQUFNLGlCQUFpQixHQUFHLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUN0RCxpQkFBaUIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQzdCLENBQUM7SUFFRixJQUFJLGlCQUFpQixLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDN0IsZ0JBQU0sQ0FBQyxLQUFLLENBQUMsNERBQTRELENBQUMsQ0FBQztRQUMzRSxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQsTUFBTSxVQUFVLEdBQUcsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO0lBQzlFLElBQUksVUFBVSxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDdEIsZ0JBQU0sQ0FBQyxLQUFLLENBQ1YsMEJBQTBCLFdBQVcseUJBQXlCLENBQy9ELENBQUM7UUFDRixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBQ0QsTUFBTSxJQUFJLEdBQUcsVUFBVSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBRXBDLE1BQU0sOEJBQThCLEdBQUcsSUFBSSxNQUFNLENBQy9DLFdBQVcsV0FBVyxpQkFBaUIsRUFDdkMsR0FBRyxDQUNKLENBQUM7SUFFRixNQUFNLFVBQVUsR0FBRyw4QkFBOEIsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDN0QsSUFBSSxVQUFVLEVBQUUsQ0FBQztRQUNmLE9BQU87WUFDTCxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQztZQUNwQixJQUFJLEVBQUUsV0FBVztZQUNqQixPQUFPLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJLEVBQUU7U0FDN0IsQ0FBQztJQUNKLENBQUM7SUFFRCxNQUFNLDZCQUE2QixHQUFHLElBQUksTUFBTSxDQUM5QyxhQUFhLFdBQVcsbUJBQW1CLEVBQzNDLEdBQUcsQ0FDSixDQUFDO0lBRUYsTUFBTSxTQUFTLEdBQUcsNkJBQTZCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzNELElBQUksU0FBUyxFQUFFLENBQUM7UUFDZCxPQUFPO1lBQ0wsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQztZQUN2QyxJQUFJLEVBQUUsV0FBVztZQUNqQixPQUFPLEVBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQztTQUN0QixDQUFDO0lBQ0osQ0FBQztJQUVELE9BQU8sU0FBUyxDQUFDO0FBQ25CLENBQUM7QUFyREQsNEVBcURDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgKGMpIEhhc2hpQ29ycCwgSW5jLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IE1QTC0yLjBcbiAqL1xuXG5pbXBvcnQgcGF0aCBmcm9tIFwicGF0aFwiO1xuaW1wb3J0IGZzIGZyb20gXCJmcy1leHRyYVwiO1xuaW1wb3J0IHsgbG9nZ2VyIH0gZnJvbSBcIi4vbG9nZ2luZ1wiO1xuaW1wb3J0IHsgZXhlYyB9IGZyb20gXCIuL3V0aWxcIjtcblxuZXhwb3J0IGZ1bmN0aW9uIGlzR3JhZGxlUHJvamVjdCh3b3JraW5nRGlyZWN0b3J5OiBzdHJpbmcpOiBib29sZWFuIHtcbiAgY29uc3QgYnVpbGRHcmFkbGVQYXRoID0gcGF0aC5qb2luKHdvcmtpbmdEaXJlY3RvcnksIFwiYnVpbGQuZ3JhZGxlXCIpO1xuXG4gIHRyeSB7XG4gICAgZnMuYWNjZXNzU3luYyhidWlsZEdyYWRsZVBhdGgsIGZzLmNvbnN0YW50cy5SX09LIHwgZnMuY29uc3RhbnRzLldfT0spO1xuICAgIHJldHVybiB0cnVlO1xuICB9IGNhdGNoIHtcbiAgICBsb2dnZXIuZGVidWcoYE5vIGJ1aWxkLmdyYWRsZSBmb3VuZCBhdCAke2J1aWxkR3JhZGxlUGF0aH1gKTtcbiAgICByZXR1cm4gZmFsc2U7XG4gIH1cbn1cblxuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGdldEdyYWRsZURlcGVuZGVuY2llcygpIHtcbiAgbGV0IG91dHB1dDtcbiAgdHJ5IHtcbiAgICBvdXRwdXQgPSBhd2FpdCBleGVjKFwiZ3JhZGxlXCIsIFtcImRlcGVuZGVuY2llc1wiLCBcIi0tY29uc29sZT1wbGFpblwiXSwge1xuICAgICAgZW52OiBwcm9jZXNzLmVudixcbiAgICB9KTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIGxvZ2dlci5kZWJ1ZyhgVW5hYmxlIHRvIHJ1biAnZ3JhZGxlIGRlcGVuZGVuY2llcyc6ICR7ZX1gKTtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgY29uc3QgbGluZXMgPSBvdXRwdXQuc3BsaXQoL1xcclxcbnxcXHJ8XFxuLyk7XG5cbiAgLy8gZmluZCB0aGUgaW1wbGVtZW50YXRpb24gc2VjdGlvblxuICBjb25zdCBpbXBsZW1lbnRhdGlvblNlY3Rpb24gPSBsaW5lcy5maW5kSW5kZXgoKGxpbmUpID0+XG4gICAgbGluZS5pbmNsdWRlcyhcImltcGxlbWVudGF0aW9uIC0gXCIpLFxuICApO1xuICBpZiAoaW1wbGVtZW50YXRpb25TZWN0aW9uID09PSAtMSkge1xuICAgIGxvZ2dlci5kZWJ1ZyhcbiAgICAgIGBVbmFibGUgdG8gZmluZCBpbXBsZW1lbnRhdGlvbiBzZWN0aW9uIGluIG91dHB1dCBvZiAnZ3JhZGxlIGRlcGVuZGVuY2llcyc6ICR7b3V0cHV0fWAsXG4gICAgKTtcbiAgICByZXR1cm4gdW5kZWZpbmVkO1xuICB9XG5cbiAgY29uc3QgZW1wdHlMaW5lUmVnZXggPSAvXlxccyokLztcbiAgY29uc3QgaW1wbGVtZW50YXRpb25TZWN0aW9uTGluZXMgPSBsaW5lcy5zbGljZShpbXBsZW1lbnRhdGlvblNlY3Rpb24gKyAxKTtcbiAgY29uc3Qgc2VjdGlvbkVuZCA9IGltcGxlbWVudGF0aW9uU2VjdGlvbkxpbmVzLmZpbmRJbmRleCgobGluZSkgPT5cbiAgICBlbXB0eUxpbmVSZWdleC50ZXN0KGxpbmUpLFxuICApO1xuICBjb25zdCBpbXBsZW1lbnRhdGlvbkxpbmVzID0gaW1wbGVtZW50YXRpb25TZWN0aW9uTGluZXMuc2xpY2UoMCwgc2VjdGlvbkVuZCk7XG5cbiAgLy8gZmluZCB0aGUgYXBpIHNlY3Rpb25cbiAgY29uc3QgYXBpU2VjdGlvbiA9IGxpbmVzLmZpbmRJbmRleCgobGluZSkgPT4gbGluZS5pbmNsdWRlcyhcImFwaSAtIFwiKSk7XG4gIGlmIChhcGlTZWN0aW9uID09PSAtMSkge1xuICAgIGxvZ2dlci5kZWJ1ZyhcbiAgICAgIGBVbmFibGUgdG8gZmluZCBhcGkgc2VjdGlvbiBpbiBvdXRwdXQgb2YgJ2dyYWRsZSBkZXBlbmRlbmNpZXMnOiAke291dHB1dH1gLFxuICAgICk7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIGNvbnN0IGFwaVNlY3Rpb25MaW5lcyA9IGxpbmVzLnNsaWNlKGFwaVNlY3Rpb24gKyAxKTtcbiAgY29uc3QgYXBpU2VjdGlvbkVuZCA9IGFwaVNlY3Rpb25MaW5lcy5maW5kSW5kZXgoKGxpbmUpID0+XG4gICAgZW1wdHlMaW5lUmVnZXgudGVzdChsaW5lKSxcbiAgKTtcblxuICBjb25zdCBhcGlMaW5lcyA9IGFwaVNlY3Rpb25MaW5lcy5zbGljZSgwLCBhcGlTZWN0aW9uRW5kKTtcblxuICBjb25zdCBwcmVmaXhSZWdleCA9IC9eXFxzKlsrXFxcXC1dK1xccyovO1xuICBjb25zdCBzdWZmaXhSZWdleCA9IC9cXHMrXFwoW25jKl1cXClcXHMqJC87XG5cbiAgcmV0dXJuIFsuLi5pbXBsZW1lbnRhdGlvbkxpbmVzLCAuLi5hcGlMaW5lc11cbiAgICAuZmlsdGVyKChsaW5lKSA9PiBsaW5lICE9PSBcIk5vIGRlcGVuZGVuY2llc1wiKVxuICAgIC5tYXAoKGxpbmUpID0+IGxpbmUucmVwbGFjZShwcmVmaXhSZWdleCwgXCJcIikucmVwbGFjZShzdWZmaXhSZWdleCwgXCJcIikpO1xufVxuXG5leHBvcnQgdHlwZSBEZXBlbmRlbmN5SW5mb3JtYXRpb24gPSB7XG4gIGdyb3VwOiBzdHJpbmc7XG4gIG5hbWU6IHN0cmluZztcbiAgdmVyc2lvbjogc3RyaW5nO1xufTtcbmV4cG9ydCBmdW5jdGlvbiBnZXREZXBlbmRlbmN5SW5mb3JtYXRpb25Gcm9tTGluZShcbiAgbGluZTogc3RyaW5nLFxuKTogRGVwZW5kZW5jeUluZm9ybWF0aW9uIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgcGFja2FnZU5hbWVSZWdleCA9IC9eXFxzKihbXjpdKyk6KFteOl0rKSg/OjooW15cXHNdKykpPy87XG4gIGNvbnN0IG1hdGNoZXMgPSBsaW5lLm1hdGNoKHBhY2thZ2VOYW1lUmVnZXgpO1xuICBpZiAoIW1hdGNoZXMpIHtcbiAgICBsb2dnZXIuZGVidWcoXG4gICAgICBcIlVuZXhwZWN0ZWQgZm9ybWF0IGZvciBncmFkbGUgYnVpbGQuIFBsZWFzZSBmaWxlIGFuIGlzc3VlIGF0IGh0dHBzOi8vY2RrLnRmL2J1Z1wiLFxuICAgICk7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIGlmIChtYXRjaGVzWzNdID09PSB1bmRlZmluZWQpIHtcbiAgICByZXR1cm4ge1xuICAgICAgZ3JvdXA6IFwiXCIsXG4gICAgICBuYW1lOiBtYXRjaGVzWzFdLFxuICAgICAgdmVyc2lvbjogbWF0Y2hlc1syXSxcbiAgICB9O1xuICB9XG5cbiAgcmV0dXJuIHtcbiAgICBncm91cDogbWF0Y2hlc1sxXSxcbiAgICBuYW1lOiBtYXRjaGVzWzJdLFxuICAgIHZlcnNpb246IG1hdGNoZXNbM10sXG4gIH07XG59XG5cbi8qXG4gKiBFeGFtcGxlIG91dHB1dDpcbiAgaW1wbGVtZW50YXRpb24gLSBJbXBsZW1lbnRhdGlvbiBkZXBlbmRlbmNpZXMgZm9yIHRoZSAnbWFpbicgZmVhdHVyZS4gKG4pXG4gICstLS0gY29tLmhhc2hpY29ycDpjZGt0ZjowLjE4LjAgKG4pXG4gICstLS0gc29mdHdhcmUuY29uc3RydWN0czpjb25zdHJ1Y3RzOjEwLjAuMjUgKG4pXG4gICstLS0ganVuaXQ6anVuaXQ6NC4xMy4yIChuKVxuICBcXC0tLSBvcmcuanVuaXQuanVwaXRlcjpqdW5pdC1qdXBpdGVyOjUuOC4wIChuKVxuKi9cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBnZXRHcmFkbGVQYWNrYWdlVmVyc2lvbihwYWNrYWdlTmFtZTogc3RyaW5nKSB7XG4gIGNvbnN0IHRyYW5zbGF0aW9uTWFwOiBSZWNvcmQ8c3RyaW5nLCBzdHJpbmc+ID0ge1xuICAgIGpzaWk6IFwianNpaS1ydW50aW1lXCIsXG4gIH07XG4gIGNvbnN0IGdyYWRsZVBhY2thZ2VOYW1lID0gdHJhbnNsYXRpb25NYXBbcGFja2FnZU5hbWVdIHx8IHBhY2thZ2VOYW1lO1xuICBsb2dnZXIuZGVidWcoXG4gICAgXCJSdW5uaW5nICdncmFkbGUgZGVwZW5kZW5jaWVzJyB0byBmaW5kIHBhY2thZ2UgdmVyc2lvblwiLFxuICAgIGdyYWRsZVBhY2thZ2VOYW1lLFxuICApO1xuXG4gIGNvbnN0IGxpbmVzID0gYXdhaXQgZ2V0R3JhZGxlRGVwZW5kZW5jaWVzKCk7XG4gIGlmICghbGluZXMgfHwgbGluZXMubGVuZ3RoID09PSAwKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIC8vIGxvb3AgdGhyb3VnaCB0aGUgc3Vic2VxdWVudCBsaW5lcyB0byBmaW5kIHRoZSBvbmUgc3RhcnRpbmcgd2l0aCBwYWNrYWdlIG5hbWVcbiAgZm9yIChjb25zdCBsaW5lIG9mIGxpbmVzKSB7XG4gICAgaWYgKGxpbmUuaW5jbHVkZXMoYDoke2dyYWRsZVBhY2thZ2VOYW1lfTpgKSkge1xuICAgICAgY29uc3QgZGVwID0gZ2V0RGVwZW5kZW5jeUluZm9ybWF0aW9uRnJvbUxpbmUobGluZSk7XG4gICAgICBpZiAoZGVwKSB7XG4gICAgICAgIHJldHVybiBkZXAudmVyc2lvbjtcbiAgICAgIH1cbiAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICB9XG5cbiAgLy8gRGlkbid0IGZpbmQgdGhlIHJpZ2h0IGZpbGUsIHJlYWQgdGhlIGJ1aWxkLmdyYWRsZSBmaWxlIGFzIGEgYmFja3VwXG4gIGNvbnN0IGJ1aWxkVmVyc2lvbiA9IGF3YWl0IGdldEdyYWRsZVBhY2thZ2VWZXJzaW9uRnJvbUJ1aWxkKHBhY2thZ2VOYW1lKTtcbiAgaWYgKGJ1aWxkVmVyc2lvbikge1xuICAgIHJldHVybiBidWlsZFZlcnNpb24udmVyc2lvbjtcbiAgfVxuXG4gIHJldHVybiB1bmRlZmluZWQ7XG59XG5cbmV4cG9ydCBhc3luYyBmdW5jdGlvbiBnZXRHcmFkbGVQYWNrYWdlVmVyc2lvbkZyb21CdWlsZChwYWNrYWdlTmFtZTogc3RyaW5nKSB7XG4gIGNvbnN0IGJ1aWxkR3JhZGxlUGF0aCA9IHBhdGguam9pbihwcm9jZXNzLmN3ZCgpLCBcImJ1aWxkLmdyYWRsZVwiKTtcbiAgY29uc3QgYnVpbGRHcmFkbGVDb250ZW50cyA9IGF3YWl0IGZzLnJlYWRGaWxlKGJ1aWxkR3JhZGxlUGF0aCwgXCJ1dGYtOFwiKTtcbiAgY29uc3QgYnVpbGRMaW5lcyA9IGJ1aWxkR3JhZGxlQ29udGVudHMuc3BsaXQoL1xcclxcbnxcXHJ8XFxuLyk7XG5cbiAgY29uc3QgZGVwZW5kZW5jaWVzUmVnZXggPSAvXlxccypkZXBlbmRlbmNpZXNcXHMqXFx7L2k7XG4gIGNvbnN0IGRlcGVuZGVuY2llc1N0YXJ0ID0gYnVpbGRMaW5lcy5maW5kSW5kZXgoKGxpbmUpID0+XG4gICAgZGVwZW5kZW5jaWVzUmVnZXgudGVzdChsaW5lKSxcbiAgKTtcblxuICBpZiAoZGVwZW5kZW5jaWVzU3RhcnQgPT09IC0xKSB7XG4gICAgbG9nZ2VyLmRlYnVnKGBVbmFibGUgdG8gZmluZCBkZXBlbmRlbmNpZXMgc2VjdGlvbiBpbiBvdXRwdXQgYnVpbGQuZ3JhZGxlYCk7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIGNvbnN0IGZvdW5kSW5kZXggPSBidWlsZExpbmVzLmZpbmRJbmRleCgobGluZSkgPT4gbGluZS5pbmNsdWRlcyhwYWNrYWdlTmFtZSkpO1xuICBpZiAoZm91bmRJbmRleCA9PT0gLTEpIHtcbiAgICBsb2dnZXIuZGVidWcoXG4gICAgICBgVW5hYmxlIHRvIGZpbmQgcGFja2FnZSAke3BhY2thZ2VOYW1lfSBpbiBvdXRwdXQgYnVpbGQuZ3JhZGxlYCxcbiAgICApO1xuICAgIHJldHVybiB1bmRlZmluZWQ7XG4gIH1cbiAgY29uc3QgbGluZSA9IGJ1aWxkTGluZXNbZm91bmRJbmRleF07XG5cbiAgY29uc3QgY29sb25TZXBhcmF0ZWRQYWNrYWdlTmFtZVJlZ2V4ID0gbmV3IFJlZ0V4cChcbiAgICBgKFteOl0rKToke3BhY2thZ2VOYW1lfSg/OjooW15cXFxcc10rKSk/YCxcbiAgICBcImlcIixcbiAgKTtcblxuICBjb25zdCBjb2xvbk1hdGNoID0gY29sb25TZXBhcmF0ZWRQYWNrYWdlTmFtZVJlZ2V4LmV4ZWMobGluZSk7XG4gIGlmIChjb2xvbk1hdGNoKSB7XG4gICAgcmV0dXJuIHtcbiAgICAgIGdyb3VwOiBjb2xvbk1hdGNoWzFdLFxuICAgICAgbmFtZTogcGFja2FnZU5hbWUsXG4gICAgICB2ZXJzaW9uOiBjb2xvbk1hdGNoWzJdIHx8IFwiXCIsXG4gICAgfTtcbiAgfVxuXG4gIGNvbnN0IGZpbGVTZXBhcmF0ZWRQYWNrYWdlTmFtZVJlZ2V4ID0gbmV3IFJlZ0V4cChcbiAgICBgamF2YS8oLiopLyR7cGFja2FnZU5hbWV9LyhbXi9dKykvLipcXFxcLmphcmAsXG4gICAgXCJpXCIsXG4gICk7XG5cbiAgY29uc3QgZmlsZU1hdGNoID0gZmlsZVNlcGFyYXRlZFBhY2thZ2VOYW1lUmVnZXguZXhlYyhsaW5lKTtcbiAgaWYgKGZpbGVNYXRjaCkge1xuICAgIHJldHVybiB7XG4gICAgICBncm91cDogZmlsZU1hdGNoWzFdLnJlcGxhY2UoL1xcLy9nLCBcIi5cIiksXG4gICAgICBuYW1lOiBwYWNrYWdlTmFtZSxcbiAgICAgIHZlcnNpb246IGZpbGVNYXRjaFsyXSxcbiAgICB9O1xuICB9XG5cbiAgcmV0dXJuIHVuZGVmaW5lZDtcbn1cbiJdfQ==