"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectDebugInformation = exports.getPackageVersion = exports.getNodeVersion = exports.getGoVersion = exports.getPipenvVersion = exports.getPipVersion = exports.getPythonVersion = exports.getGradleVersion = exports.getMavenVersion = exports.getJavaVersion = exports.getDotnetVersion = exports.getLanguage = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const path = __importStar(require("path"));
const os = __importStar(require("os"));
const logging_1 = require("./logging");
const util_1 = require("./util");
const terraform_1 = require("./terraform");
const version_1 = require("./version");
const fs_extra_1 = require("fs-extra");
const gradle_1 = require("./gradle");
function getLanguage(projectPath = process.cwd()) {
    try {
        const cdktfJson = require(path.resolve(projectPath, "cdktf.json"));
        return cdktfJson.language;
    }
    catch (_a) {
        // We can not detect the language
        logging_1.logger.debug(`Unable to detect language in ${projectPath}`);
        return undefined;
    }
}
exports.getLanguage = getLanguage;
async function getBinaryVersion(binary, versionCommand) {
    const noOp = () => { }; // eslint-disable-line @typescript-eslint/no-empty-function
    try {
        const result = await (0, util_1.exec)(binary, [versionCommand], { env: process.env, stdio: [undefined, undefined, "ignore"] }, noOp, noOp);
        return result.trim().replace(/\r?\n|\r/g, "");
    }
    catch (e) {
        logging_1.logger.debug(`Failed to run ${binary} ${versionCommand}: ${e}`);
        return undefined;
    }
}
function getDotnetVersion() {
    return getBinaryVersion("dotnet", "--version");
}
exports.getDotnetVersion = getDotnetVersion;
function getJavaVersion() {
    return getBinaryVersion("java", "-version");
}
exports.getJavaVersion = getJavaVersion;
function getMavenVersion() {
    return getBinaryVersion("mvn", "--version");
}
exports.getMavenVersion = getMavenVersion;
function getGradleVersion() {
    return getBinaryVersion("gradle", "--version");
}
exports.getGradleVersion = getGradleVersion;
async function getPythonVersion() {
    return ((await getBinaryVersion("python3", "--version")) ||
        (await getBinaryVersion("python", "--version")));
}
exports.getPythonVersion = getPythonVersion;
function getPipVersion() {
    return getBinaryVersion("pip", "--version");
}
exports.getPipVersion = getPipVersion;
function getPipenvVersion() {
    return getBinaryVersion("pipenv", "--version");
}
exports.getPipenvVersion = getPipenvVersion;
function getGoVersion() {
    return getBinaryVersion("go", "version");
}
exports.getGoVersion = getGoVersion;
function getNodeVersion() {
    return getBinaryVersion("node", "--version");
}
exports.getNodeVersion = getNodeVersion;
async function getNodeModuleVersion(packageName) {
    // Use the presence of the pnpm lock file as a signal that
    // we should interrogate packages via pnpm instead of npm.
    const usingPnpm = await (0, fs_extra_1.pathExists)("pnpm-lock.yaml");
    return usingPnpm
        ? getPnpmNodeModuleVersion(packageName)
        : getNpmNodeModuleVersion(packageName);
}
async function getPnpmNodeModuleVersion(packageName) {
    var _a, _b, _c;
    let output;
    try {
        output = await (0, util_1.exec)("pnpm", ["list", packageName, "--json"], {
            env: { ...process.env },
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'pnpm list ${packageName} --json': ${e}`);
        return undefined;
    }
    let json;
    try {
        json = JSON.parse(output);
    }
    catch (e) {
        logging_1.logger.debug(`Unable to parse output of 'pnpm list ${packageName} --json': ${e}`);
        return undefined;
    }
    if (!json ||
        !Array.isArray(json) ||
        json.length === 0 ||
        !((_c = (_b = (_a = json[0]) === null || _a === void 0 ? void 0 : _a.dependencies) === null || _b === void 0 ? void 0 : _b[packageName]) === null || _c === void 0 ? void 0 : _c.version)) {
        logging_1.logger.debug(`Unable to find '${packageName}' in 'pnpm list ${packageName} --json': ${output}`);
        return undefined;
    }
    return json[0].dependencies[packageName].version;
}
async function getNpmNodeModuleVersion(packageName) {
    let output;
    try {
        output = await (0, util_1.exec)("npm", ["list", packageName, "--json"], {
            env: { ...process.env },
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'npm list ${packageName} --json': ${e}`);
        return undefined;
    }
    let json;
    try {
        json = JSON.parse(output);
    }
    catch (e) {
        logging_1.logger.debug(`Unable to parse output of 'npm list ${packageName} --json': ${e}`);
        return undefined;
    }
    if (!json.dependencies ||
        !json.dependencies[packageName] ||
        !json.dependencies[packageName].version) {
        logging_1.logger.debug(`Unable to find '${packageName}' in 'npm list ${packageName} --json': ${output}`);
        return undefined;
    }
    return json.dependencies[packageName].version;
}
async function getPythonPackageVersion(packageName) {
    let output;
    try {
        output = await (0, util_1.exec)("pipenv", ["run", "pip", "show", packageName], {
            env: process.env,
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'pipenv run pip show ${packageName}': ${e}`);
    }
    // If we couldn't get the output using pipenv, try to get it using pip directly
    if (!output) {
        try {
            output = await (0, util_1.exec)("pip", ["show", packageName], {
                env: process.env,
            });
        }
        catch (e) {
            logging_1.logger.debug(`Unable to run 'pip show ${packageName}': ${e}`);
        }
    }
    if (!output) {
        return undefined;
    }
    const versionInfo = output
        .split(/\r\n|\r|\n/)
        .find((line) => line.startsWith("Version:"));
    if (!versionInfo) {
        logging_1.logger.debug(`Unable to find version in output of 'pipenv run pip show ${packageName}' / 'pip show ${packageName}': ${output}`);
        return undefined;
    }
    const version = versionInfo.split(":")[1].trim();
    return version;
}
// Once https://github.com/NuGet/Home/issues/7752 gets resolved we can also try with --json
async function getCSharpPackageVersion(packageName) {
    const translationMap = {
        jsii: "Amazon.JSII.Runtime",
        constructs: "Constructs",
        cdktf: "HashiCorp.Cdktf",
    };
    const cSharpPackageName = translationMap[packageName] || packageName;
    let output;
    try {
        output = await (0, util_1.exec)("dotnet", ["list", "package", "--include-transitive"], {
            env: process.env,
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'dotnet list package --include-transitive': ${e}`);
        return undefined;
    }
    const versionLine = output
        .split(/\r\n|\r|\n/)
        .find((line) => line.includes(`> ${cSharpPackageName}`));
    if (!versionLine) {
        logging_1.logger.debug(`Unable to find version for '${cSharpPackageName}' in output of 'dotnet list package --include-transitive': ${output}`);
        return undefined;
    }
    // A line might look like this:
    //   Transitive Package                                           Resolved
    //   > Amazon.JSII.Runtime                                        1.17.1
    // Therefore we reverse split by whitespace and reverse
    // so that the first thing we find is the version number
    return versionLine
        .split(" ")
        .reverse()
        .find((part) => part !== "");
}
async function getGoPackageVersion(packageName) {
    var _a;
    const translationMap = {
        jsii: "jsii-runtime-go",
        cdktf: "github.com/hashicorp/terraform-cdk-go",
    };
    const goPackageName = translationMap[packageName] || packageName;
    let output;
    try {
        output = await (0, util_1.exec)("go", ["list", "-m", "all"], {
            env: process.env,
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'go list -m all': ${e}`);
        return undefined;
    }
    let versionLine = output
        .split(/\r\n|\r|\n/)
        .find((line) => line.includes(goPackageName));
    if (!versionLine) {
        logging_1.logger.debug(`Unable to find version for '${goPackageName}' in output of 'go list -m all': ${output}`);
        return undefined;
    }
    // We are dealing with a local version
    if (versionLine.includes("=>")) {
        logging_1.logger.debug(`Found local version for '${goPackageName}': ${versionLine}`);
        versionLine = versionLine.split("=>")[0].trim();
    }
    return (_a = versionLine.split(" ").pop()) === null || _a === void 0 ? void 0 : _a.replace("v", "");
}
async function getMavenPackageVersion(packageName) {
    const translationMap = {
        jsii: "jsii-runtime",
    };
    const javaPackageName = translationMap[packageName] || packageName;
    let output;
    try {
        output = await (0, util_1.exec)("mvn", ["dependency:list"], {
            env: process.env,
        });
    }
    catch (e) {
        logging_1.logger.debug(`Unable to run 'mvn dependency:list': ${e}`);
        return undefined;
    }
    const resolutionPart = output.split("The following files have been resolved")[1];
    if (!resolutionPart) {
        logging_1.logger.debug(`Unable to find resolution passage in output of 'mvn dependency:list': ${output}`);
        return undefined;
    }
    // We need to search for the package name AND a colon to not match a line like
    // [INFO]    com.hashicorp:cdktf-provider-aws:jar:9.0.9:compile
    // when looking for the cdktf package (see Github issue #1994)
    const versionLine = resolutionPart
        .split(/\r\n|\r|\n/)
        .find((line) => line.includes(`${javaPackageName}:`));
    if (!versionLine) {
        logging_1.logger.debug(`Unable to find version for '${javaPackageName}' in output of 'mvn dependency:list': ${output}`);
        return undefined;
    }
    // Example line: [INFO]    com.hashicorp:cdktf:jar:0.0.0:compile
    const versionStartDelimiter = `${javaPackageName}:jar:`;
    const versionStart = versionLine.indexOf(versionStartDelimiter) + versionStartDelimiter.length;
    const versionEndDelemiter = ":compile";
    const versionEnd = versionLine.indexOf(versionEndDelemiter);
    return versionLine.substring(versionStart, versionEnd);
}
async function getJavaPackageVersion(packageName) {
    if ((0, gradle_1.isGradleProject)("./")) {
        return (0, gradle_1.getGradlePackageVersion)(packageName);
    }
    return getMavenPackageVersion(packageName);
}
async function getPackageVersion(language, packageName) {
    const noOp = async () => undefined; // eslint-disable-line @typescript-eslint/no-empty-function
    const getLibraryVersionMap = {
        typescript: getNodeModuleVersion,
        python: getPythonPackageVersion,
        go: getGoPackageVersion,
        csharp: getCSharpPackageVersion,
        java: getJavaPackageVersion,
    };
    const libVersion = await (getLibraryVersionMap[language] || noOp)(packageName);
    return libVersion !== null && libVersion !== void 0 ? libVersion : undefined;
}
exports.getPackageVersion = getPackageVersion;
async function collectDebugInformation() {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
    const debugOutput = {};
    const language = getLanguage();
    debugOutput["language"] = language !== null && language !== void 0 ? language : null;
    debugOutput["cdktf-cli"] = version_1.DISPLAY_VERSION;
    const node = getNodeVersion();
    debugOutput["node"] = (_a = (await node)) !== null && _a !== void 0 ? _a : null;
    if (language) {
        const cdktf = getPackageVersion(language, "cdktf");
        const constructs = getPackageVersion(language, "constructs");
        const jsii = getPackageVersion(language, "jsii");
        debugOutput["cdktf"] = (_b = (await cdktf)) !== null && _b !== void 0 ? _b : null;
        debugOutput["constructs"] = (_c = (await constructs)) !== null && _c !== void 0 ? _c : null;
        debugOutput["jsii"] = (_d = (await jsii)) !== null && _d !== void 0 ? _d : null;
    }
    debugOutput["terraform"] = await terraform_1.terraformVersion;
    debugOutput["arch"] = os.arch();
    debugOutput["os"] = `${os.platform()} ${os.release()}`;
    switch (language) {
        case "python":
            {
                const python = getPythonVersion();
                const pip = getPipVersion();
                const pipenv = getPipenvVersion();
                debugOutput["python"] = (_e = (await python)) !== null && _e !== void 0 ? _e : null;
                debugOutput["pip"] = (_f = (await pip)) !== null && _f !== void 0 ? _f : null;
                debugOutput["pipenv"] = (_g = (await pipenv)) !== null && _g !== void 0 ? _g : null;
            }
            break;
        case "java":
            {
                const java = getJavaVersion();
                const maven = getMavenVersion();
                const gradle = getGradleVersion();
                debugOutput["java"] = (_h = (await java)) !== null && _h !== void 0 ? _h : null;
                debugOutput["maven"] = (_j = (await maven)) !== null && _j !== void 0 ? _j : null;
                debugOutput["gradle"] = (_k = (await gradle)) !== null && _k !== void 0 ? _k : null;
            }
            break;
        case "csharp":
            debugOutput["dotnet"] = (_l = (await getDotnetVersion())) !== null && _l !== void 0 ? _l : null;
            break;
        case "go":
            debugOutput["go"] = (_m = (await getGoVersion())) !== null && _m !== void 0 ? _m : null;
            break;
    }
    return debugOutput;
}
exports.collectDebugInformation = collectDebugInformation;
//# sourceMappingURL=data:application/json;base64,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