"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstructsMakerProviderTarget = exports.ConstructsMakerModuleTarget = exports.ConstructsMakerTarget = void 0;
const config_1 = require("./config");
class ConstructsMakerTarget {
    constructor(constraint, targetLanguage) {
        this.constraint = constraint;
        this.targetLanguage = targetLanguage;
        if (this.constraint instanceof config_1.TerraformModuleConstraint) {
            this.fileName = `${this.typesPath(this.constraint.fileName)}.ts`;
        }
        else {
            this.fileName = `${this.typesPath(this.constraint.name)}.ts`;
        }
    }
    static from(constraint, targetLanguage) {
        if (constraint instanceof config_1.TerraformModuleConstraint) {
            return new ConstructsMakerModuleTarget(constraint, targetLanguage);
        }
        else {
            return new ConstructsMakerProviderTarget(constraint, targetLanguage);
        }
    }
    get version() {
        return this.constraint.version;
    }
    get source() {
        return this.constraint.source;
    }
    get name() {
        return this.constraint.name;
    }
    get fqn() {
        return this.constraint.fqn;
    }
    get namespace() {
        return this.constraint.namespace;
    }
    get moduleKey() {
        return this.fqn.replace(/\//gi, "_");
    }
}
exports.ConstructsMakerTarget = ConstructsMakerTarget;
class ConstructsMakerModuleTarget extends ConstructsMakerTarget {
    get isModule() {
        return true;
    }
    get isProvider() {
        return false;
    }
    get srcMakName() {
        switch (this.targetLanguage) {
            case config_1.Language.GO:
                return this.name.replace(/-/gi, "_");
            case config_1.Language.JAVA:
            case config_1.Language.CSHARP:
            case config_1.Language.PYTHON:
                return this.simplifiedFqn;
            default:
                return this.simplifiedName;
        }
    }
    get trackingPayload() {
        return {
            name: this.name,
            fullName: this.fqn,
            version: this.version,
            type: "module",
        };
    }
    typesPath(name) {
        return `modules/${name}`;
    }
    get simplifiedName() {
        var _a, _b;
        return ((_b = (_a = this.namespace) === null || _a === void 0 ? void 0 : _a.replace(/\//gi, ".").replace(/-/gi, "_")) !== null && _b !== void 0 ? _b : this.name);
    }
    get simplifiedFqn() {
        return this.fqn.replace(/\//gi, ".").replace(/-/gi, "_");
    }
}
exports.ConstructsMakerModuleTarget = ConstructsMakerModuleTarget;
class ConstructsMakerProviderTarget extends ConstructsMakerTarget {
    get isModule() {
        return false;
    }
    get isProvider() {
        return true;
    }
    get srcMakName() {
        switch (this.targetLanguage) {
            case config_1.Language.JAVA:
                // "null" is a reserved keyword and can't be used as a package name
                return this.isNullProvider ? "nullprovider" : this.simplifiedName;
            case config_1.Language.CSHARP:
                // "null" is a reserved keyword and can't be used as a namespace
                return this.isNullProvider ? "Providers.Null" : this.simplifiedName;
            case config_1.Language.PYTHON:
                return this.simplifiedName;
            case config_1.Language.GO:
                return this.name.replace(/-/gi, "_");
            default:
                return this.constraint.fqn;
        }
    }
    get trackingPayload() {
        return {
            name: this.name,
            fullName: this.source,
            version: this.version,
            type: "provider",
        };
    }
    typesPath(name) {
        return `providers/${name}/index`;
    }
    get isNullProvider() {
        return this.constraint.name === "null";
    }
    get simplifiedName() {
        return this.name.replace(/\//gi, ".").replace(/-/gi, "_");
    }
}
exports.ConstructsMakerProviderTarget = ConstructsMakerProviderTarget;
//# sourceMappingURL=data:application/json;base64,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