"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerraformProviderLock = void 0;
const path_1 = __importDefault(require("path"));
const promises_1 = __importDefault(require("fs/promises"));
const dependency_manager_1 = require("./dependencies/dependency-manager");
const commons_1 = require("@cdktf/commons");
const TerraformLockFileName = ".terraform.lock.hcl";
class TerraformProviderLock {
    constructor(stackWorkingDirectory) {
        this.stackWorkingDirectory = stackWorkingDirectory;
        this._providerLockData = null;
    }
    get lockFilePath() {
        return path_1.default.join(this.stackWorkingDirectory, TerraformLockFileName);
    }
    async hasProviderLockFile() {
        try {
            await promises_1.default.stat(this.lockFilePath);
            return true;
        }
        catch (e) {
            return false;
        }
    }
    async readProviderLockFile() {
        try {
            const lockFile = (await promises_1.default.readFile(this.lockFilePath)).toString();
            return lockFile;
        }
        catch (e) {
            commons_1.logger.debug("Unable to read provider lock file", e);
        }
        return "";
    }
    parseProviderLockFile(contents) {
        const providerLockData = {
            providers: {},
        };
        let currentProvider;
        contents.split(/\r\n|\r|\n/).forEach((line) => {
            if (currentProvider) {
                const constraintMatch = line.match(/^\s*constraints\s+=\s+"(.*)"/);
                if (constraintMatch) {
                    providerLockData.providers[currentProvider].constraints =
                        new dependency_manager_1.ProviderConstraint(currentProvider, constraintMatch[1]);
                    return;
                }
                const versionMatch = line.match(/^\s*version\s+=\s+"(.*)"/);
                if (versionMatch) {
                    providerLockData.providers[currentProvider].version = versionMatch[1];
                    return;
                }
                const endMatch = line.match(/^\s*}/);
                if (endMatch) {
                    currentProvider = undefined;
                }
                return;
            }
            const providerMatch = line.match(/provider "(.*)"/);
            if (providerMatch) {
                currentProvider = providerMatch[1];
                providerLockData.providers[currentProvider] = {
                    name: new dependency_manager_1.ProviderConstraint(currentProvider, ">=0").simplifiedName,
                };
            }
        });
        return providerLockData;
    }
    async providers(forceReread = false) {
        if (!this._providerLockData || forceReread) {
            const contents = await this.readProviderLockFile();
            this._providerLockData = this.parseProviderLockFile(contents);
        }
        return this._providerLockData.providers;
    }
    async hasMatchingProvider(constraint) {
        var _a, _b;
        const providerLockData = await this.providers();
        const lockedProvider = providerLockData[constraint.source];
        if (lockedProvider) {
            return (_a = lockedProvider.constraints) === null || _a === void 0 ? void 0 : _a.matchesVersion((_b = constraint.version) !== null && _b !== void 0 ? _b : ">0");
        }
        return false;
    }
}
exports.TerraformProviderLock = TerraformProviderLock;
//# sourceMappingURL=data:application/json;base64,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