"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynthStack = exports.StackDependencies = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const fs = __importStar(require("fs-extra"));
const path = __importStar(require("path"));
const chalk = __importStar(require("chalk"));
const indent_string_1 = __importDefault(require("indent-string"));
const cdktf_1 = require("cdktf");
const perf_hooks_1 = require("perf_hooks");
const commons_1 = require("@cdktf/commons");
const cdktf_config_1 = require("./cdktf-config");
const hcl_tools_1 = require("@cdktf/hcl-tools");
const chalkColour = new chalk.Instance();
class StackDependencies {
    constructor(stacks) {
        this.stacks = stacks;
        this.pendingStacks = [];
        this.inflightStacks = [];
        this.deployedStacks = [];
        this.pendingStacks = [...this.stacks];
    }
    startRun(stack) {
        this.pendingStacks = this.pendingStacks.filter((item) => item.name !== stack.name);
        this.inflightStacks.push(stack);
    }
    finishRun(stack) {
        this.inflightStacks = this.inflightStacks.filter((item) => item.name !== stack.name);
        this.deployedStacks.push(stack);
    }
    get pendingDeployableStacks() {
        return this.pendingStacks.filter((pendingStack) => {
            const unmetDependencies = pendingStack.dependencies.filter((dependency) => {
                return !this.deployedStacks.some((deployedStack) => {
                    return deployedStack.name === dependency;
                });
            });
            return unmetDependencies.length === 0;
        });
    }
}
exports.StackDependencies = StackDependencies;
class SynthStack {
    static async synth(abortSignal, command, outdir, workingDirectory = process.cwd(), graceful = false, // will not exit the process but rethrow the error instead
    noColor = false, synthOrigin, hcl = false) {
        // start performance timer
        const startTime = perf_hooks_1.performance.now();
        const isDirectory = (source) => fs.lstatSync(source).isDirectory();
        const getDirectories = (source) => {
            if (!fs.existsSync(source))
                return [];
            return fs
                .readdirSync(source)
                .map((name) => path.join(source, name))
                .filter(isDirectory);
        };
        const existingDirectories = getDirectories(path.join(outdir, cdktf_1.Manifest.stacksFolder));
        const env = Object.fromEntries(Object.entries(process.env).filter(
        // We don't want to pass Terraform variables to the synth command since they should only be used at execution time
        ([key]) => !key.startsWith("TF_VAR_")));
        // Increases the default memory available to Node.js when synthesizing a TypeScript CDK project.
        const nodeOptsSwitch = "--max-old-space-size";
        const nodeOptsSetting = `${nodeOptsSwitch}=4096`;
        if (env.NODE_OPTIONS && !env.NODE_OPTIONS.includes(nodeOptsSwitch)) {
            commons_1.logger.warn(`WARNING: Found NODE_OPTIONS environment variable without a setting for ${nodeOptsSwitch}
The synthesizing step for TypeScript may need an increased amount of memory if multiple large providers
are used with locally generated bindings. You can ignore this if you don't use CDKTF with TypeScript.
If not present, the cdktf-cli sets it to NODE_OPTIONS="${nodeOptsSetting}" by default. But as
your environment already contains a NODE_OPTIONS variable, we won't override it. Hence, the app command
might fail while synthesizing with an out of memory error.`);
        }
        else if (!env.NODE_OPTIONS) {
            // increase memory to allow ts-node (when using TypeScript) to handle large amounts of generated code in memory
            env.NODE_OPTIONS = `${env.NODE_OPTIONS || ""} ${nodeOptsSetting}`.trim();
        }
        const currentContext = process.env[cdktf_1.CONTEXT_ENV]
            ? JSON.parse(process.env.CDKTF_CONTEXT_JSON)
            : {};
        const relativeModules = getRelativeTerraformModules();
        try {
            await (0, commons_1.shell)(command, [], {
                shell: true,
                env: {
                    ...env,
                    CDKTF_OUTDIR: outdir,
                    CDKTF_CONTINUE_SYNTH_ON_ERROR_ANNOTATIONS: "true",
                    [cdktf_1.CONTEXT_ENV]: JSON.stringify({
                        ...currentContext,
                        cdktfRelativeModules: relativeModules,
                    }),
                    SYNTH_HCL_OUTPUT: hcl.toString(),
                },
                cwd: workingDirectory,
                signal: abortSignal,
                noColor: noColor,
            });
        }
        catch (e) {
            const errorOutput = chalkColour `{redBright cdktf encountered an error while synthesizing}

Synth command: {blue ${command}}
Error:         {redBright ${e.message}}
${e.stderr
                ? chalkColour `
Command output on stderr:

{dim ${(0, indent_string_1.default)(e.stderr, 4)}}
`
                : ""}
${e.stdout
                ? chalkColour `
Command output on stdout:

{dim ${(0, indent_string_1.default)(e.stdout, 4)}}
`
                : ""}`;
            await this.synthErrorTelemetry(synthOrigin);
            if (graceful) {
                e.errorOutput = errorOutput;
                throw e;
            }
            console.error(`ERROR: ${errorOutput}`);
            process.exit(1);
        }
        // Apply formatting to HCL files if hcl output was selected
        if (hcl) {
            await SynthStack.formatHclFiles(outdir);
        }
        // end performance timer
        const endTime = perf_hooks_1.performance.now();
        let stacks = [];
        try {
            stacks = await SynthStack.readSynthesizedStacks(outdir);
        }
        catch (e) {
            const errorMessage = `ERROR: synthesis failed, because app was expected to call 'synth()', but didn't. Thus "${outdir}/${cdktf_1.Manifest.fileName}"  was not created: ${e}`;
            if (graceful) {
                throw new Error(errorMessage);
            }
            commons_1.logger.error(errorMessage);
            process.exit(1);
        }
        await this.synthTelemetry(endTime - startTime, stacks, synthOrigin);
        if (stacks.length === 0) {
            commons_1.logger.error("ERROR: No Terraform code synthesized.");
        }
        const stackNames = stacks.map((s) => s.name);
        const orphanedDirectories = existingDirectories.filter((e) => !stackNames.includes(path.basename(e)));
        for (const orphanedDirectory of orphanedDirectories) {
            fs.rmSync(orphanedDirectory, { recursive: true });
        }
        return stacks;
    }
    static async formatHclFiles(outDir) {
        const manifestPath = path.join(outDir, cdktf_1.Manifest.fileName);
        if (!(await fs.pathExists(manifestPath))) {
            throw new Error(`Could not find manifest file at ${manifestPath}. In case --skip-synth was passed, please try again without the flag.`);
        }
        const manifest = JSON.parse(fs.readFileSync(manifestPath).toString());
        for (const stackName in manifest.stacks) {
            const stack = manifest.stacks[stackName];
            const filePath = path.join(outDir, stack.synthesizedStackPath);
            const hclContent = fs.readFileSync(filePath).toString();
            const formattedHcl = await (0, hcl_tools_1.format)(hclContent);
            fs.writeFileSync(filePath, formattedHcl);
        }
    }
    static async readSynthesizedStacks(outdir) {
        const manifestPath = path.join(outdir, cdktf_1.Manifest.fileName);
        if (!(await fs.pathExists(manifestPath))) {
            throw new Error(`Could not find manifest file at ${manifestPath}. In case --skip-synth was passed, please try again without the flag.`);
        }
        const stacks = [];
        const manifest = JSON.parse(fs.readFileSync(manifestPath).toString());
        for (const stackName in manifest.stacks) {
            const stack = manifest.stacks[stackName];
            const filePath = path.join(outdir, stack.synthesizedStackPath);
            let jsonContent = {};
            if (filePath.endsWith(".tf.json")) {
                jsonContent = JSON.parse(fs.readFileSync(filePath).toString());
            }
            else {
                const metadataPath = path.join(outdir, stack.stackMetadataPath);
                jsonContent = JSON.parse(fs.readFileSync(metadataPath).toString());
            }
            stacks.push({
                ...stack,
                workingDirectory: path.join(outdir, stack.workingDirectory),
                content: JSON.stringify(jsonContent, null, 2),
            });
        }
        return stacks;
    }
    static async synthTelemetry(totalTime, stacks, synthOrigin) {
        const config = (0, commons_1.readConfigSync)();
        await (0, commons_1.sendTelemetry)("synth", {
            totalTime: totalTime,
            language: config.language,
            synthOrigin,
            stackMetadata: stacks.map((stack) => JSON.parse(stack.content)["//"].metadata),
            requiredProviders: stacks.map((stack) => JSON.parse(stack.content)["terraform"].required_providers),
        });
    }
    static async synthErrorTelemetry(synthOrigin) {
        await (0, commons_1.sendTelemetry)("synth", { error: true, synthOrigin });
    }
}
exports.SynthStack = SynthStack;
function getRelativeTerraformModules() {
    let cfg;
    try {
        cfg = cdktf_config_1.CdktfConfig.read();
    }
    catch (e) {
        commons_1.logger.trace("Could not read cdktf.json: " + e);
        return [];
    }
    return cfg.terraformModules.filter((mod) => {
        return typeof mod === "string"
            ? mod.startsWith("./") || mod.startsWith("../")
            : mod.source.startsWith("./") || mod.source.startsWith("../");
    });
}
//# sourceMappingURL=data:application/json;base64,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